/*  Copyright (C) 2011 Ben Asselstine
 *  This file originates in the doodlehash project.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope tmouth it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
 *  02110-1301, USA.
 */
#include <config.h>
#include "mouth.h"
#include <libxml/tree.h>
#include <glib.h>
#include <string.h>
#include "util.h"
#include "object.h"
DoodleHashMouth* doodlehash_mouth_new()
{
  DoodleHashMouth *m = g_malloc (sizeof (DoodleHashMouth));
  if (!m)
    return NULL;
  memset (m, 0, sizeof (DoodleHashMouth));
  m->base = doodlehash_object_new();
  return m;
}

DoodleHashMouth* doodlehash_mouth_copy(const DoodleHashMouth*orig)
{
  DoodleHashMouth *mouth = doodlehash_mouth_new();
  g_free (mouth->base);
  mouth->base = doodlehash_object_copy(orig->base);
  mouth->connect_to_head_x = orig->connect_to_head_x;
  mouth->connect_to_head_y = orig->connect_to_head_y;
  mouth->connect_to_moustache_x = orig->connect_to_moustache_x;
  mouth->connect_to_moustache_y = orig->connect_to_moustache_y;
  return mouth;
}

int doodlehash_mouth_load_from_svg(DoodleHashMouth *m, const char *file)
{
  if (doodlehash_object_load_from_svg_file(m->base, file) != 0)
    return -1;
  RsvgHandle *handle = rsvg_handle_new_from_data((const guint8*)m->base->data, 
                                                 m->base->data_len, NULL);
  if (find_connect_point(handle, "#connect-to-moustache", 
                         &m->connect_to_moustache_x, 
                         &m->connect_to_moustache_y) != 0)
    return -2;
  if (find_connect_point(handle, "#connect-to-head", 
                         &m->connect_to_head_x, &m->connect_to_head_y) != 0)
    return -3;
  rsvg_handle_close(handle, NULL);
  return 0;
}

DoodleHashMouth* doodlehash_mouth_load_from_svg_file(const char *file)
{
  DoodleHashMouth *mouth = doodlehash_mouth_new();
  if (doodlehash_mouth_load_from_svg(mouth, file) == 0)
    return mouth;
  else
    {
      doodlehash_mouth_free(mouth);
      return NULL;
    }
}

void doodlehash_mouth_get_width_and_height(DoodleHashMouth* mouth, guint32* width, guint32* height)
{
  doodlehash_object_get_width_and_height(mouth->base, width, height);
}

void doodlehash_mouth_set_width(DoodleHashMouth* mouth, guint32 width)
{
  double scale = (double) width / (double)mouth->base->width;
  mouth->connect_to_head_x *= scale;
  doodlehash_object_set_width(mouth->base, mouth->base->width * scale);
}

void doodlehash_mouth_set_height(DoodleHashMouth* mouth, guint32 height)
{
  double scale = (double)mouth->base->height / (double)height;
  mouth->connect_to_head_y *= scale;
  doodlehash_object_set_height(mouth->base, mouth->base->height * scale);
}

void doodlehash_mouth_get_head_connection_point(DoodleHashMouth* mouth, guint32* x, guint32* y)
{
  *x = mouth->connect_to_head_x;
  *y = mouth->connect_to_head_y;
}

guint32 doodlehash_mouth_get_head_connection_point_x(DoodleHashMouth* mouth)
{
  return mouth->connect_to_head_x;
}

guint32 doodlehash_mouth_get_head_connection_point_y(DoodleHashMouth* mouth)
{
  return mouth->connect_to_head_y;
}

void doodlehash_mouth_set_head_connection_point(DoodleHashMouth* mouth, guint32 x, guint32 y)
{
  mouth->connect_to_head_x = x;
  mouth->connect_to_head_y = y;
}

guint32 doodlehash_mouth_get_width(DoodleHashMouth* mouth)
{
  return doodlehash_object_get_width(mouth->base);
}

guint32 doodlehash_mouth_get_height(DoodleHashMouth* mouth)
{
  return doodlehash_object_get_height(mouth->base);
}

void doodlehash_mouth_get_moustache_connection_point(DoodleHashMouth* mouth, guint32* x, guint32* y)
{
  *x = mouth->connect_to_moustache_x;
  *y = mouth->connect_to_moustache_y;
}

void doodlehash_mouth_set_moustache_connection_point(DoodleHashMouth* mouth, guint32 x, guint32 y)
{
  mouth->connect_to_moustache_x = x;
  mouth->connect_to_moustache_y = y;
}

guint32 doodlehash_mouth_get_moustache_connection_point_x(DoodleHashMouth* mouth)
{
  return mouth->connect_to_moustache_x;
}

guint32 doodlehash_mouth_get_moustache_connection_point_y(DoodleHashMouth* mouth)
{
  return mouth->connect_to_moustache_y;
}

void doodlehash_mouth_set_random_colours(DoodleHashMouth* mouth, guint code)
{
  doodlehash_object_set_random_colours(mouth->base, code);
}

void doodlehash_mouth_set_colours(DoodleHashMouth* mouth, guint code, char *s, char *f)
{
  doodlehash_object_set_colours(mouth->base, code, s, f);
}

gchar *doodlehash_mouth_get_fill_colour(DoodleHashMouth* mouth, guint code)
{
  return doodlehash_object_get_fill_colour(mouth->base, code);
}

gchar *doodlehash_mouth_get_stroke_colour(DoodleHashMouth* mouth, guint code)
{
  return doodlehash_object_get_stroke_colour(mouth->base, code);
}

void doodlehash_mouth_scale(DoodleHashMouth *m, double scale)
{
  m->connect_to_head_x *= scale;
  m->connect_to_head_y *= scale;
  doodlehash_object_scale(m->base, scale);
}

void doodlehash_mouth_free(DoodleHashMouth* m)
{
  if (m && m->base)
    doodlehash_object_free (m->base);
  g_free (m);
}
