/*  Copyright (C) 2011 Ben Asselstine
 *  This file originates in the doodlehash project.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
 *  02110-1301, USA.
 */
#include <config.h>
#include "chin.h"
#include <libxml/tree.h>
#include <glib.h>
#include <string.h>
#include "util.h"
#include "object.h"

DoodleHashChin* doodlehash_chin_new()
{
  DoodleHashChin *c = g_malloc (sizeof (DoodleHashChin));
  if (!c)
    return NULL;
  memset (c, 0, sizeof (DoodleHashChin));
  c->base = doodlehash_object_new();
  return c;
}
DoodleHashChin* doodlehash_chin_copy(const DoodleHashChin*orig)
{
  DoodleHashChin *chin = doodlehash_chin_new();
  g_free (chin->base);
  chin->base = doodlehash_object_copy(orig->base);
  chin->connect_to_head_x = orig->connect_to_head_x;
  chin->connect_to_head_y = orig->connect_to_head_y;
  return chin;
}

int doodlehash_chin_load_from_svg(DoodleHashChin *c, const char *file)
{
  if (doodlehash_object_load_from_svg_file(c->base, file) != 0)
    return -1;
  RsvgHandle *handle = rsvg_handle_new_from_data((const guint8*)c->base->data, 
                                                 c->base->data_len, NULL);
  if (find_connect_point(handle, "#connect-to-head", 
                         &c->connect_to_head_x, &c->connect_to_head_y) != 0)
    return -2;
  rsvg_handle_close(handle, NULL);
  return 0;
}

DoodleHashChin* doodlehash_chin_load_from_svg_file(const char *file)
{
  DoodleHashChin *chin = doodlehash_chin_new();
  if (doodlehash_chin_load_from_svg(chin, file) == 0)
    return chin;
  else
    {
      doodlehash_chin_free(chin);
      return NULL;
    }
}


void doodlehash_chin_get_width_and_height(DoodleHashChin* chin, guint32* width, guint32* height)
{
  doodlehash_object_get_width_and_height(chin->base, width, height);
}

void doodlehash_chin_set_width(DoodleHashChin* chin, guint32 width)
{
  double scale = (double) width / (double)chin->base->width;
  chin->connect_to_head_x *= scale;
  doodlehash_object_set_width(chin->base, chin->base->width * scale);
}

void doodlehash_chin_set_height(DoodleHashChin* chin, guint32 height)
{
  double scale = (double)chin->base->height / (double)height;
  chin->connect_to_head_y *= scale;
  doodlehash_object_set_height(chin->base, chin->base->height * scale);
}

void doodlehash_chin_get_head_connection_point(DoodleHashChin* chin, guint32* x, guint32* y)
{
  *x = chin->connect_to_head_x;
  *y = chin->connect_to_head_y;
}

void doodlehash_chin_set_head_connection_point(DoodleHashChin* chin, guint32 x, guint32 y)
{
  chin->connect_to_head_x = x;
  chin->connect_to_head_y = y;
}

guint32 doodlehash_chin_get_width(DoodleHashChin* chin)
{
  return doodlehash_object_get_width(chin->base);
}

guint32 doodlehash_chin_get_height(DoodleHashChin* chin)
{
  return doodlehash_object_get_height(chin->base);
}

guint32 doodlehash_chin_get_head_connection_point_x(DoodleHashChin* chin)
{
  return chin->connect_to_head_x;
}

guint32 doodlehash_chin_get_head_connection_point_y(DoodleHashChin* chin)
{
  return chin->connect_to_head_y;
}

void doodlehash_chin_set_random_colours(DoodleHashChin* chin, guint code)
{
  doodlehash_object_set_random_colours(chin->base, code);
}

void doodlehash_chin_set_colours(DoodleHashChin* chin, guint code, char *s, char *f)
{
  doodlehash_object_set_colours(chin->base, code, s, f);
}

gchar *doodlehash_chin_get_fill_colour(DoodleHashChin* chin, guint code)
{
  return doodlehash_object_get_fill_colour(chin->base, code);
}

gchar *doodlehash_chin_get_stroke_colour(DoodleHashChin* chin, guint code)
{
  return doodlehash_object_get_stroke_colour(chin->base, code);
}

void doodlehash_chin_scale(DoodleHashChin *c, double scale)
{
  c->connect_to_head_x *= scale;
  c->connect_to_head_y *= scale;
  doodlehash_object_scale(c->base, scale);
}

void doodlehash_chin_free(DoodleHashChin* chin)
{
  if (chin && chin->base)
    doodlehash_object_free(chin->base);
  g_free (chin);
}
