/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "slide-editor.h"

#include <inttypes.h>
#include <string.h>

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <libgnomecanvas/libgnomecanvas.h>

#include <gnome-canvas-helpers.h>

#include "debug.h"
#include "display.h"
#include "presentation.h"
#include "renderer.h"
#include "slide-view.h"

enum {
	PROP_0,
	PROP_ZOOM,
};

struct _CriaSlideEditorPriv {
	CriaDisplay	* display;
	GnomeCanvasGroup* group;
	GnomeCanvasItem	* shadow,
			* border;
	CriaSlide	* slide;
};

static	void	cria_slide_editor_get_property         (GObject		* object,
					        	guint		  prop_id,
					        	GValue		* value,
					        	GParamSpec	* param_spec);
static	void	cria_slide_editor_init	               (CriaSlideEditor	* self);
static	void	cria_slide_editor_set_property         (GObject		* object,
						        guint		  prop_id,
						        const	GValue	* value,
						        GParamSpec	* param_spec);
static gboolean	cria_slide_editor_popup_menu	       (GtkWidget	* self);
static	void	cria_slide_editor_slide_changed        (CriaSlideEditor	* self,
						        gpointer	  data);

static go_unit_t
get_slide_width(CriaSlide* slide) {
#warning "get rid of this one"
	const GoPoint * size = cria_slide_get_size(slide);
	go_unit_t retval = size->x;
	return retval;
}

static go_unit_t
get_slide_height(CriaSlide* slide) {
#warning "get rid of this one"
	const GoPoint * size = cria_slide_get_size(slide);
	go_unit_t retval = size->y;
	return retval;
}

static gboolean
cria_slide_editor_button_press_event(GtkWidget* widget, GdkEventButton* event) {
#warning "SlideEditor::buttonPressEvent(): find out where the click appeared and populate the menu"
#warning "SlideEditor::buttonPressEvent(): show the menu"
	return FALSE;
}

static void
cria_slide_editor_class_init (CriaSlideEditorClass* cria_slide_editor_class) {
	GObjectClass	* g_object_class;
	GtkWidgetClass	* widget_class;

	/* setting up the object class */
	g_object_class = G_OBJECT_CLASS(cria_slide_editor_class);
	g_object_class->set_property = cria_slide_editor_set_property;
	g_object_class->get_property = cria_slide_editor_get_property;

	/* setting up the widget class */
	widget_class = GTK_WIDGET_CLASS(cria_slide_editor_class);
	widget_class->button_press_event = cria_slide_editor_button_press_event;
	widget_class->popup_menu = cria_slide_editor_popup_menu;

	/* setting up the slide view interface */
	_cria_slide_view_install_properties(g_object_class);
}

CriaSlide*
cria_slide_editor_get_slide(CriaSlideEditor* self) {
	g_return_val_if_fail(CRIA_IS_SLIDE_EDITOR(self), NULL);
	
	return self->priv->slide;
}

static void
cria_slide_editor_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* param_spec) {
	CriaSlideEditor	* self;

	self = CRIA_SLIDE_EDITOR (object);

	switch (prop_id) {
	case CRIA_SLIDE_VIEW_PROP_SLIDE:
		g_value_set_object(value, self->priv->slide);
		break;
	case PROP_ZOOM:
		g_value_set_double(value, cria_display_get_zoom(self->priv->display));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

gdouble
cria_slide_editor_get_zoom(CriaSlideEditor* self) {
	g_assert(self != NULL);
	g_assert(CRIA_IS_SLIDE_EDITOR(self));
	g_assert(self->priv != NULL);

	return cria_display_get_zoom(self->priv->display);
}

GType
cria_slide_editor_get_type (void) {
	static GType	type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (CriaSlideEditorClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_slide_editor_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof (CriaSlideEditor),
			0,
			(GInstanceInitFunc)cria_slide_editor_init,
			0
		};

		static const GInterfaceInfo view_info = {
			NULL, /* GInterfaceInitFunc */
			NULL, /* GInterfaceFinalizeFunc */
			NULL  /* data */
		};

		type = g_type_register_static(GTK_TYPE_SCROLLED_WINDOW,
					      "CriaSlideEditor",
					      &info,
					      0);
		g_type_add_interface_static(type, CRIA_TYPE_SLIDE_VIEW, &view_info);
	}

	return type;
}

static void
cria_slide_editor_init(CriaSlideEditor   * self) {
	/* prepare private data */
	self->priv = g_new0(CriaSlideEditorPriv,1);

	/* set up the scrolled window */
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(self),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(self),
					    GTK_SHADOW_IN);

	/* prepare the canvas */
	self->priv->display	= cria_display_new();
	cria_display_set_padding(self->priv->display, SLIDE_EDITOR_PADDING);

	/* prepare the canvas groups */
/*	self->priv->group = GNOME_CANVAS_GROUP(gnome_canvas_item_new(cria_display_get_top_group(self->priv->display),
								     gnome_canvas_group_get_type(),
								     "x", 0.0,
								     "y", 0.0,
								     NULL));
	gnome_canvas_item_show(GNOME_CANVAS_ITEM(self->priv->group));
*/
	/* add the shadow */
	self->priv->shadow = gnome_canvas_item_new(cria_display_get_top_group(self->priv->display),
						   gnome_canvas_rect_get_type(),
						   "fill-color", "gray",
						   "x1", 0.0 + SLIDE_EDITOR_SHADOW_OFFSET,
						   "y1", 0.0 + SLIDE_EDITOR_SHADOW_OFFSET,
						   "x2", 0.0 + SLIDE_EDITOR_SHADOW_OFFSET + get_slide_width(self->priv->slide),
						   "y2", 0.0 + SLIDE_EDITOR_SHADOW_OFFSET + get_slide_height(self->priv->slide),
						   NULL);
	gnome_canvas_item_show(self->priv->shadow);
	gnome_canvas_item_lower_to_bottom(self->priv->shadow);

	/* add the slide border */
	self->priv->border = gnome_canvas_item_new(cria_display_get_top_group(self->priv->display),
						   gnome_canvas_rect_get_type(),
						   "outline-color", "black",
						   "width-pixels", 1,
						   "x1", 0.0,
						   "y1", 0.0,
						   "x2", 0.0 + get_slide_width(self->priv->slide),
						   "y2", 0.0 + get_slide_height(self->priv->slide),
						   NULL);
	gnome_canvas_item_show(self->priv->border);
	gnome_canvas_item_raise_to_top(self->priv->border);

	gtk_widget_show(GTK_WIDGET(self->priv->display));

	gtk_scrolled_window_set_hadjustment(GTK_SCROLLED_WINDOW(self), gtk_layout_get_hadjustment(GTK_LAYOUT(self->priv->display)));
	gtk_scrolled_window_set_vadjustment(GTK_SCROLLED_WINDOW(self), gtk_layout_get_vadjustment(GTK_LAYOUT(self->priv->display)));
	gtk_container_add(GTK_CONTAINER(self), GTK_WIDGET(self->priv->display));

	/* connect neccessary signals */
	g_signal_connect(self, "notify::slide",
			 G_CALLBACK(cria_slide_editor_slide_changed), NULL);
}

GtkWidget*
cria_slide_editor_new(CriaSlide* slide) {
	g_assert(slide != NULL && CRIA_IS_SLIDE(slide));
	
	return g_object_new(CRIA_TYPE_SLIDE_EDITOR, "slide", slide, NULL);
}

static gboolean
cria_slide_editor_popup_menu(GtkWidget* widget) {
#warning "SlideEditor::buttonPressEvent(): find out where the cursor is and populate the menu"
#warning "SlideEditor::buttonPressEvent(): show the menu"
	return FALSE;
}

void
cria_slide_editor_set_slide(CriaSlideEditor* self, CriaSlide* slide) {
	g_return_if_fail(CRIA_IS_SLIDE_EDITOR(self));
	g_return_if_fail(CRIA_IS_SLIDE(slide));
	
	g_debug("cria_slide_editor_set_slide(): start");

	if(self->priv->slide != slide) {
		const GoPoint	* slide_s;
		GoPoint		* display_s;
		
		if(self->priv->slide != NULL) {
			g_object_unref(self->priv->slide);
		}

		self->priv->slide = g_object_ref(slide);

		slide_s = cria_slide_get_size(self->priv->slide);
		display_s = cria_slide_get_display_size(self->priv->slide, gdk_screen_get_default());
		cria_display_set_sizes(self->priv->display, slide_s, display_s);
		//g_free(slide_s);
		//g_free(display_s);
	
		g_object_notify(G_OBJECT(self), "slide");
	
	}
	g_debug("cria_slide_editor_set_slide(): end");
}

static void
cria_slide_editor_set_property (
		GObject		* object,
		guint		  prop_id,
		const	GValue	* value,
		GParamSpec	* param_spec)
{
	CriaSlideEditor	* self;
	
	self = CRIA_SLIDE_EDITOR(object);
	
	switch(prop_id) {
	case CRIA_SLIDE_VIEW_PROP_SLIDE:
		cria_slide_editor_set_slide(self, g_value_get_object(value));
		break;
	case PROP_ZOOM:
		cria_slide_editor_set_zoom(self, g_value_get_double(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object,
						  prop_id,
						  param_spec);
		break;
	}
}

void
cria_slide_editor_set_zoom(CriaSlideEditor* self, gdouble zoom) {
	g_assert(self != NULL);
	g_assert(G_IS_OBJECT(self));
	g_assert(CRIA_IS_SLIDE_EDITOR(self));
	g_assert(self->priv != NULL);
	g_assert(self->priv->display != NULL && CRIA_IS_DISPLAY(self->priv->display));

	cria_display_set_zoom(self->priv->display, zoom);
}

static void
cria_slide_editor_post_rendering(CriaSlideEditor* self) {
	g_assert(self != NULL);
	g_assert(CRIA_IS_SLIDE_EDITOR(self));
	g_assert(self->priv != NULL);
	g_assert(self->priv->display != NULL);
	g_assert(GNOME_IS_CANVAS(self->priv->display));

	gnome_canvas_item_raise_to_top(self->priv->border);
	gnome_canvas_item_lower_to_bottom(self->priv->shadow);
}

static void
cria_slide_editor_slide_changed(CriaSlideEditor* self, gpointer data) {
	g_assert(self != NULL);
	g_assert(CRIA_IS_SLIDE_EDITOR(self));

	cria_renderer_render_slide(self->priv->display, self->priv->slide);

	cria_slide_editor_post_rendering(self);
}

