/*
Copyright (C)  2006  Daniele Zelante

This file is part of cmgl.

cmgl is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

cmgl is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with cmgl; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/*@LICENSE*/
// $Id: vad.cxx,v 1.2 2006/01/02 22:29:15 zeldan Exp $

#include <math.h>
#include <float.h>

#include <comf/bstorage.hxx>
#include <comf/tstorage.hxx>

using namespace comf;

#include "vad.hxx"

CMGL_NS_BEGIN

VaD::VaD()
{
	_v=0;
	_d=DBL_MIN;
}

VaD::VaD(const double v)
{
	_v=v;
	_d=fabs(_v * stdError);
	if (_d<DBL_MIN) _d = DBL_MIN;
}

VaD::VaD(const double v, const double d)
{
	_v=v;
	_d=fabs(d);
}


const double VaD::stdError = DBL_EPSILON;

const VaD operator + (const VaD & a, const VaD & b)
{
	return VaD(a._v+b._v,hypot(a._d,b._d));
}

const VaD operator - (const VaD & a, const VaD & b)
{
	return VaD(a._v-b._v,hypot(a._d,b._d));
}

const VaD operator * (const VaD & a, const VaD & b)
{
	return VaD(a._v*b._v,hypot(b._v*a._d,a._v*b._d));
}

const VaD operator * (const VaD & a, const double b)
{
	return VaD(a._v*b,fabs(b*a._d));
}

const VaD operator * (const double a, const VaD & b)
{
	return VaD(b._v*a,fabs(a*b._d));
}

const VaD operator / (const VaD & a, const VaD & b)
{
	return VaD(a._v/b._v,hypot(a._d/b._v,(a._v*b._d)/(b._v*b._v)));
}

const VaD operator / (const VaD & a, double b)
{
	return VaD(a._v/b,fabs(a._d/b));
}

const VaD operator / (const double a, const VaD & b)
{
	return VaD(a/b._v,fabs((a*b._d)/(b._v*b._v)));
}


bool operator == (const VaD & a, const VaD & b)
{
	return (fabs(a._v-b._v)<=hypot(a._d,b._d));
}

bool operator != (const VaD & a, const VaD & b)
{
	return !(a==b);
}

bool operator > (const VaD & a, const VaD & b)
{
	return (a._v>b._v)&&!(a==b);
}

bool operator < (const VaD & a, const VaD & b)
{
	return (a._v<b._v)&&!(a==b);
}

bool operator >= (const VaD & a, const VaD & b)
{
	return (a._v>b._v)||(a==b);
}

bool operator <= (const VaD & a, const VaD & b)
{
	return (a._v<b._v)||(a==b);
}


VaD & VaD::operator += (const VaD & a)
{
	*this = *this + a;
	return * this;
}

VaD & VaD::operator -= (const VaD & a)
{
	*this = *this - a;
	return * this;
}

VaD & VaD::operator *= (const VaD & a)
{
	*this = *this * a;
	return * this;
}

VaD & VaD::operator /= (const VaD & a)
{
	*this = *this / a;
	return * this;
}

const VaD & VaD::operator +()
{
	return *this;
}

const VaD VaD::operator -()
{
	return VaD(-_v,_d);
}


const VaD pow(const VaD & a, const VaD & b)
{
	return VaD(::pow(a._v,b._v),
		::hypot(b._v*::pow(a._v,b._v-1)*a._d,::pow(a._v,b._v)*::log(a._v)*b._d));
}

const VaD pow(const VaD & a, double b)
{
	return VaD(::pow(a._v,b),::fabs(b*::pow(a._v,b-1)*a._d));
}

const VaD pow(double a, const VaD & b)
{
	return VaD(::pow(a,b._v),::fabs(::pow(a,b._v)*::log(a)*b._d));
}

const VaD sqrt(const VaD & a)
{
	return VaD(::sqrt(a._v),a._d/(2*::sqrt(a._v)));
}


const VaD exp(const VaD & a)
{
	return VaD(::exp(a._v),::fabs(::exp(a._v)*a._d));
}

const VaD log(const VaD & a)
{
	return VaD(::log(a._v),::fabs(a._d/a._v));
}


void VaD::serialize(BOStorage & s) const
{
	s << _v << _d;
}

void VaD::serialize(BIStorage & s)
{
	s >> _v >> _d;
}


void VaD::serialize(TOStorage & s) const
{
	s << classname() << "(" << _v << "," << _d << ")";
}

void VaD::serialize(TIStorage & s)
{
	s >> classname() >> "(" >> _v >> "," >> _d >> ")";
 
	if (_d<0) _d = -_d;
	if (_d<DBL_MIN) _d = DBL_MIN;
}


CMGL_NS_END
