/*
Copyright (C)  2006  Daniele Zelante

This file is part of cmgl.

cmgl is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

cmgl is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with cmgl; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/*@LICENSE*/
// $Id: stats.hxx,v 1.2 2006/01/02 22:29:15 zeldan Exp $

#ifndef CMGL_STATS_HXX
#define CMGL_STATS_HXX

#include <comf/defs.hxx>
#include <comf/types.hxx>
#include <comf/exception.hxx>

#include "linear2d.hxx"
#include "linear3d.hxx"
#include "defs.hxx"


CMGL_NS_BEGIN

template <typename T, typename C=T> class Minimal
{
	private:
	T _value;
	bool _init;

	public:
	Minimal() : _init(false) {}
	
	void operator &=(C x)
	{
		_value = _init ? std::min(_value,x) : x;
		_init=true;
	}
	
	operator C () const
	{
		if(!_init) COMF_throwCOMFEXCEPTION("no data");
		return _value;
	}
};	

template <typename T, typename C=T> class Maximal
{
	private:
	T _value;
	bool _init;

	public:
	Maximal() : _init(false) {}
	
	void operator &=(C x)
	{
		_value = _init ? std::max(_value,x) : x;
		_init=true;
	}
	
	operator C () const
	{
		if(!_init) COMF_throwCOMFEXCEPTION("no data");
		return _value;
	}
};	



class Stat 
{
	public:
 
	enum Compute    // richiede di calcolare solo alcune informazioni
	{
		SUM = 1,     // somma
		SUM2 = 2,    // somma dei quadrati
		MINMAX = 4,  // minimo e massimo

		SIGMA = SUM|SUM2,
		ALL = SIGMA|MINMAX
	};

	Stat(Compute);

	void add(double);              /// aggiunge un valore al set
	int count() const;           /// numero di valori
	double sum() const;            /// somma 
	double sum2() const;           /// somma dei quadrati
	double average() const;        /// media
	double average2() const;       /// media dei quadrati
	double min() const;            /// minimo
	double max() const;            /// massimo
	double mid() const;            /// punto medio (massimo+minimo)/2
	double sigma() const;          /// deviazione standard (non campionaria)
	double delta() const;          /// estenzione (massimo-minimo)
	int nMin();                  /// indice del primo valore minimo
	int nMax();                  /// indice del primo valore massimo

	void clear();                /// cancella tutti i valori accumulati


	protected:
	int _count;
	double _sum;
	double _qsum;
	double _min;
	double _max;
	int _nMin;
	int _nMax;
	Compute _ops;

};


class Stat2D 
{
	public:
	Stat2D();

	void add(const Vector2D &);         // aggiunge un vettore al set
	int count() const;                  // numero di vettori
	const Vector2D sum() const;               // somma di tutti i vettori
	double sum2() const;                  // somme dei quadrati delle norme di tutti i vettori
	const Vector2D average() const;           // vettore medio (Sum()/Count())
	double average2() const;              // media dei quadrati delle norme dei vettori
	double sigma() const;                 // devizione standard (vedi sorgente per formula)

	void clear();          // cancella tutti i vettori accumulati

	protected:
	int _count;
	Vector2D _sum;
	double _qsum;

};


class Stat3D
{
	public:

	Stat3D();

	void add(const Vector3D &);         // aggiunge un vettore al set
	int count() const;                  // numero di vettori
	const Vector3D sum() const;               // somma di tutti i vettori
	double sum2() const;                  // somme dei quadrati delle norme di tutti i vettori
	const Vector3D average() const;           // vettore medio (Sum()/Count())
	double average2() const;              // media dei quadrati delle norme dei vettori
	double sigma() const;                 // devizione standard (vedi sorgente per formula)

	void clear();          // cancella tutti i vettori accumulati

	protected:
	int _count;
	Vector3D _sum;
	double _qsum;


};

CMGL_NS_END

#endif

