package Bibulus::general;

use strict;
use warnings;
use Carp;

sub genpopulate {
  my $self = shift;

  # Used by andlist():
  $self->{LANGTEXT}{$self->lang}{and} ||= '[default:and]';

  # Punctuation:
  $self->{LANGTEXT}{$self->lang}{comma} ||= ', ';
  $self->{LANGTEXT}{$self->lang}{fullstop} ||= '. ';
  $self->{LANGTEXT}{$self->lang}{abbrsign} ||= '.' . $self->nobreakspace;

  # Used by in():
  $self->{LANGTEXT}{$self->lang}{in} ||= '[default:in]';

  # Used by edition():
  $self->{LANGTEXT}{$self->lang}{edition} ||= '[default:edition]';

  # Used by technicalreport():
  $self->{LANGTEXT}{$self->lang}{technicalreport} ||= '[default:technicalreport]';

  $self->{LANGTEXT}{$self->lang}{numberabbr} ||= '[default:numberabbr]';

  # Used by chapter():
  $self->{LANGTEXT}{$self->lang}{chapter} ||= '[default:chapter]';

  # Used by thesis():
  $self->{LANGTEXT}{$self->lang}{mastersthesis} ||= '[default:mastersthesis]';
  $self->{LANGTEXT}{$self->lang}{phdthesis} ||= '[default:phdthesis]';

  # Used by volume():
  $self->{LANGTEXT}{$self->lang}{volume} ||= '[default:volume]';
  $self->{LANGTEXT}{$self->lang}{of} ||= '[default:of]';

  # Used by numberseries():
  $self->{LANGTEXT}{$self->lang}{number} ||= '[default:number]';

  $self->{LANGTEXT}{$self->lang}{month1} ||= '1';
  $self->{LANGTEXT}{$self->lang}{month2} ||= '2';
  $self->{LANGTEXT}{$self->lang}{month3} ||= '3';
  $self->{LANGTEXT}{$self->lang}{month4} ||= '4';
  $self->{LANGTEXT}{$self->lang}{month5} ||= '5';
  $self->{LANGTEXT}{$self->lang}{month6} ||= '6';
  $self->{LANGTEXT}{$self->lang}{month7} ||= '7';
  $self->{LANGTEXT}{$self->lang}{month8} ||= '8';
  $self->{LANGTEXT}{$self->lang}{month9} ||= '9';
  $self->{LANGTEXT}{$self->lang}{month10} ||= '10';
  $self->{LANGTEXT}{$self->lang}{month11} ||= '11';
  $self->{LANGTEXT}{$self->lang}{month12} ||= '12';
}

sub getword {
  my $self = shift;
  my $field = shift;

  unless (defined($self->{LANGTEXT}{$self->lang}{$field})) {
    croak "The word '$field' is not defined";
  }

  unless (ref($self->{LANGTEXT}{$self->lang}{$field})) {
    return $self->{LANGTEXT}{$self->lang}{$field};
  }

  return $self->{LANGTEXT}{$self->lang}{$field}{$self->{STYLE}{abbr}};
}

# Take a list of names and return them separated with commas or ands
# or whatever is used in this language.
# The default will return "X", "X and Y", "X, Y and Z", "X, Y, Z and U".
sub andlist {
  my $self = shift;
  my @names = @_;

  my $r = '';

  foreach my $i (0 .. $#names) {
    $r .= $names[$i];

    if ($i < $#names - 1) {
      $r .= $self->comma;
    } elsif ($i == $#names - 1) {
      $r .= $self->space . $self->getword('and') . $self->space;
    }
  }

  return $r;
}

# Return a comma (followed by a space if necessary).
sub comma {
  my $self = shift;
  return $self->getword('comma');
}

# Return a full stop (followed by a space if necessary).
sub fullstop {
  my $self = shift;
  return $self->getword('fullstop');
}

# Return an abbreviation sign (followed by a space or a non-breakable
# one if necessary).
# This is the function to use for abbreviating names of people and months.
sub abbrsign {
  my $self = shift;
  return $self->getword('abbrsign');
}

# Take a text $t and return "in $t".  $t is typically a reference to
# another entry.
sub in {
  my $self = shift;
  my $t = shift;
  return $self->getword('in') . $self->nobreakspace . $t;
}

# Take an edition number and return "fifth edition" of "5th ed." or
# whatever is needed.  Abbreviation of "edition" is governed by
# $self->{STYLE}{abbredition}.
# Don't abbreviate if it's not relevant in a given language!
sub edition {
  my $self = shift;
  my $num = shift;

  return $self->ordinal($num) . $self->nobreakspace
    . $self->getword('edition');
}


# Either return "Technical Report" or "Technical Report #$number",
# depending on whether $num is given or not.  Abbreviate if necessary.
sub technicalreport {
  my $self = shift;
  my $num = shift;

  my $l = $self->{LANG}; # current language

  my $tr;

  $tr = $self->getword('technicalreport');

  if ($num) {
    $tr .= $self->nobreakspace . $num;
  }

  return $tr;
}

sub chapter {
  my $self = shift;
  my ($ch) = @_;

  return $self->getword('chapter') . $self->nobreakspace . $ch;
}

sub thesis {
  my $self = shift;
  my $type = shift;

  if ($type eq 'phd') {
    return $self->getword('phdthesis');
  } elsif ($type eq 'master') {
    return $self->getword('mastersthesis');
  } else {
    warn "Unknown thesis type $type\n";
    return $type;
  }
}

sub volume {
  my $self = shift;
  my $vol = shift;
  my $series = shift;
  my $volword = $self->getword('volume');

  if (!$series) {
    return $volword . $self->nobreakspace . $vol;
  } else {
    return $volword . $self->nobreakspace . $vol . $self->nobreakspace
      . $self->getword('of') . $self->nobreakspace . $series;
  }
}

sub numberseries {
  my $self = shift;
  my ($number, $series) = @_;

  my $t = $self->getword('number') . $self->nobreakspace . $number;
  if ($series) {
    $t .= $self->space . $self->in($series);
  }
  return $t;
}

# Return the ordinal number of number $number
# (e.g., 5 -> "fifth" or "5th").
# Abbreviation depends of $self->{STYLE}{abbrordinals}.
sub ordinal {
  my $self = shift;
  my ($number) = @_;

  if ($self->{STYLE}{abbrordinals}) {
    if (defined($self->{LANGTEXT}{general}{$self->{LANG}}{ordinal}{$number}{abbr})) {
      return $self->{LANGTEXT}{general}{$self->{LANG}}{ordinal}{$number}{abbr};
    } else {
      return $number . $self->{LANGTEXT}{general}{$self->{LANG}}{th};
    }
  } else { # not abbreviated
    if (defined($self->{LANGTEXT}{general}{$self->{LANG}}{ordinal}{$number}{full})) {
      return $self->{LANGTEXT}{general}{$self->{LANG}}{ordinal}{$number}{full};
    } elsif (defined($self->{LANGTEXT}{general}{$self->{LANG}}{ordinal}{$number})) {
      return $self->{LANGTEXT}{general}{$self->{LANG}}{ordinal}{$number};
    } else {
      return $number . $self->{LANGTEXT}{general}{$self->{LANG}}{th};
    }
  }
}

sub monthyear {
  my $self = shift;
  my ($month, $year) = @_;

  return $self->getword("month$month") . $self->space . $year;
}

sub year {
  my $self = shift;
  my ($year) = @_;

  return $year;
}

1;
__END__

=head1 NAME

Bibulus::general -- fall-back language support

=head1 SEE ALSO

F<Bibulus>.

The homepage is
L<http://www.nongnu.org/bibulus/>.

=head1 AUTHOR

Thomas M. Widmann, L<twid@cpan.org>

=head1 COPYRIGHT AND LICENSE

Copyright 2003 by Thomas M. Widmann

This module is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
USA.

=cut
