package Bibulus;

# For copyright and license please see the final part of this file.

use warnings;
use Carp;
use XML::Twig;

# The following modules used to be part of this file
# but they were split out to improve legibility:
use Bibulus::formatentry;
use Bibulus::formatting;
use Bibulus::labels;
use Bibulus::language;
use Bibulus::primitives;
use Bibulus::sorting;
use Bibulus::location;
use Bibulus::punctuation;

# Debugging is a package global:
my $DEBUG = 0;
sub setdebug {
  $DEBUG = 1;
  print STDERR 'This is Bibulus $Id: Bibulus.pm,v 1.30 2004/05/11 20:48:41 twid Exp $', "\n";
}

sub new {
  my $class = shift;
  my $self = {};
  bless($self, $class);

  # default: if a title is only xref'ed once, it will be inlined:
  $self->inlinecrossref(1);

  # tell what you're doing!
  $self->verbose(1);

  # set default language:
  $self->lang('none');

  # set default style parameters:
  $self->style(von => 'partoffamily',
	       nameorder => 'givenfirst',
	       titlefont => 'emph',
	       reportfont => 'emph',
	       thesisfont => 'plain',
	       articlefont => 'plain',
	       abbr => 'full',
	      );

  return $self;
}

# main program logic
sub getbib {
  my $self = shift;
  $self->allfound;
  $self->doinlinecrossref;
  $self->gensortkeys;
  $self->sortbib;
  $self->labels;
  return $self->genbib;
}

# set verbosity:
sub verbose {
  my $self = shift;
  my ($v) = @_;
  $self->{VERBOSE} = $v;
}

# add a text to the tree
# (it's not used internally, but it's useful for users
# who can do stuff like \bibulusadd{label}{note}{Good book!})
sub add {
  my $self = shift;
  my ($ref, $field, $text) = @_;
  foreach my $n (@{$self->{EL}}) {
    my $id = $n->id;
    $id eq $ref or next;

    $n->first_child($field)->suffix($text);
  }
}

# max. number of crossrefs to an entry that will be inlined
sub inlinecrossref {
  my $self = shift;
  $self->{INLINECROSSREF} = shift;
}

# inline titles that are referenced rarely
sub doinlinecrossref {
  my $self = shift;

  my %willdie;

  $self->{CITEALL} and return;

  foreach my $n (0..$#{$self->{EL}}) {
    my $i = $self->{EL}[$n];
    my $id = $i->id;

    unless (defined($id)) {
        print STDERR "ID not defined.\n";
        next;
    }

    print "Including $id\n" if $DEBUG;
    if ($self->{CITES}{$id} != 0) { # crossref'ed entries

      # is it crossref'ed rarely enough?
      if ($self->{CITES}{$id} <= $self->{INLINECROSSREF}) {

	# Change some tags:
	# magazine: title -> journal
	if ($i->tag eq 'magazine' and defined($i->first_child('title'))) {
	  $i->first_child('title')->set_gi('journal');
	}
	# book: title-> booktitle
	if ($i->tag eq 'magazine' and defined($i->first_child('title'))) {
	  $i->first_child('title')->set_gi('booktitle');
	}

	# find entries crossref'ing $id
      J: foreach my $j (@{$self->{EL}}) {
	  if (defined($j->first_child('crossref'))) {
	    my $idj = $j->first_child('crossref')->atts->{id};
	    $idj eq $id or next J;

	    print "Inlining $id\n" if $DEBUG;
	    # delete the crossref:
	    $j->first_child('crossref')->delete;

	    # paste in various fields --
	    # this should probably be restricted to certain fields:
	    foreach my $n ($i->children) {
	      my $x = $n->copy;
	      $x->paste('last_child', $j);
	    }
	  }
	}

	# mark for deletion:
	$willdie{$self->{EL}[$n]} = 1;
	print "This entry will be removed.\n" if $DEBUG;
      }
    }
  }
  # prune inlined titles:
  @{$self->{EL}} = grep {!defined($willdie{$_})} @{$self->{EL}};
}

# Check whether all cited titles were found;
# return 1 if all were there, 0 otherwise;
# print warnings for everything not found.
sub allfound {
  my $self = shift;
  my $found = 1;

  foreach my $i (keys %{$self->{CITES}}) {
    unless (defined($self->{FOUND}{$i})) {
      warn "Entry with id = '$i' not found.\n";
      $found = 0;
    }
  }

  return $found;
}

# for selecting books to cite
sub cite {
  my $self = shift;
  my (@cites) = @_;
  foreach (@cites) {
    $self->{CITES}{$_} = 0;
  }
}

# for citing everything (LaTeX \nocite{*})
sub citeall {
  my $self = shift;
  $self->{CITEALL} = 1;
}

# for defining bibliography style
sub style {
  my $self = shift;
  my %style = @_;

  foreach my $i (keys %style) {
      $self->{STYLE}{$i} = $style{$i};
  }

  # here there should be a possible to override language strings
}

# Load XML database file.
# It is possible to call this function multiple times.
sub load {
  my $self = shift;
  my ($filename) = @_;
  my %h =
    (
     # Note that only the following entry types
     # are inserted into the XML tree.  All other types
     # are ignored
     article => sub {insertentry($self, @_)},
     book => sub {insertentry($self, @_)},
     booklet => sub {insertentry($self, @_)},
     inbook => sub {insertentry($self, @_)},
     incollection => sub {insertentry($self, @_)},
     inproceedings => sub {insertentry($self, @_)},
     manual => sub {insertentry($self, @_)},
     misc => sub {insertentry($self, @_)},
     reprint => sub {insertentry($self, @_)},
     magazine => sub {insertentry($self, @_)},
     personal => sub {insertentry($self, @_)},
     lecture => sub {insertentry($self, @_)},
     oral => sub {insertentry($self, @_)},
     image => sub {insertentry($self, @_)},
     map => sub {insertentry($self, @_)},
     audio => sub {insertentry($self, @_)},
     video => sub {insertentry($self, @_)},
     software => sub {insertentry($self, @_)},
     play => sub {insertentry($self, @_)},
     thesis => sub {insertentry($self, @_)},
     proceedings => sub {insertentry($self, @_)},
     report => sub {insertentry($self, @_)},
     unpublished => sub {insertentry($self, @_)},
    );
  foreach my $i (keys %{$self->{HOOK}}) {
    $h{$i} = $self->{HOOK}{$i};
  }

  # Count the number of files we're parsing:
  ++$self->{DBCOUNTER};
  # And print a message if needed:
  print "Database file \#$self->{DBCOUNTER}: $filename\n" if $self->{VERBOSE};

  # Let XML::Twig do the parsing and calling of handlers:
  my $bib = new XML::Twig
    (Id => 'id',
     twig_handlers => \%h,
     pretty_print => 'indented');
  $bib->parsefile($filename);
}

sub _whenparsing {
  my $self = shift;
 my ($pattern, $function) = @_;
  $self->{HOOK}{$pattern} = $function;
}

sub whenparsing {
  my $self = shift;
  my ($pattern, $code) = @_;
  $self->_whenparsing($pattern,
		      sub {
			my($t, $elt)= @_;
			my $i = $elt->text;
			$i = &$code($i);
			$elt->set_text($i);
		      });
}

# insert an entry into the main list if it's cited:
sub insertentry {
  my $self = shift;
  my ($t, $elt) = @_;

#  print STDERR "Language is ", $self->lang, "\n" if $DEBUG;
#  print "insertentry(", $elt->tag, ")\n" if $DEBUG;

  if (defined($self->{CITEALL}) or defined($self->{CITES}{$elt->id})) {
    push @{$self->{EL}}, $elt;
    $self->{FOUND}{$elt->id} = 1;

    # handle crossrefs:
    if (defined($elt->first_child('crossref'))) {
      my $id = $elt->first_child('crossref')->atts->{id};
      if (defined($self->{CITES}{$id})) {
	# if it has been cited on its own, don't count the crossrefs
	$self->{CITES}{$id} == 0 or
	  ++$self->{CITES}{$id};
      } else {
	$self->{CITES}{$id} = 1;
      }
    }
  } else {
    $t->purge;
  }
}

# for debugging:
sub printxml {
  my $self = shift;
 foreach my $i (@{$self->{EL}}) {
    $i->print;
  }
}

# print them
sub genbib {
  my $self = shift;
  my $i;
  my $t;
  $t = $self->bibliography_start();
  foreach $i (@{$self->{EL}}) {
    my $tmp = $self->formatentry($i);
    $t .= $tmp if defined($tmp);
  }
  $t .= $self->bibliography_end();
  return $t;
}

sub warning {
  my $self = shift;

  my $t = join('', @_);
  warn $t, " for id = ", $self->getatt('id'), ".\n"
}

# Return the contents of an XML atrribute to the entry, or the empty
# string if it is not defined.  Optional argument to control whether
# we should complain about missing data.
sub getatt {
  my $self = shift;
  my $field = shift; # 'id', 'type'
  my $obl = 0;
  @_ and $_[0] and $obl = 1;

  my $i = $self->{CUREL}->atts->{$field};
  unless (defined($i)) {
    $self->warning("No $field attribute") if $obl;
    return '';
  }

  return $i;
}

# Return the contents of an XML field, or the empty string
# if it is not defined.  Optional argument to control whether
# we should complain about missing data.
sub getfield {
  my $self = shift;
  my $field = shift; # 'year', 'volume', etc.
  my $obl = 0;
  @_ and $_[0] and $obl = 1;

  my $i = $self->{CUREL}->first_child($field);
  unless (defined($i)) {
    $self->warning("No <$field>") if $obl;
    return '';
  }

  return $i->text;
}

# Check whether field is not defined
sub empty {
  my $self = shift;
  my $field = shift;
  return !($self->getfield($field));
}


1;
__END__

=head1 NAME

Bibulus - Perl extension for processing bibliographies

=head1 SYNOPSIS

  use Bibulus;

=head1 DESCRIPTION

A Perl module for processing bibliographies.  See the documentation in
the F<doc> directory.

=head2 EXPORT

None.

=head1 SEE ALSO

F<Bibulus::LaTeX>, F<Bibulus::HTML> and C<bib2xml>.

The homepage is
L<http://www.nongnu.org/bibulus/>.

=head1 AUTHOR

Thomas M. Widmann, L<twid@cpan.org>

=head1 COPYRIGHT AND LICENSE

Copyright 2003 by Thomas M. Widmann

This module is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
USA.

=cut
