package Bibulus::general;

use strict;
use warnings;
use Carp;

sub populate {
  my $self = shift;

  # Used by andlist():
  $self->{LANGTEXT}{general}{and} ||= '[default:and]';

  # Punctuation:
  $self->{LANGTEXT}{general}{comma} ||= ', ';
  $self->{LANGTEXT}{general}{fullstop} ||= '. ';
  $self->{LANGTEXT}{general}{abbrsign} ||= '.' . $self->nobreakspace;

  # Used by in():
  $self->{LANGTEXT}{general}{in} ||= '[default:in]';

  # Used by edition():
  $self->{LANGTEXT}{general}{edition} ||= '[default:edition]';

  # Used by technicalreport():
  $self->{LANGTEXT}{general}{technicalreport} ||= '[default:technicalreport]';

  $self->{LANGTEXT}{general}{numberabbr} ||= '[default:numberabbr]';

  # Used by chapter():
  $self->{LANGTEXT}{general}{chapter} ||= '[default:chapter]';

  # Used by thesis():
  $self->{LANGTEXT}{general}{mastersthesis} ||= '[default:mastersthesis]';
  $self->{LANGTEXT}{general}{phdthesis} ||= '[default:phdthesis]';

  # Used by volume():
  $self->{LANGTEXT}{general}{volume} ||= '[default:volume]';
  $self->{LANGTEXT}{general}{of} ||= '[default:of]';

  # Used by numberseries():
  $self->{LANGTEXT}{general}{number} ||= '[default:number]';

  # Needed so that we know when we've been here:
  $self->{LANGINIT}{general} = 1;
}

# Take a list of names and return them separated with commas or ands
# or whatever is used in this language.
# The default will return "X", "X and Y", "X, Y and Z", "X, Y, Z and U".
sub andlist {
  my $self = shift;
  my @names = @_;

  my $r = '';

  foreach my $i (0 .. $#names) {
    $r .= $names[$i];

    if ($i < $#names - 1) {
      $r .= $self->comma;
    } elsif ($i == $#names - 1) {
      $r .= " $self->{LANGTEXT}{$self->{LANG}}{and} ";
    }
  }

  return $r;
}

# Return a comma (followed by a space if necessary).
sub comma {
  my $self = shift;
  return $self->{LANGTEXT}{$self->{LANG}}{comma};
}

# Return a full stop (followed by a space if necessary).
sub fullstop {
  my $self = shift;
  return $self->{LANGTEXT}{$self->{LANG}}{fullstop};
}

# Return an abbreviation sign (followed by a space or a non-breakable
# one if necessary).
# This is the function to use for abbreviating names of people and months.
sub abbrsign {
  my $self = shift;
  return $self->{LANGTEXT}{$self->{LANG}}{abbrsign};
}

# Take a text $t and return "in $t".  $t is typically a reference to
# another entry.
sub in {
  my $self = shift;
  my $t = shift;
  return $self->{LANGTEXT}{$self->{LANG}}{in} . $self->nobreakspace . $t;
}

# Take an edition number and return "fifth edition" of "5th ed." or
# whatever is needed.  Abbreviation of "edition" is governed by
# $self->{STYLE}{abbredition}.
# Don't abbreviate if it's not relevant in a given language!
sub edition {
  my $self = shift;
  my $num = shift;

  my $l = $self->{LANG}; # current language
  my $edword;
  if ($self->{STYLE}{abbredition}
      and defined($self->{LANGTEXT}{$l}{edition}{abbr})) {
    $edword = $self->{LANGTEXT}{$l}{edition}{abbr};
  } elsif (!$self->{STYLE}{abbredition}
      and defined($self->{LANGTEXT}{$l}{edition}{full})) {
    $edword = $self->{LANGTEXT}{$l}{edition}{full};
  } else {
    $edword = $self->{LANGTEXT}{$l}{edition};
  }

  return $self->ordinal($num) . ' '  # should this be $self->nobreakspace?
    . $edword;
}


# Either return "Technical Report" or "Technical Report #$number",
# depending on whether $num is given or not.  Abbreviate if necessary.
sub technicalreport {
  my $self = shift;
  my $num = shift;

  my $l = $self->{LANG}; # current language

  my $tr;
  if ($self->{STYLE}{abbrwords}
      and defined($self->{LANGTEXT}{$l}{technicalreport}{abbr})) {
    $tr = $self->{LANGTEXT}{$l}{technicalreport}{abbr};
  } elsif (!$self->{STYLE}{abbrwords}
      and defined($self->{LANGTEXT}{$l}{technicalreport}{full})) {
    $tr = $self->{LANGTEXT}{$l}{technicalreport}{full};
  } else {
    $tr = $self->{LANGTEXT}{$l}{technicalreport};
  }

  if ($num) {
    $tr .= $self->nobreakspace . $num;
  }

  return $tr;
}

sub chapter {
  my $self = shift;
  my ($ch) = @_;

  return $self->{LANGTEXT}{$self->{LANG}}{chapter}
    . $self->nobreakspace . $ch;
}

sub thesis {
  my $self = shift;
  my $type = shift;

  if ($type eq 'phd') {
    return $self->{LANGTEXT}{$self->{LANG}}{phdthesis};
  } elsif ($type eq 'master') {
    return $self->{LANGTEXT}{$self->{LANG}}{mastersthesis};
  } else {
    warn "Unknown thesis type $type\n";
    return $type;
  }
}

sub volume {
  my $self = shift;
  my $vol = shift;
  my $series = shift;
  my $volword = $self->{LANGTEXT}{$self->{LANG}}{volume};
  if (!$series) {
    return $volword . $self->nobreakspace . $vol;
  } else {
    return $volword . $self->nobreakspace . $vol . $self->{LANGTEXT}{$self->{LANG}}{of}
      . $self->nobreakspace . $series;
  }
}

sub numberseries {
  my $self = shift;
  my ($number, $series) = @_;

  my $t = $self->{LANGTEXT}{$self->{LANG}}{number} . $self->nobreakspace . $number;
  if ($series) {
    $t .= ' ' . $self->in($series);
  }
  return $t;
}

# Return the ordinal number of number $number
# (e.g., 5 -> "fifth" or "5th").
# Abbreviation depends of $self->{STYLE}{abbrordinals}.
sub ordinal {
  my $self = shift;
  my ($number) = @_;

  if ($self->{STYLE}{abbrordinals}) {
    if (defined($self->{LANGTEXT}{$self->{LANG}}{ordinal}{$number}{abbr})) {
      return $self->{LANGTEXT}{$self->{LANG}}{ordinal}{$number}{abbr};
    } else {
      return $number . $self->{LANGTEXT}{$self->{LANG}}{th};
    }
  } else { # not abbreviated
    if (defined($self->{LANGTEXT}{$self->{LANG}}{ordinal}{$number}{full})) {
      return $self->{LANGTEXT}{$self->{LANG}}{ordinal}{$number}{full};
    } elsif (defined($self->{LANGTEXT}{$self->{LANG}}{ordinal}{$number})) {
      return $self->{LANGTEXT}{$self->{LANG}}{ordinal}{$number};
    } else {
      return $number . $self->{LANGTEXT}{$self->{LANG}}{th};
    }
  }
}


1;
__END__

=head1 NAME

Bibulus::general -- fall-back language support

=head1 SEE ALSO

F<Bibulus>.

The homepage is
L<http://www.nongnu.org/bibulus/>.

=head1 AUTHOR

Thomas M. Widmann, L<twid@cpan.org>

=head1 COPYRIGHT AND LICENSE

Copyright 2003 by Thomas M. Widmann

This module is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2 of the License, or (at your
option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
USA.

=cut
