/**
 * jline - Java console input library
 * Copyright (c) 2002,2003 Marc Prud'hommeaux marc@apocalypse.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package jline;

import java.io.*;
import java.util.*;


/** 
 *  Representation of the input terminal for a platform. Handles
 *	any initialization that the platform may need to perform
 *	in order to allow the {@link ConsoleReader} to correctly handle
 *	input.
 *  
 *  @author  <a href="mailto:marc@apocalypse.org">Marc Prud'hommeaux</a>
 */
public abstract class Terminal
{
	private static Terminal term;


	/** 
	 *  Configure and return the {@link Terminal} instance for the
	 *  current platform. This will initialize any system settings
	 *  that are required for the console to be able to handle
	 *  input correctly, such as setting tabtop, buffered input, and
	 *  character echo.
	 *
	 *  @see #initializeTerminal
	 */
	public static synchronized Terminal setupTerminal ()
	{
		if (term != null)
			return term;

		final Terminal t;

		String os = System.getProperty ("os.name").toLowerCase ();
		if (os.indexOf ("windows") != -1)
			t = new WindowsTerminal ();
		else
			t = new UnixTerminal ();

		try
		{
			t.initializeTerminal ();
		}
		catch (Exception e)
		{
			e.printStackTrace ();
		}

		return term = t;
	}


	/** 
	 *  Initialize any system settings
	 *  that are required for the console to be able to handle
	 *  input correctly, such as setting tabtop, buffered input, and
	 *  character echo.
	 */
	public abstract void initializeTerminal ()
		throws Exception;


	/** 
	 *  Returns the current width of the terminal (in characters)
	 */
	public abstract int getTerminalWidth ();


	/** 
	 *  Returns the current height of the terminal (in lines)
	 */
	public abstract int getTerminalHeight ();



	/** 
	 *  <p>Terminal that is used for Unix platforms.</p>
	 *
	 *	<p>
	 *  <strong>WARNING:</strong> this class executes the "stty"
	 *  commmand using {@link Runtime#exec} in order to set and query
	 *  various terminal parameters. It will fail in a strict security,
	 *  and standard disclaimers about java programs that fork separate
	 *  commands apply. It also requires that "stty" is in the user's
	 *  PATH variable (which it almost always is).
	 *	</p>
	 *  
	 *  @author  <a href="mailto:marc@apocalypse.org">Marc Prud'hommeaux</a>
	 */
	public static class UnixTerminal
		extends Terminal
	{
		private Map terminfo;
		private int width = -1;
		private int height = -1;


		/** 
		 *  Remove line-buffered input by invoking "stty -icanon min 1"
		 *  against the current terminal.
		 */
		public void initializeTerminal ()
			throws IOException, InterruptedException
		{
			// save the initial tty configuration
			final String ttyConfig = stty ("-g");

			// sanity check
			if (ttyConfig.length () == 0
				|| ttyConfig.indexOf ("=") == -1
				|| ttyConfig.indexOf (":") == -1)
			{
				throw new IOException ("Unrecognized stty code: " + ttyConfig);
			}


			// set the console to be character-buffered instead of line-buffered
			stty ("-icanon min 1");

			// at exit, restore the original tty configuration (for JDK 1.3+)
			try
			{
				Runtime.getRuntime ().addShutdownHook (new Thread ()
				{
					public void start ()
					{
						try
						{
							stty (ttyConfig);
						}
						catch (Exception e)
						{
							e.printStackTrace ();
						}
					}
				});
			}
			catch (AbstractMethodError ame)
			{
				// JDK 1.3+ only method. Bummer.
			}
		}


		/** 
	 	 *	Returns the value of "stty size" width param.
		 *
		 *	<strong>Note</strong>: this method caches the value from the
		 *	first time it is called in order to increase speed, which means
		 *	that changing to size of the terminal will not be reflected
		 *	in the console.
	 	 */
		public int getTerminalWidth ()
		{
			if (width != -1)
				return width;

			int val = 80;
			try
			{
				String size = stty ("size");
				if (size.length () != 0 && size.indexOf (" ") != -1)
				{
					val = Integer.parseInt (
						size.substring (size.indexOf (" ") + 1));
				}
			}
			catch (Exception e)
			{
			}
			return width = val;
		}
	
	
		/** 
	 	 *	Returns the value of "stty size" height param.
		 *
		 *	<strong>Note</strong>: this method caches the value from the
		 *	first time it is called in order to increase speed, which means
		 *	that changing to size of the terminal will not be reflected
		 *	in the console.
	 	 */
		public int getTerminalHeight ()
		{
			if (height != -1)
				return height;

			int val = 24;

			try
			{
				String size = stty ("size");
				if (size.length () != 0 && size.indexOf (" ") != -1)
				{
					val = Integer.parseInt (
						size.substring (0, size.indexOf (" ")));
				}
			}
			catch (Exception e)
			{
			}

			return height = val;
		}


		/** 
		 *  Execute the stty command with the specified arguments
		 *  against the current active terminal.
		 */
		private static String stty (String args)
			throws IOException, InterruptedException
		{
			return exec ("stty " + args + " < /dev/tty").trim ();
		}


		/** 
		 *  Execute the specified command and return the output
		 *  (both stdout and stderr).
		 */
		private static String exec (String cmd)
			throws IOException, InterruptedException
		{
			return exec (new String [] { "sh", "-c", cmd });
		}


		/** 
		 *  Execute the specified command and return the output
		 *  (both stdout and stderr).
		 */
		private static String exec (String [] cmd)
			throws IOException, InterruptedException
		{
			ByteArrayOutputStream bout = new ByteArrayOutputStream ();
	
			Process p = Runtime.getRuntime ().exec (cmd);
			int c;
			InputStream in;
				
			in = p.getInputStream ();
			while ((c = in.read ()) != -1)
				bout.write (c);
	
			in = p.getErrorStream ();
			while ((c = in.read ()) != -1)
				bout.write (c);
	
			p.waitFor ();
	
			String result = new String (bout.toByteArray ());
			return result;
		}
	}


	/** 
	 *  Terminal that is used for Windows platforms.
	 *  
	 *  @author  <a href="mailto:marc@apocalypse.org">Marc Prud'hommeaux</a>
	 */
	public static class WindowsTerminal
		extends Terminal
	{
		public void initializeTerminal ()
		{
			// nothing we need to do (or can do) for windows.
		}


		/** 
	 	 *	Always returng 80, since we can't access this info on Windows.
	 	 */
		public int getTerminalWidth ()
		{
			return 80;
		}
	
	
		/** 
	 	 *	Always returng 24, since we can't access this info on Windows.
	 	 */
		public int getTerminalHeight ()
		{
			return 80;
		}
	}
}
