/**
 * jline - Java console input library
 * Copyright (c) 2002,2003 Marc Prud'hommeaux marc@apocalypse.org
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package jline;

import java.io.*;
import java.text.*;
import java.util.*;

/** 
 *	<p>
 *	A {@link CompletionHandler} that deals with multiple distinct completions
 *	by outputting the complete list of possibilities to the console. This
 *	mimics the behavior of the
 *	<a href="http://www.gnu.org/directory/readline.html">readline</a>
 *	library.
 *	</p>
 *
 *  @author  <a href="mailto:marc@apocalypse.org">Marc Prud'hommeaux</a>
 */
public class CandidateListCompletionHandler
	implements CompletionHandler
{
	private static ResourceBundle loc = ResourceBundle.getBundle (
		CandidateListCompletionHandler.class.getName ());


	public boolean complete (ConsoleReader reader, List candidates, int pos)
		throws IOException
	{
		CursorBuffer buf = reader.getCursorBuffer ();

		// if there is only one completion, then fill in the buffer
		if (candidates.size () == 1)
		{
			String value = candidates.get (0).toString ();

			// fail if the only candidate is the same as the current buffer
			if (value.toString ().equals (buf.toString ()))
				return false;
			setBuffer (reader, value, pos);
			return true;
		}
		else if (candidates.size () > 1)
		{
			String value = getUnambiguousCompletions (candidates);
			setBuffer (reader, value, pos);
		}

		reader.printNewline ();
		printCandidates (reader, candidates);

		// redraw the current console buffer
		reader.drawLine ();

		return true;
	}


	private static void setBuffer (ConsoleReader reader,
		String value, int offset)
		throws IOException
	{
		while (reader.getCursorBuffer ().cursor >= offset
			&& reader.backspace ());
		reader.putString (value);
		reader.setCursorPosition (offset + value.length ());
	}


	/** 
	 *  Print out the candidates. If the size of the candidates
	 *  is greated than the {@link getAutoprintThreshhold},
	 *  they prompt with aq warning.
	 *  
	 *  @param  candidates  the list of candidates to print
	 */
	private final void printCandidates (ConsoleReader reader,
		Collection candidates)
		throws IOException
	{
		// copy the values and make them distinct, without otherwise
		// affecting the ordering
		Collection copy = new LinkedList ();
		for (Iterator i = candidates.iterator (); i.hasNext (); )
		{
			Object next = i.next ();
			if (!(copy.contains (next)))
				copy.add (next);
		}

		candidates = copy;

		if (candidates.size () > reader.getAutoprintThreshhold ())
		{
			reader.printString (MessageFormat.format (
				loc.getString ("display-candidates"),
				new Object [] { new Integer (candidates.size ()) } ));

			reader.flushConsole ();

			int c;
			
			while ((c = reader.readCharacter ()) != -1)
			{
				if (loc.getString ("display-candidates-no")
					.startsWith (new String (new char [] { (char)c })))
				{
					reader.printNewline ();
					return;
				}
				else if (loc.getString ("display-candidates-yes")
					.startsWith (new String (new char [] { (char)c })))
					break;
				else
					reader.beep ();
			}
		}

		reader.printNewline ();
		reader.printColumns (copy);
	}




	/** 
	 *  Returns a root that matches all the {@link String} elements
	 *  of the specified {@link List}, or null if there are
	 *  no commalities. For example, if the list contains
	 *  <i>foobar</i>, <i>foobaz</i>, <i>foobuz</i>, the
	 *  method will return <i>foob</i>.
	 */
	private final String getUnambiguousCompletions (List candidates)
	{
		if (candidates == null || candidates.size () == 0)
			return null;

		// convert to an array for speed
		String [] strings = (String [])candidates.toArray (
			new String [candidates.size ()]);

		String first = strings [0];
		StringBuffer candidate = new StringBuffer ();
		for (int i = 0; i < first.length (); i++)
		{
			if (startsWith (first.substring (0, i + 1), strings))
				candidate.append (first.charAt (i));
			else
				break;
		}

		return candidate.toString ();
	}


	/** 
	 *  @return  true is all the elements of <i>candidates</i>
	 *  			start with <i>starts</i>
	 */
	private final boolean startsWith (String starts, String [] candidates)
	{
		// debug ("startsWith: " + starts);
		for (int i = 0; i < candidates.length; i++)
		{
			if (!candidates [i].startsWith (starts))
				return false;
		}

		return true;
	}
}

