package freenet.transport;

import freenet.*;
import freenet.support.io.*;
import freenet.support.Logger;
import java.io.*;
import java.net.Socket;
import java.net.InetAddress;
import java.util.HashMap;
import java.nio.*;

public final class tcpConnection extends Connection {
    
    private Socket sock;
    private InputStream in;
    private OutputStream out;
    private ByteBuffer accumulator;
    private NIOInputStream nioin;
    private NIOOutputStream nioout;
    private Object closeLock;
    private boolean closed = false;
    private boolean reallyClosed = false;
    private boolean shouldThrottle = false;
    private boolean shouldThrottleNow = false;
    private boolean instanceCounted = false; // if the constructor throws, WE STILL GET FINALIZED!
    // We do not throttle in FnpLink because it slows down negotiations drastically
    // We do not directly throttle, ever, because of nio.
    // Hence throttling is implemented in *SelectorLoop, not here
    // We should have a Bandwidth associated with the connection to support multiple independant throttles
    
    //profiling
    //WARNING:remove before release
    public static volatile int instances=0;
    public static volatile int openInstances=0;
    public static volatile long createdInstances=0;
    public static volatile long closedInstances=0;
    public static volatile long finalizedInstances=0;
    private static final Object profLock = new Object();
    static ReadSelectorLoop rsl;
    static WriteSelectorLoop wsl;
    private static Bandwidth ibw = null;
    private static Bandwidth obw = null;
    
    private static final int streamBufferSize() {
	return freenet.Core.streamBufferSize;
    }
    
    public static final HashMap socketConnectionMap = new HashMap();
    
    static synchronized public void setInputBandwidth(Bandwidth bw) {
	ibw = bw;
	if(rsl != null)
	    rsl.setBandwidth(ibw);
    }
    
    static synchronized public void setOutputBandwidth(Bandwidth bw) {
	obw = bw;
	if(wsl != null)
	    wsl.setBandwidth(obw);
    }
    
    static synchronized public void startSelectorLoops() {
	// Start NIO loops
	try {
	    if(rsl == null) {
		rsl = new ReadSelectorLoop(ibw);
		Thread rslThread = new Thread(rsl, " read interface thread");
		rslThread.setDaemon(true);
		rslThread.start(); // inactive until given registrations
	    }
	    if(wsl == null) {
		wsl = new WriteSelectorLoop(obw);
		Thread wslThread = new Thread(wsl, " write interface thread");
		wslThread.setDaemon(true);
		wslThread.start(); // inactive until given jobs
	    }
	} catch (Throwable t) {
	    System.err.println("Could not initialize network I/O system! Exiting");
	    t.printStackTrace(System.err);
	    System.exit(1);
	}
    }
    
    static public ReadSelectorLoop getRSL() {
	return rsl;
    }
    
    static public WriteSelectorLoop getWSL() {
	return wsl;
    }
    
    public boolean shouldThrottle() { return shouldThrottleNow; };
    
    public boolean countAsThrottled() { return shouldThrottle; };
    
    public void enableThrottle() { 
	if(Core.logger.shouldLog(Logger.DEBUG))
	    Core.logger.log(this, "Enabling throttle for "+this, 
			    new Exception("debug"), Logger.DEBUG);
	shouldThrottleNow = shouldThrottle;
    };
    
    public Socket getSocket() throws IOException {
	if(closed) throw new IOException("already closed "+this);
	else return sock;
    }
    
    /**
     * @return whether close() has been called. IMPORTANT NOTE: This does
     * not necessarily mean we have completed a full blocking close()!
     */
    public boolean isClosed() {
	return closed;
    }
    
    public boolean isInClosed() {
	NIOInputStream is = nioin;
	return (is == null) || (is.isClosed());
    }
    
    public boolean isOutClosed() {
	NIOOutputStream os = nioout;
	return (os == null) || os.isClosed();
    }
    
    static public Connection getConnectionForSocket(Socket s) {
	synchronized(socketConnectionMap) {
	    return (Connection)socketConnectionMap.get(s);
	}
    }
    
    private final tcpTransport t;
    
    private tcpConnection(tcpTransport t) {
        super(t);
        this.t = t;
	closeLock = new Object();
	startSelectorLoops();
    }
    
    public static boolean logBytes = false;
    
    /**
     * Used to create an outbound connection.
     */
    tcpConnection(tcpTransport t, tcpAddress addr,
		  boolean dontThrottle, boolean doThrottle)
	throws ConnectFailedException {
        this(t);
	
	boolean logDEBUG = Core.logger.shouldLog(Logger.DEBUG);
	if(logDEBUG)
	    Core.logger.log(this, "tcpConnection (outbound)", 
			    new Exception("debug"), Logger.DEBUG);
        
        try {
            long time = System.currentTimeMillis();
            sock = t.getSocketFactory().createSocket(addr.getHost(), addr.getPort());
	    if(wsl != null) wsl.putBandwidth(80); 
	    if(rsl != null) rsl.putBandwidth(80);
	    // FIXME: how much does a TCP handshake really cost?
	    if(sock == null) throw new IOException("could not create socket");
	    if(!sock.getTcpNoDelay()) {
		if(logDEBUG)
		    Core.logger.log(this, "Disabling Nagle's Algorithm!", 
				    Core.logger.DEBUG);
		sock.setTcpNoDelay(true);
	    }
            Core.diagnostics.occurrenceContinuous("socketTime",
                                                  System.currentTimeMillis() - time);
						  
	    /** NIO related stuff***/
	    sock.getChannel().configureBlocking(false);
	    accumulator = ByteBuffer.allocate(16*1024); //FIXME:hardcoded
	    accumulator.limit(0).position(0);
	    nioout = new NIOOutputStream(sock.getChannel(),this);
	    nioin = new NIOInputStream(accumulator,sock.getChannel(),this);
	    nioout.configWSL(wsl);
	    nioin.configRSL(rsl);
            //in = new MyBufferedInputStream(sock.getInputStream());
	    //in = new MyBufferedInputStream(nioin);
	    in = nioin;
	    // Buffering InputStreams on nio is a major issue due to
	    // handover issues. Leave it alone for now, maybe it's fast
	    // enough already.
	    if(dontThrottle) {
		//out = sock.getOutputStream();
		// Main reason for buffering is to deal with overhead
		// of 1 byte writes
		out = new BufferedOutputStream(nioout, streamBufferSize());
		if(logDEBUG) Core.logger.log(this, "Not throttling connection",
					     Core.logger.DEBUG);
	    } else {
		byte[] b = sock.getInetAddress().getAddress();
		
		if((!doThrottle) && (b[0] == (byte)127)) {
		    if(logDEBUG)
			Core.logger.log(this, "not throttling local connection", 
					Logger.DEBUG);
		    //out = sock.getOutputStream();
		    out = new BufferedOutputStream(nioout, streamBufferSize());
		    // Main reason for buffering is to deal with overhead
		    // of 1 byte writes
		} else if((!doThrottle) &&
			  ((b[0] == 10) || (b[0] == (byte)192 && b[1] == (byte)168) 
			   || (b[0] == (byte)172 && b[1] >= 16 && b[1] < 32))) {
		    if(logDEBUG)
			Core.logger.log(this, "not throttling LAN connection",
					Logger.DEBUG);
		    //out = sock.getOutputStream();
		    out = new BufferedOutputStream(nioout, streamBufferSize());
		} else {
		    if(logDEBUG)
			Core.logger.log(this, "Throttling connection",
					Core.logger.DEBUG);
		    out = new BufferedOutputStream(nioout, streamBufferSize());
		    shouldThrottle = true;
		}
	    }
	    if (logBytes)
		out = new DiagnosticsOutputStream(out);
	    out = new BufferedOutputStream(out);
        } catch (IOException e) {
	    if(!closed) {
		try {
		    closed = true;
		    if(sock != null) sock.close();
		} catch (IOException ex) {};
		synchronized(socketConnectionMap) {
		    socketConnectionMap.remove(sock);
		}
	    }
	    String desc = e.getMessage();
	    if(desc == null) desc = "(null)";
            throw new ConnectFailedException(addr, desc);
        } catch (RuntimeException e) {
	    if(!closed) {
		try {
		    closed = true;
		    if(sock != null) sock.close();
		} catch (IOException ex) {};
		synchronized(socketConnectionMap) {
		    socketConnectionMap.remove(sock);
		}
	    }
	    throw e;
	}
	if(logDEBUG)
	    Core.logger.log(this, "Created outbound tcpConnection "+this+" ("+
			    t+","+addr+","+dontThrottle+","+doThrottle+")",
			    new Exception("debug"), Logger.DEBUG);
	try {
	    synchronized(socketConnectionMap) {
		socketConnectionMap.put(sock, this);
	    }
	    //profiling
	    //WARNING:remove before release
	    synchronized(profLock) {
		instances++;
		openInstances++;
		createdInstances++;
		logInstances("outbound");
	    }
	    // Register AFTER on connection map
	    rsl.register(sock, nioin);
	} catch (Throwable e) {
	    Core.logger.log(this, "WTF? Failed final construction: "+
			    this+": "+e, e, Logger.ERROR);
	} // because of below
	// Here because then it only gets executed on a SUCCESSFUL construction
	instanceCounted = true;
    }
    
    protected final void logInstances(String s) {
	synchronized(profLock) {
		if(Core.logger.shouldLog(Logger.DEBUG))
	    Core.logger.log(this, s+" ("+this+") instances: "+instances+
			    ", openInstances: "+openInstances+
			    ", created: "+createdInstances+
			    ", closed: "+closedInstances+
			    ", finalized: "+finalizedInstances+
			    ", table: "+socketConnectionMap.size(),
			    openInstances>instances ? Logger.NORMAL : 
			    Logger.DEBUG);
	}
    }
    
    /**
     * Used to accept an incoming connection.
     */
    tcpConnection(tcpTransport t, Socket sock, int designator,
		  boolean dontThrottle, boolean doThrottle) throws IOException {
        this(t);
	boolean logDEBUG = Core.logger.shouldLog(Logger.DEBUG);
	if(logDEBUG)
	    Core.logger.log(this, "tcpConnection (inbound)", Logger.DEBUG);
        
        this.sock = sock;
	if(sock == null) throw new IllegalArgumentException("sock null");
	
	/** NIO related stuff***/
	sock.getChannel().configureBlocking(false);
	accumulator = ByteBuffer.allocate(16*1024); //FIXME:hardcoded
	accumulator.limit(0).position(0);
	nioout = new NIOOutputStream(sock.getChannel(),this);
	nioin = new NIOInputStream(accumulator,sock.getChannel(),this);
	nioout.configWSL(wsl);
	nioin.configRSL(rsl);
	// Buffering InputStreams on nio is a major issue due to
	// handover issues. Leave it alone for now, maybe it's fast
	// enough already.
	
        //      sock.setSoLinger(true, 0);
        if (designator < 0) {
            //in=new MyBufferedInputStream(sock.getInputStream());
	    in = nioin;
        } else {
            byte[] b = {(byte) ((designator >> 8) & 0xff) ,
            (byte) (designator & 0xff) };
            //in = new SequenceInputStream(new ByteArrayInputStream(b),
              //                           new MyBufferedInputStream(sock.getInputStream()));
	      in = new SequenceInputStream(new ByteArrayInputStream(b),
					   nioin);
        }
	//out = sock.getOutputStream();
	out = nioout; // Buffer set up below
	
	if(!sock.getTcpNoDelay()) {
	    if(logDEBUG)
		Core.logger.log(this, "Disabling Nagle's Algorithm!", 
				Logger.DEBUG);
	    sock.setTcpNoDelay(true);
	}

	if(dontThrottle) {
	    if(logDEBUG)
		Core.logger.log(this, "Not throttling incoming connection", 
				Logger.DEBUG);
	} else {
	    
	    byte[] b = sock.getInetAddress().getAddress();
	    
	    if((!doThrottle) && b[0] == (byte)127) {
		if(logDEBUG)
		    Core.logger.log(this, "not throttling local connection", 
				    Logger.DEBUG);
	    } else if ((!doThrottle) &&
		       ((b[0] == (byte)10) || (b[0] == (byte)192 && b[1] == (byte)168) || (b[0] == (byte)172 &&
b[1] >= 16 && b[1] < 32))) {
		if(logDEBUG)
		    Core.logger.log(this, "not throttling LAN connection",
				    Logger.DEBUG);
	    } else {
		if(logDEBUG)
		    Core.logger.log(this, "Throttling incoming connection", 
				    Logger.DEBUG);
		shouldThrottle = true;
	    }
	}
	if (logBytes)
	    out = new DiagnosticsOutputStream(out);
	out = new BufferedOutputStream(out);
	if(logDEBUG)
	    Core.logger.log(this, "Accepted inbound tcpConnection "+this+" ("+
			    t+","+sock+","+designator+","+dontThrottle+","+
			    doThrottle+")", new Exception("debug"), 
			    Logger.DEBUG);
	try {
	    synchronized(socketConnectionMap) {
		socketConnectionMap.put(sock, this);
	    }	
	    //profiling
	    //WARNING:remove before release
	    synchronized(profLock) {
		instances++;
		openInstances++;
		createdInstances++;
		logInstances("inbound");
	    }
	    rsl.register(sock, nioin);
	    // Register AFTER on connection map
	} catch (Throwable e) {
	    Core.logger.log(this, "WTF? Failed construction (B): "+
			    this+": "+e, e, Logger.ERROR);
	}
	// Here because then it only gets executed on a SUCCESSFUL construction
	instanceCounted = true;
    }
    
    /**
     * @return the input buffer, this will be set up for reading
     * i.e. position = 0, limit = end of bytes available ("flipped")
     */
    public ByteBuffer getInputBuffer() {
	return accumulator;
    }
    
    Exception closeException;
    
    public final void close() {
	close(false);
    }
    
    public final void close(boolean fromCloseThread) {
	closeException = new Exception("debug");
	boolean logDEBUG =Core.logger.shouldLog(Logger.DEBUG);  
	if(logDEBUG)
	    Core.logger.log(this, "Closing("+fromCloseThread+
			    ") tcpConnection "+this, 
			    closeException, Logger.DEBUG);
	if(fromCloseThread) {
	    closed = true;
	    synchronized(closeLock) {
		if(!reallyClosed) {
		    reallyClosed = true;
			if(logDEBUG) Core.logger.log(this, "reallyClosing "+this,
				    Logger.DEBUG);
		    try {
			
			if(sock != null) {
			    sock.getChannel().close();
			    sock.close();
			    wsl.onClosed(sock.getChannel());
			}
			// It will not change to null, but if we had an incomplete initialization and are being called from finalize(), it might BE null.
		    } catch (IOException e) {
			// It may have been closed remotely in which case
			// sock.close will throw an exception.  We really don't
			// care though.
			if(logDEBUG) Core.logger.log(this, "Caught IOException "+e+
					" closing "+this, e, Logger.DEBUG);
		    } catch (Throwable t) {
			System.err.println(t);
			t.printStackTrace();
			Core.logger.log(this, "Caught "+t+" closing "+this, t, Logger.ERROR);
			t = t.getCause();
			if(t != null) {
			    Core.logger.log(this, "Cause: "+t, t, Logger.ERROR);
			    t.printStackTrace();
			}
		    } finally {
			if(instanceCounted) {
			    synchronized(profLock) {
				openInstances--;
				closedInstances++;
				logInstances("closing");
			    }
			}
		    }
		}
		// Remove from map after really closing it
		if(sock != null) {
		    synchronized(socketConnectionMap) {
			socketConnectionMap.remove(sock);
		    }
		}
	    }
	    NIOInputStream ni = nioin;
	    if(ni != null) ni.closed();
		nioin = null;
		NIOOutputStream no = nioout;
		if(no != null) no.closed();
		nioout = null;
	} else {
	    if(!closed) rsl.queueClose(sock.getChannel());
	    closed = true;
	}
	try {
	    InputStream i = in;
	    if(i != null) i.close();
	} catch(Throwable e) {}
	in=new NullInputStream();
	
	try {
	    OutputStream o = out;
	    if(o != null) o.close();
	} catch(Throwable e) {}
	out=new SocketExceptionOutputStream("Connection already closed "+this);
    }
    
    private boolean finalized = false;
    
    protected void finalize() {
	if(finalized) return;
	finalized = true;
	logInstances("Finalizing");
	if(!closed) Core.logger.log(this, "finalized without being closed!"+this, Logger.NORMAL);
	try {
	    close(true);
	} catch (Throwable t) {
	    Core.logger.log(this, "Caught "+t+" closing "+this+" in finalize()", t, Logger.NORMAL);
	}
	if(!reallyClosed) Core.logger.log(this, "finalized without being reallyClosed!: "+this, Logger.NORMAL);
	//profiling
	//WARNING:remove before release
	synchronized(profLock) {
	    if(instanceCounted) {
		instances--;
		finalizedInstances++;
	    }
	    logInstances("finalized");
	}
    }
    
    public final InputStream getIn() {
        return in;
    }
    
    public final NIOInputStream getUnderlyingIn() {
	return nioin;
    }
    
    public final OutputStream getOut() {
        return out;
    }
    
    
    public final void setSoTimeout(int timeout) throws IOException {
        if (!closed)
            sock.setSoTimeout(timeout);
        else
            throw new IOException("Already closed "+this);
	if(Core.logger.shouldLog(Logger.DEBUG))
	    Core.logger.log(this, "Set SO_TIMEOUT to "+timeout+" for "+this,
			    Logger.DEBUG);
    }
    
    public final int getSoTimeout() throws IOException {
	if(!closed) return sock.getSoTimeout();
	else throw new IOException("Lost socket "+this);
    }
    
    public final Address getMyAddress(ListeningAddress lstaddr) {
        try {
            return new tcpAddress(t, sock.getLocalAddress(),
            ((tcpListeningAddress) lstaddr).getPort());
        }
        catch (BadAddressException e) {  // shouldn't really be possible
            return null;
        }
    }
    
    public final Address getMyAddress() {
        try {
            return new tcpAddress(t, sock.getLocalAddress(), sock.getLocalPort());
        }
        catch (BadAddressException e) {  // shouldn't really be possible
            return null;
        }
    }
    
    public final Address getPeerAddress(ListeningAddress lstaddr) {
        try {
            return new tcpAddress(t, sock.getInetAddress(),
            ((tcpListeningAddress) lstaddr).getPort());
        }
        catch (BadAddressException e) {  // shouldn't really be possible
            return null;
        }
    }
    
    public final Address getPeerAddress() {
	if(sock == null) return null;
        try {
            return new tcpAddress(t, sock.getInetAddress(), sock.getPort());
        }
        catch (BadAddressException e) {  // shouldn't really be possible
            return null;
        }
    }
    
    public final String toString() {
        // Socket.toString() does a bunch of stuff like reverse lookups.
        // no good.
        //return getTransport().getName()+" connection: " + sock;
        StringBuffer sb = new StringBuffer(getTransport().getName());
        sb.append("/connection: ");
	Socket sock = this.sock;
	if(closed) {
	    sb.append("CLOSED"); //you won't believe it till you see it with
	} else if (sock!=null) {
	    InetAddress addr = sock.getInetAddress();
	    if(addr != null) //this is becoming increasingly common --zab
		sb.append(addr.getHostAddress());
	    sb.append(":");
	    sb.append(sock.getPort());
	} else sb.append("null");
	sb.append(","+super.toString());
        return sb.toString();
    }
    
//     // FIXME: this is an evil workaround for bugs in NIO
    
//     protected class MyBufferedInputStream extends BufferedInputStream {
	
// 	protected MyBufferedInputStream(InputStream is) {
// 	    super(is);
// 	}
	
// 	public int read() throws IOException {
// 	    try {
// 		return super.read();
// 	    } catch (java.nio.channels.CancelledKeyException e) {
// 		throw new IOException("Channel cancelled already!");
// 	    }
// 	}
	
// 	public int read(byte[] data, int offset, int length)
// 	    throws IOException {
// 	    try {
// 		return super.read(data, offset, length);
// 	    } catch (java.nio.channels.CancelledKeyException e) {
// 		throw new IOException("Channel cancelled already!");
// 	    }
// 	}
	
// 	public void close() {
// 	    Core.logger.log(this, "Close() called - "+
// 			    tcpConnection.this, Logger.DEBUG);
// 	}
//     }
    
    
}



