/* -*- Mode: java; c-basic-indent: 4; tab-width: 4 -*- */
//TODO: override register and unregister so that they can't be used.


package freenet.transport;


import java.nio.channels.*;
import java.nio.*;
import java.util.*;
import java.io.IOException;
import freenet.Core;
import freenet.support.BlockingQueue;
import freenet.support.Logger;
import freenet.support.io.Bandwidth;
import freenet.support.sort.*;
import freenet.support.Comparable;

/**
 * a loop that writes data to the network. 
 * its operation is basically the opposite of 
 * the read loop.
 */

public final class WriteSelectorLoop extends ThrottledSelectorLoop {
  	
	/**
	 * queue where the sendjobs are placed
	 * this is different from the queues in ASL
	 */
	BlockingQueue jobs;
	
	/**
	 * hashtable which filters all requests to add something
	 * to the queue.
	 */
	Hashtable uniqueness;
	
	//TreeSet ts = new TreeSet();
	SortAlgorithm sorter;
	
	public final int uniquenessLength() {
		return uniqueness.size();
	}

	/**
	 * parameters of the uniquenss hashtable.  This depends entirely on 
	 * the machine power.  Its better to have it bigger than have 
	 * to refactor in runtime.
	 */
	private static final int TABLE_SIZE=512;
	private static final float TABLE_FACTOR=(float)0.6;

	/**
	 * nothing special about this constructor
	 */
	public WriteSelectorLoop(Bandwidth bw) throws IOException {
		
		super(bw);
		jobs = new BlockingQueue();
		uniqueness = new Hashtable(TABLE_SIZE,TABLE_FACTOR);
		sorter=new QuickSorter();
	}
	
	/**
	 * nothing special about this constructor
	 */
	public WriteSelectorLoop() throws IOException {
		
		super();
		jobs = new BlockingQueue();
		uniqueness = new Hashtable(TABLE_SIZE,TABLE_FACTOR);
		sorter=new QuickSorter();
	}
	
	protected Object idSync = new Object();
	protected long idCount = 0;
	/**
	 * these triplets enter the queue for sending.
	 */
	private final class SendJob implements NIOCallback, 
										   freenet.support.Comparable {
		public ByteBuffer data;
		public SelectableChannel destination;
		public NIOWriter client;
		public int position;
		public long id;
		
		public int compareTo(Object o) {
			if(o instanceof SendJob) {
				SendJob j = (SendJob)o;
				if(j.id>id) return -1;
				if(j.id<id) return 1;
				return 0;
			} else return -1;
		}
		
		public SendJob(byte [] _data, SelectableChannel destination, NIOWriter client) {
			this.data=ByteBuffer.wrap(_data);
			this.position = 0;
			this.destination = destination;
			this.client=client;
			synchronized(idSync) {
				id = idCount++;
			}
		}
		
		public SendJob(byte [] _data, int offset, int length, SelectableChannel destination, NIOWriter client) {
			this.data=ByteBuffer.wrap(_data,offset,length);
			this.position = offset;
			this.destination = destination;
			this.client=client;
			synchronized(idSync) {
				id = idCount++;
			}
		}
		
		public final String toString() {
			return SendJob.this.getClass().getName()+": "+data+","+destination+
				","+client+","+position+","+id;
		}
		
		public final void closed() {
			client.closed();
			if(uniqueness.containsKey(destination)) {
				Core.logger.log(this, "Removing "+this+
								" from uniqueness in SendJob.closed()!!",
								Logger.NORMAL);
				uniqueness.remove(destination);
			}
		}
		
		public final void queuedClose() {
			client.queuedClose();
			if(uniqueness.containsKey(destination)) {
				Core.logger.log(this, "Removing "+this+
								" from uniqueness in SendJob.queuedClose()!",
								Logger.NORMAL);
				uniqueness.remove(destination);
			}
		}
		
		public final void registered() {
			client.registered();
		}
		
		public final void unregistered() {
			client.unregistered();
		}
		
		public final boolean shouldThrottle() {
			return client.shouldThrottle();
		}
		
		public final boolean countAsThrottled() {
			return client.shouldThrottle();
		}
	}
	
	public final void onClosed(SelectableChannel sc) {
		if(sc == null) throw new NullPointerException();
		if(uniqueness.containsKey(sc)) {
			if (logDebug)Core.logger.log(this, "Removing "+sc+":"+uniqueness.get(sc)+
							" from uniqueness in onClosed()",
							Logger.DEBUG);
			uniqueness.remove(sc);
		}
		super.onClosed(sc);
	}
	
	/**
	 * this method adds a byte[] to be sent to a Channel. 
	 * it should be called from a different thread 
	 * return value: false if there's a job already on this channel.
	 */
	 public final boolean send(byte [] data, SelectableChannel destination, NIOWriter client) throws IOException{
	 
		if (data.length==0 || destination ==null) throw new IOException ("no data to send?");
		
	 	if (!checkValid(destination))return false;
		
		SendJob job = new SendJob(data,destination,client);
		
		//it doesn't really matter what the actual mapping is
		uniqueness.put(destination,job);
		if (logDebug)Core.logger.log(this, "Added "+destination+","+job+" to uniqueness, now "+uniqueness.size(), Logger.DEBUG);
		
		//because the objects are removed from this queue
		jobs.enqueue(job);
		if (logDebug)Core.logger.log(this, "Queued "+destination+","+job,
						Logger.DEBUG);
		return true;
	 }
	 
	 public final boolean send(byte [] data, int offset, int len, SelectableChannel destination, NIOWriter client) throws IOException{
	 
		if (len==0 || destination==null) throw new IOException ("no data to send?");
		
	 	if (!checkValid(destination))return false;
		
		SendJob job = new SendJob(data,offset,len,destination,client);
		
		//it doesn't really matter what the actual mapping is
		uniqueness.put(destination,job);
		if (logDebug)Core.logger.log(this, "Added "+destination+","+job+" to uniqueness, now "+uniqueness.size(), Logger.DEBUG);
		
		//because the objects are removed from this queue
		jobs.enqueue(job);
		if (logDebug)Core.logger.log(this, "Queued "+destination+","+job,
						Logger.DEBUG);
		return true;
	 }
	 
	 private final boolean checkValid(SelectableChannel destination) {
		//if we are already registered, return false
		SelectionKey key = destination.keyFor(sel);
		if (key != null && key.isValid()) {
			if(logDebug)
				Core.logger.log(this, "Send failed due to valid key",
								Logger.DEBUG);
			return false;
		}
		
		//check if we have this channel in the hashtable
		if (uniqueness.containsKey(destination)) {
			if(logDebug)
				Core.logger.log(this, "Send failed due to key in hash",
								Logger.DEBUG);
			return false;
		}
		
		return true;
	 }
	 
	 /**
	  * recent long waits in CHOS.write prompted me to do this here as well
	  */
	 protected final void fixKeys() {
	 	//this never adds anything
	 	/*Iterator i = sel.keys().iterator();
		while (i.hasNext()) {
			SelectionKey current = (SelectionKey) i.next();
			if (current.isValid() && current.isWritable())
				currentSet.add(current);
		}*/
	 }
	 /**
	  * overriden so that it will block when there is nothing
	  * to send.  Otherwise the selector enters an endless loop
	  */
	 protected final void beforeSelect() {
		 boolean success = false;
		 while(!success) {
			 try {
				 success = mySelect(0);
			 } catch (IOException e) {
				 Core.logger.log(this, "selectNow() failed in WSL.beforeSelect(): "+e,
								 e, Logger.ERROR);
			 }
		 }
		 throttleBeforeSelect(); // to avoid CancelledKeyExceptions
		 //this queue is local to the thread.
		 LinkedList waitingJobs = new LinkedList();
		 boolean logDEBUG = logDebug;
		 if(logDEBUG)
			 Core.logger.log(this, "beforeSelect()", Logger.DEBUG);
		 // first check if any channel is registered in the 
		 // selector.
		 try{
			 boolean firstIteration = true;
			 // We want to run the first iteration anyway in case there are
			 // some jobs to copy; but we do not want to block unless there
			 // is nothing else to do.
			 while(waitingJobs.isEmpty() && 
				   (firstIteration || (sel.keys().isEmpty()))) {
				 firstIteration = false;
				 if (sel.keys().isEmpty() && jobs.isEmpty()) {
					 //check if the jobs queue is empty,
					 //and if it is, block on it.
					 if(logDEBUG)
						 Core.logger.log(this, "Waiting for job to add to "+
										 "queue", Logger.DEBUG);
					 long delay = 0;
					 synchronized(throttleLock) {
						 long now = System.currentTimeMillis();
						 if(throttling && logDEBUG)
							 Core.logger.log(this, "Throttling at "+now+" until "+
											 reregisterThrottledTime, Logger.DEBUG);
						 delay = throttling ? (reregisterThrottledTime - now) : 0;
						 if(delay < 0) delay = 0;
						 if (logDebug)Core.logger.log(this, "Delay: "+delay, Logger.DEBUG);
					 }
					 int x = (delay > Integer.MAX_VALUE) ? Integer.MAX_VALUE
						 : ((int)delay);
					 if(logDEBUG) 
						 Core.logger.log(this, "Delay will be "+x+" ms: ",
										 Logger.DEBUG);
					 Object o = jobs.dequeue(x);
					 Core.logger.log(this, "Dequeued", Logger.DEBUG);
					 if(o != null) {
						 SendJob current = (SendJob)o;
						 synchronized(throttleLock) {
							 if(throttling && current.shouldThrottle()) {
								 ChannelAttachmentPair pair = 
									 new ChannelAttachmentPair(current.destination, current);
								 throttleDisabledQueue.add(pair);
								 if(logDEBUG)
									 Core.logger.log(this, "Moving new job "+
													 pair+" onto delay queue", 
													 Logger.DEBUG);
								 continue;
							 }
						 }
						 waitingJobs.add(o);
						 if(logDEBUG)
							 Core.logger.log(this, "Dequeued job",
											 Logger.DEBUG);
						 continue;
					 } else {
						 throttleBeforeSelect();
						 continue;
					 }
				 } else {
					 //don't block, just copy the elements to
					 //the local queue
					 long startTime = System.currentTimeMillis();
					 while(jobs.size() >0) {
						 SendJob current = (SendJob)(jobs.dequeue());
						 synchronized(throttleLock) {
							 if(throttling && current.shouldThrottle()) {
								 ChannelAttachmentPair pair = 
									 new ChannelAttachmentPair(current.destination, current);
								 throttleDisabledQueue.add(pair);
								 if(logDEBUG)
									 Core.logger.log(this, "Moving new job "+
													 pair+" onto delay queue", 
													 Logger.DEBUG);
								 continue;
							 }
						 }
						 if(logDEBUG)
							 Core.logger.log(this, "Copying job "+current+
											 " to queue", Logger.DEBUG);
						 waitingJobs.add(current);
					 }
					 long endTime = System.currentTimeMillis();
					 if(logDEBUG)
						 Core.logger.log(this, "Took "+(endTime-startTime)+
										 " millis copying queue ("+iter+")", 
										 Logger.DEBUG);
				 }
			 }
		 } catch(InterruptedException e) {
			 Core.logger.log(this, "Interrupted: "+e, e, Logger.NORMAL);
			 e.printStackTrace();
		 }
		 
		 
		//register the channel  with the selector.
		//do not remove the channel from the uniqueness table
		Iterator i = waitingJobs.iterator();
		while (i.hasNext()) {
			SendJob currentJob = (SendJob) i.next();
			try {
				if(logDEBUG)
					Core.logger.log(this, "Registering channel "+currentJob+
									" with selector", Logger.DEBUG);
				currentJob.destination.register(sel, SelectionKey.OP_WRITE, currentJob);
				if(logDEBUG)
					Core.logger.log(this, "Registered channel "+currentJob+
									" with selector", Logger.DEBUG);
			}catch (ClosedChannelException e) {
				if(logDEBUG)
					Core.logger.log(this, "Channel closed: "+currentJob+": "+e,
									e, Logger.DEBUG);
				queueClose(((SocketChannel)currentJob.destination),
						   currentJob.client);
				uniqueness.remove((SocketChannel)currentJob.destination);
				if (logDebug)Core.logger.log(this, "Removed "+currentJob+" from uniqueness due to ClosedChannelException, now "+uniqueness.size(), Logger.DEBUG);
				//TOTHINK: perhaps move all callbacks outside of WSL thread
				try {
					currentJob.client.jobDone(0,false);
				} catch (Throwable t) {
					Core.logger.log(this, "Caught "+t+" notifying "+
									currentJob.client+" for "+
									currentJob.destination+" in beforeSelect",
									t, Logger.ERROR);
				}
			} catch (IllegalBlockingModeException e) {
				Core.logger.log(this, "Could not register channel "+currentJob+
								" with selector: "+currentJob+": "+e, e, 
								Logger.ERROR);
				e.printStackTrace();
			}
		}
	 }
	 
	//check if any of the channels got closed;
	//REDFLAG: the behavior of select() on channels that have
	//been closed remotely is not tested.
	//TEST PROPERLY AND IMPLEMENT CHECKS HERE!!!
	protected final boolean inspectChannels() {
		if(logDebug)
			Core.logger.log(this, "inspectChannels()", Logger.DEBUG);
		//throw new UnsupportedOperationException();
		return false;
	}
	
	//at this stage the selected set should contain only channels
	//that are ready to be written to and have something to be sent.
	protected final boolean processConnections() {
		boolean logDEBUG = logDebug;
		if(logDEBUG)
			Core.logger.log(this, "processConnections()", Logger.DEBUG);
		boolean success = true;
		int throttledBytes = 0;
		int pseudoThrottledBytes = 0;
		int bytesSent = 0;
		Iterator i = sel.selectedKeys().iterator();
		Vector ts =new Vector(sel.selectedKeys().size());
		while(i.hasNext()) {
			SelectionKey curKey = (SelectionKey)i.next();
			//if (!(curKey.isValid() && curKey.isWritable() && curKey.channel().isOpen())) continue;
			SendJob currentJob = (SendJob)(curKey.attachment());
			ts.add(currentJob);
		}
		sorter.sort(new VectorSorter(ts));
		if (logDebug)Core.logger.log(this, "Sorted jobs, "+ts.size(), Logger.DEBUG);
		i = ts.iterator();
		long prevID = -1;
		try{
			boolean noThrottled = (System.currentTimeMillis() < 
								   reregisterThrottledTime);
			boolean noMoreThrottled = false;
			// Some of them may have been enableThrottle()d off thread
		while (i.hasNext()) {
			boolean localSuccess = true;
			SendJob currentJob = (SendJob)(i.next());
			SelectionKey curKey = currentJob.destination.keyFor(sel);
			long id = currentJob.id;
			if(id < prevID)
				Core.logger.log(this, "ID less than prevID: "+id+"<"+prevID,
								Logger.ERROR);
			prevID = id;
			if(currentJob == null || currentJob.data.remaining() <= 0) {
				curKey.cancel(); // leave running, but cancel
				if(curKey.channel() != null) {
					uniqueness.remove(curKey.channel());
					if (logDebug)Core.logger.log(this, "Removed "+curKey.channel()+
									" from uniqueness, now "+uniqueness.size(),
									Logger.DEBUG);
				}
				Core.logger.log(this, "Cancelled "+currentJob+
								" - already done?", Logger.ERROR);
			}
			
			//do the write.. 
			int sent = 0;
			try {
				if(!currentJob.destination.isOpen())
					throw new IOException("closed");
				if(currentJob.destination instanceof SocketChannel &&
				   (!((SocketChannel)(currentJob.destination)).
					isConnected())) throw new IOException("not connected");
				if(currentJob.client.shouldThrottle()) {
					if(noMoreThrottled) {
						// Will get cancelled by throttleConnections
						if(logDEBUG)
							Core.logger.log(this, "Skipping (A) throttled "+
											currentJob, Logger.DEBUG);
						continue;
					} else if(noThrottled) {
						// May not get cancelled by throttleConnections
						// shouldThrottle overrides countAsThrottled
						Core.logger.log(this, "Job apparently became throttled"+
										" after registration: "+currentJob,
										Logger.MINOR);
						curKey.cancel();
						ChannelAttachmentPair pair = 
							new ChannelAttachmentPair(currentJob.destination,
													  currentJob);
						synchronized(throttleLock) {
							throttleDisabledQueue.add(pair);
						}
						continue;
					}
				}
				int oldLimit = currentJob.data.limit();
				if(bw != null && currentJob.client.shouldThrottle() &&
				   currentJob.data.remaining() > 
				   bw.maximumPacketLength()) {
					currentJob.data.limit(currentJob.data.position() +
										  bw.maximumPacketLength());
					if(logDEBUG)
						Core.logger.log(this, "Limited: "+currentJob.data+
										" for "+currentJob.client+
										", limit was "+oldLimit, Logger.DEBUG);
				} else {
					if(logDEBUG)
						Core.logger.log(this, "Did not limit, "+
										currentJob.data.remaining()+"/"+
										(bw==null?"(null)": Integer.
										 toString(bw.maximumPacketLength()))+
										" for "+currentJob.client, 
										Logger.DEBUG);
				}
				try {
					sent = ((SocketChannel)curKey.channel()).
						write(currentJob.data);
				} finally {
					if(currentJob.data.limit() != oldLimit)
						currentJob.data.limit(oldLimit);
				}
				//if this was an incomplete write, leave the buffer as it is
			} catch (IOException e) {
				localSuccess=false;
				if (logDebug)Core.logger.log(this, "IOException: "+e+" writing to channel "+
								currentJob, e, Logger.DEBUG);
				queueClose((SocketChannel)currentJob.destination,
						   currentJob.client);
			} finally {
				if(sent > 0) {
					bytesSent+=(sent+OVERHEAD);
					if(currentJob.client.shouldThrottle()) {
						throttledBytes += (sent+OVERHEAD);
						if(logDEBUG)
							Core.logger.log(this, "Should throttle "+
											currentJob, Logger.DEBUG);
					} else {
						if(currentJob.client.countAsThrottled()) {
							pseudoThrottledBytes += (sent+OVERHEAD);
							if(logDEBUG)
								Core.logger.log(this, "Pseudo-throttle "+
												currentJob, Logger.DEBUG);
						} else {
							if(logDEBUG)
								Core.logger.log(this, "Should not throttle "+
												currentJob, Logger.DEBUG);
						}
					}
					if (logDebug)Core.logger.log(this, "Sent "+sent+" bytes on "+
									currentJob+", bytesSent="+bytesSent+
									", throttledBytes="+ throttledBytes+
									", psuedoThrottledBytes="+
									pseudoThrottledBytes, Logger.DEBUG);
					// In any case, demote it
					synchronized(idSync) {
						currentJob.id = idCount++;
					}
				}
				//cancel the key and remove it from the uniqueness
				//and notify Callback 
				//also notify if we get an exception
				if ((!localSuccess) || currentJob.data.remaining() == 0) {
					if (logDebug)Core.logger.log(this, "Finishing "+currentJob,
									Logger.DEBUG);
					curKey.cancel();
					if (logDebug)Core.logger.log(this, "Removing (B) "+currentJob+" ("+
									curKey+") from uniqueness, now "+
									uniqueness.size(), Logger.DEBUG);
					uniqueness.remove(curKey.channel());
					if (logDebug)Core.logger.log(this, "Removed (B) "+currentJob+" ("+
									curKey+") from uniqueness, now "+
									uniqueness.size(), Logger.DEBUG);
					try {
						currentJob.client.jobDone(currentJob.data.position()-currentJob.position,localSuccess);
					} catch (Throwable t) {
						Core.logger.log(this, "Caught "+t+" notifying "+
										currentJob.client+" for "+
										currentJob.destination+
										" in processConnections", t, 
										Logger.ERROR);
					}
					if (logDebug)Core.logger.log(this, "Finished "+currentJob,
									Logger.DEBUG);
				} else {
					if (logDebug)Core.logger.log(this, "Not finished "+currentJob,
									Logger.DEBUG);
				}
			}
			if(currentJob.client.shouldThrottle() && sent > 0) {
				// Throttle this one first
				if (logDebug)Core.logger.log(this, "Set noMoreThrottled because of "+
								currentJob, Logger.DEBUG);
				noMoreThrottled = true;
			}
		}
		if(logDEBUG)
			Core.logger.log(this, "Written "+bytesSent, Logger.DEBUG);
		if(throttledBytes != 0) {
			if(logDEBUG)
				Core.logger.log(this, "Written "+throttledBytes+
								" bytes that should be throttled", 
								Logger.DEBUG);
			throttleConnections(bytesSent, throttledBytes,
								pseudoThrottledBytes);
		}
		} catch(Throwable e) {
			Core.logger.log(this, "Exception in processConnections(): "+
							e, e, Logger.ERROR);
			e.printStackTrace();
			success=false;
		} finally {
			currentSet.clear();
		}
		return success;
	}
	
	protected final boolean shouldThrottle(Object o) {
		SendJob sj = (SendJob)o;
		NIOCallback cb = sj.client;
		return cb.shouldThrottle();
	}
	
	//again, eventually the catches will come up here
	public final void run() {
		loop();
	}
	
	protected final void queueClose(ClosePair chan) {
		if (logDebug)Core.logger.log(this, "queueClose("+chan+"), uniqueness "+
						uniqueness.size(), Logger.DEBUG);
		if(chan.sc != null) {
			uniqueness.remove(chan.sc);
			if (logDebug)Core.logger.log(this, "Removed(C) "+chan+" from uniqueness, now "+
							uniqueness.size(), Logger.DEBUG);
		}
		super.queueClose(chan);
	}
	
	protected final void onInvalidKey(SelectionKey key) {
		SocketChannel chan = (SocketChannel)(key.channel());
		if(chan != null) {
			if(uniqueness.containsKey(chan)) {
				uniqueness.remove(chan);
				if (logDebug)Core.logger.log(this, "Removed "+key+" from uniqueness due to channel being invalid, uniqueness now "+uniqueness.size(), Logger.DEBUG);
			}
		}
	}
	
	public final String analyzeUniqueness() {
		StringBuffer out = new StringBuffer();
		SocketChannel c[] = new SocketChannel[uniqueness.size()];
		c = (SocketChannel[])(uniqueness.keySet().toArray(c));
		for(int x=0;x<c.length;x++) {
			SocketChannel chan = c[x];
                        if(!chan.isOpen()) {
                                out.append("NOT OPEN: ").append(chan);
                                Object o = uniqueness.get(chan);
				if(o != null) out.append(":").append(o);
				out.append('\n');
                        } else if(!chan.isConnected()) {
                                out.append("NOT CONNECTED: ").append(chan);
                                Object o = uniqueness.get(chan);
				if(o != null) out.append(":").append(o);
				out.append('\n');
			} else {
                                out.append("Running: ").append(chan);
                                Object o = uniqueness.get(chan);
				if(o != null) out.append(":").append(o);
				out.append('\n');
			}
		}
		return out.toString();
	}
	
	protected final int myKeyOps() {
		return SelectionKey.OP_WRITE;
	}
	
	//TODO: close, log, etc.
	public final void close() {}
  }
