/* -*- Mode: java; c-basic-indent: 4; tab-width: 4 -*- */
package freenet.transport;

import java.nio.channels.*;
import java.nio.ByteBuffer;
import freenet.SelectorLoop;
import java.net.Socket;
import java.util.*;
import java.io.IOException;
import freenet.Connection;

// For logging
import freenet.Core;
import freenet.support.Logger;

/**
 * An abstract SelectorLoop.  Subclasses will be the interface selector
 * and the polling selector.
 * In this implementation, all the register/unregister methods are to be
 * called from a different thread.
 */

public abstract class AbstractSelectorLoop implements SelectorLoop{
	protected LinkedList registerWaiters, unregisterWaiters, delayedWaiters;
	protected Selector sel;
	
	protected LinkedList currentSet;
	protected ByteBuffer []buffers;
	
	private static LinkedList closeQueue = null;
	private static LinkedList preCloseQueue = null;
	private static boolean isWindows = (System.getProperty("os.name").toLowerCase().indexOf("windows") != -1);
	
	private static CloseThread closeThread= null;
	
	private static Hashtable closeUniqueness = new Hashtable(512); //FIXME:hardcoded
	
	protected static boolean logDebug;
	
	public int closeUniquenessLength() {
		return closeUniqueness.size();
	}
	
	/**
	 * Checks if this is the infamous windows 'select'-bug. Lets the NPE pass through if it isn't
	 */
	protected static void filterWindowsSelectBug(Object source,NullPointerException e) throws NullPointerException
	{
		StackTraceElement[] t = e.getStackTrace();
		if(t.length > 0 && (t[0].getMethodName().equals("processFdSet") || t[0].getMethodName().equals("processFDSet")) && t[0].getClassName().equals("sun.nio.ch.WindowsSelectorImpl$SubSelector"))  //This seems to be the case for the particular exception we are hunting for. Should we maybe tie this to windows OS only?
		{
			if(Core.logger.shouldLog(Logger.MINOR))
				Core.logger.log(source,"Worked around Sun windows JVM	'select'-bug (Sun BugId: 4729342), please update your JVM to a fixed version if possible",Logger.MINOR);
			try {
				Thread.sleep(50); //This problem might cause the selector to spin. Make that a nicer experience to the user, both log-wise and CPU-wise
			} catch (InterruptedException e1) {

			}
			return;
		}else{
			throw e;
		}
	}
	
	//when this goes up, the loop stops
	protected volatile boolean stop;


	//it is a timeout for the selector loop in ms.
	//after it expires the select thread does maintenance
	//it should be fine-tuned after extensive testing.
	protected final static int TIMEOUT = 200;
	
	//the current timeout we use
	protected int timeout;




	//whether clearing of the bufferes is needed.
	//it will only be true in a subclass, but I want it here
	private boolean cleanupNeeded;
	

	public AbstractSelectorLoop() throws IOException{
		sel = Selector.open();
		registerWaiters = new LinkedList();
		unregisterWaiters = new LinkedList();
		delayedWaiters = new LinkedList();
		stop = false;
		cleanupNeeded=false;
		timeout = TIMEOUT;
		if (closeQueue == null)
			closeQueue = new LinkedList();
		if (preCloseQueue == null)
			preCloseQueue = new LinkedList();
		if (closeThread == null) {
			closeThread = new CloseThread();
			closeThread.setDaemon(true);
			closeThread.start();
		}
		currentSet=new LinkedList();//list is even better HashSet(1024); //bigger is better
		

	}


	/**
	 * these pairs enter the register queue
	 */
	protected class ChannelAttachmentPair {
		public SelectableChannel channel;
		public Object attachment;

		public ChannelAttachmentPair(SelectableChannel chan,Object attachment) {
			this.channel = chan;
			this.attachment = attachment;
		}
		
		public String toString() {
			return channel.toString()+":"+((attachment == null) ? "(null)" :
										   attachment.toString());
		}
	}
	
	protected final class DelayedChannelAttachmentPair extends ChannelAttachmentPair {
		public final long registerTime;
		
		public DelayedChannelAttachmentPair(SelectableChannel chan,Object attachment, long delay){
			super(chan,attachment);
			registerTime = System.currentTimeMillis()+delay;
		}
	}
	/**
	 * these pairs enter the close queue
	 */
	protected final class ClosePair {
		public Connection conn;
		public Object attachment;
		public SocketChannel sc;

		public ClosePair(Connection conn, Object attachment, 
						 SocketChannel sc) {
			this.conn = conn;
			this.attachment = attachment;
			this.sc = sc;
		}
		
		public String toString() {
			return conn.toString()+":"+((attachment == null) ? "(null)" :
										attachment.toString())+":"+sc;
		}
	}
	

	public void register(SelectableChannel ch, Object attachment) throws IllegalBlockingModeException{
		if (ch == null) {
			Core.logger.log(this, "Selectable channel is NULL", Logger.ERROR);
			return;
		}
		//minimize the number of exceptions in the select thread, check early
		if (ch.isBlocking()) throw new IllegalBlockingModeException();
		synchronized(registerWaiters) {
			registerWaiters.add(new ChannelAttachmentPair(ch, attachment));
		}
	}

	public void register(Socket sock, Object attachment) throws IllegalBlockingModeException {
		//make sure this is a nio socket
		if (sock.getChannel() == null) throw new IllegalBlockingModeException();
		register(sock.getChannel(),attachment);
	}

	public void unregister(SelectableChannel chan) {
		if(logDebug)
			Core.logger.log(this, "Unregistering "+chan, 
							new Exception("debug"), Logger.DEBUG);
		synchronized(unregisterWaiters) {
			unregisterWaiters.add(new ChannelAttachmentPair(chan,null));
		}
	}

	public void unregister(Object attachment) {
		if(logDebug)
			Core.logger.log(this, "Unregistering "+attachment, 
							new Exception("debug"), Logger.DEBUG);
		synchronized(unregisterWaiters) {
			unregisterWaiters.add(new ChannelAttachmentPair(null,attachment));
		}
	}

	public boolean isOpen() {
		return sel.isOpen();
	}
	/*******************************************
	 * these methods are to be called from within the selector loop thread.
	 * they must be as fast as possible and NEVER block.
	 ******************************************/
	
	public boolean shouldRegister(ChannelAttachmentPair chan) {		
		return true;
	}
	
	protected abstract int myKeyOps();
	
	/**
	 * register/unregister waiters in the queues
	 * to the selector
	 * another reason for this thread to have high priority -
	 * we don't want it to wait for processing the queues.
	 */
	protected final void processWaiters() {

		while (delayedWaiters.size() > 0) {
			DelayedChannelAttachmentPair current = 
				(DelayedChannelAttachmentPair) delayedWaiters.removeFirst();
			if (System.currentTimeMillis() >= current.registerTime)
				registerWaiters.add(current);
			else
				delayedWaiters.add(current);
		}
		//TODO: find a way to get the try's out of the loop
		//first add
		LinkedList notRegistered = new LinkedList();
		synchronized(registerWaiters) {
			while(registerWaiters.size() >0) {
				ChannelAttachmentPair current = (ChannelAttachmentPair)registerWaiters.removeFirst();
				
				if (myKeyOps() == SelectionKey.OP_ACCEPT) //this is a server socket
					try{
						current.channel.register(sel, SelectionKey.OP_ACCEPT, current.attachment);
					}catch(ClosedChannelException e) {continue;}
				else if (myKeyOps() == SelectionKey.OP_READ) {
					//a reader, writers get registered on the fly
					if(!current.channel.isOpen()) continue;
					if(shouldRegister(current)) {
						try {
							if (current.channel.keyFor(sel) ==null ||
								!current.channel.keyFor(sel).isValid()) {
									current.channel.register(sel, SelectionKey.OP_READ,
													 current.attachment);
									((NIOCallback)current.attachment).registered();
							}
						} catch(ClosedChannelException e) {
							if(current.attachment instanceof NIOCallback)
								((NIOCallback)current.attachment).
									unregistered();
							try {
								queueClose(current);
							} catch (IllegalArgumentException x) {
								Core.logger.log(this, "Could not queue close for "+current+": "+x,x,Logger.ERROR);
							}
							continue;
						} catch (ClassCastException e) {
							// Not an NIOCallback
							continue;
						}
					} else {
						notRegistered.add(current);
					}
				}
			}
			while(!notRegistered.isEmpty()) {
				ChannelAttachmentPair current = (ChannelAttachmentPair)notRegistered.removeFirst();
				registerWaiters.add(current);
			}
		}
		
		//then remove
		synchronized(unregisterWaiters) {
		while (unregisterWaiters.size() >0) {
			ChannelAttachmentPair current = (ChannelAttachmentPair)unregisterWaiters.removeFirst();
			if (current.channel!=null) //we have a channel
				current.channel.keyFor(sel).cancel();
			// Not used by WSL, so we don't need to tell it
			else if (current.attachment!=null) { //we have only the attachment
				Iterator i = sel.keys().iterator();
				while(i.hasNext()) {
					SelectionKey curKey = (SelectionKey)i.next();
					if (curKey.attachment().equals(current.attachment)){
						curKey.cancel();
						break;
					}
				}

			}
			if (current.attachment!=null)
				try{
					((NIOCallback)current.attachment).unregistered();
				}catch(ClassCastException e) {continue;}
		}
		}
	}






	/**
	 * returns true if none of the ready channels had any data,
	 * i.e. all of them are screwed.
	 */
	protected abstract boolean inspectChannels();

	/**
	 * this fixes the keyset.  We're missing reads
	 */
	protected abstract void fixKeys();
	
	/**
	 * last-mile remedy if the selector is totaly screwed.
	 * this may very well have unforeseen consequences.
	 * lets hope we never get to use this...
	 */
	private final void reset() {
		Iterator i = sel.keys().iterator();
		Vector channels = new Vector(sel.keys().size());
		
		while (i.hasNext()) {
			SelectionKey current = (SelectionKey)i.next();
			if(current.channel().isOpen())
				channels.add(new ChannelAttachmentPair(current.channel(),current.attachment()));
			current.cancel();
		}
		
		try {
			sel.close();
			
			sel = Selector.open();
			
			
			i = channels.iterator();
			while (i.hasNext()) {
				ChannelAttachmentPair current = (ChannelAttachmentPair)i.next();
				current.channel.register(sel,myKeyOps(),current.attachment);
			}
		}catch(IOException e) {} //at this moment we're already in deep trouble
		//catch this exception won't help much
		catch(NullPointerException e) {
			AbstractSelectorLoop.filterWindowsSelectBug(this,e);
			//TODO: Do something here?
		}
		
	}
	
	
	/**
	 * works the connections on the ready set.
	 * to be overrriden by subclasses.
	 * I want to throw as few exceptions as possible here, so its return
	 * value indicates whether it finished processing.
	 */
	protected abstract boolean processConnections();


	/**
	 * Performs maintenace stuff on the loop.
	 * There really isn't anything necessary I can think of,
	 * so this method is empty.  Subclasses are free to override it.
	 */
	protected void performMaintenance(){

	}

	/**
	 * do something before blocking.  override if you need
	 * the reader for example would call clearBuffers here
	 */
	protected void beforeSelect() {}
	
	/* Handle the process of moving close-items from the 
	 * preCloseQueue to the actual closeQueue. This method should be called
	 * by the selector thread and allows it to control when actual
	 * channels are closed
	 */
	private void handleCloseQueuePipe()
	{
		synchronized(closeQueue) {
			synchronized(preCloseQueue) {
				boolean wasEmpty = closeQueue.isEmpty();
				while(!preCloseQueue.isEmpty())
					closeQueue.addLast(preCloseQueue.removeFirst());
				if(wasEmpty)
					closeQueue.notify();
			}
		}	
	}
	
	protected final boolean mySelect(int x) throws IOException {
		try{
			if(x == 0) {
				//Core.logger.log(this, "selectNow()", Logger.DEBUG);
				currentlyActive = sel.selectNow();
			} else {
				long start = System.currentTimeMillis();
				currentlyActive = sel.select(x);
				long end = System.currentTimeMillis();
				if((end - start) < 2) fastReturn = true;
			}
		} catch(ClosedChannelException e) {
			Core.logger.log(this, "mySelect caught "+e, e, Logger.MINOR);
			return false;
		} catch (IOException e) {
			String msg = e.getMessage().toLowerCase();
			if(msg.indexOf("interrupted system call") != -1) {
				Core.logger.log(this, "Caught interrupted system call: "+e, 
								e, Logger.MINOR);
				return false;
			} else if(msg.indexOf("number of specified semaphore events") 
					  != -1) {
				Core.logger.log(this, "Trying to workaround {"+msg+"}",
								Logger.NORMAL);
				try {
					reset();
				} catch (Throwable t) {
					String err = "Workaround for {"+msg+
						"} failed: "+t+" - report to "+
						"support@freenetproject.org, with "+
						"stack trace";
					Core.logger.log(this, err, t, Logger.ERROR);
					System.err.println(err);
					e.printStackTrace(System.err);
					t.printStackTrace(System.err);
					System.exit(88);
					// let logger continue, and we may want some post mortem
				}
			} else if(msg.indexOf("the parameter is incorrect") == 0) {
				try {
					Core.logger.log(this, "Error: "+e+" - trying to workaround", 
									Logger.NORMAL);
					String jvmver = "";
					try {
						jvmver = System.getProperty("java.vm.version");
					} catch (Throwable t) {};
					if(jvmver.startsWith("1.4.0")) {
						Core.logger.log(this, "JVM version: "+jvmver+
										" - java 1.4.0 has major problems with NIO, "+
										"you should upgrade to 1.4.1 or later", 
										Logger.ERROR);
					}
					reset();
				} catch (Throwable t) {
					String err = "Workaround for "+e+" failed: "+t;
					Core.logger.log(this, err, t, Logger.ERROR);
					System.err.println(err);
					e.printStackTrace(System.err);
					t.printStackTrace(System.err);
					System.exit(88);
					// let logger continue, and we may want some post mortem
				}
			} else {
				throw e;
			}
		} catch(CancelledKeyException e) {
			Core.logger.log(this, "mySelect caught "+e, e, Logger.MINOR);
			return false;
		} catch(NullPointerException e) {
			AbstractSelectorLoop.filterWindowsSelectBug(this,e);
			try {
				Thread.sleep(100); //This bug sometimes causes the selector to stop blocking. This makes a machine useable even if that happens
			} catch (InterruptedException ex) {};
			//continue; //TODO: Is continue the right thing to do here or should we do nothing instead?
		} catch (Error e) {
			if(e.getMessage().indexOf("POLLNVAL") >= 0) {
				// GRRRRR!
				Core.logger.log(this, "POLLNVAL detected in "+this+" ("+e+
								"), trying to workaround - I hate JVMs!", 
								Logger.NORMAL);
				try {
					reset();
				} catch (Throwable t) {
					Core.logger.log(this, "POLLNVAL workaround failed!: "+t+
									" - report to support@freenetproject."+
									"org with stack trace", t, Logger.ERROR);
					System.err.println("POLLNVAL workaround failed!: "+t+
									   " - report to support@freenetproject"+
									   ".org with stack trace");
					e.printStackTrace();
					t.printStackTrace();
					System.exit(88);
					// let logger continue, and we may want some post mortem
				}
				return false;
			} else throw e;
		}
		return true;
	}
	
	long iter=0;
	int currentlyActive=0;
	boolean fastReturn = false;
	
	/**
	 * the actual selector loop itself.
	 * lets try to make this as smart as possible.
	 * I want to have this loop in a single location, so that
	 * children won't mess with it.
	 */
	protected final void loop() {
		int consecutiveDuds = 0;
		while(!stop) {
			logDebug = Core.logger.shouldLog(Logger.DEBUG);
			try{ //a very wide and generic net
			beforeSelect();
			//moved this up
			//process any changes to the channels
			processWaiters();
			//sel.selectedKeys().clear();
			//select on the selector
			iter++;
			if(!mySelect(timeout)) continue;
			if(isWindows)
				handleCloseQueuePipe();
						
//			Core.logger.log(this, "Returned from selector, "+currentlyActive+
//							" connections ("+iter+")", Logger.DEBUG);
			
			//currentSet=sel.keys();
			currentSet.clear();
			currentSet.addAll(sel.selectedKeys());
			
			if(logDebug)
				Core.logger.log(this, "Keys ready before fixKeys: "+
								currentlyActive+"/"+
								sel.keys().size(), Logger.DEBUG);
			
			fixKeys();
			//if (currentlyActive != currentSet.size()) Core.logger.log(this, "read the freaking book! "+ currentlyActive +" != "+ currentSet.size() ,Logger.ERROR);
			currentlyActive = currentSet.size();
			if(logDebug)
				Core.logger.log(this, "Keys ready: "+currentlyActive+"/"+
								sel.keys().size(), Logger.DEBUG);
			
			//if at this point no channels are active, it means
			//we were just woken up or timed out.
			if(currentlyActive == 0) {
				//Core.logger.log(this, "Performing maintenance on selector ("
				//iter+")", Logger.DEBUG);
				performMaintenance();
				//continue; - keys might be usable BEFORE selection
			}
			
			
			//if it is more than 0, we ewhile (readySockets.size() > 0) {ither have data coming or a
			//screwed channel.

			//remove the screwed channels, if all of them were screwed, continue
			
			if(inspectChannels()) {
				if(fastReturn) {
					consecutiveDuds++;
					if(consecutiveDuds > 5) {
						Thread.sleep(50);
						consecutiveDuds = 0;
					}
				}
				continue;
			} else consecutiveDuds = 0;
			
			//and if not, process the rest
			
			if (!processConnections()) throw new Exception();
					//can't think of anything smarter at this time
					//some quick-but-sophisticated recovery mechanism is needed
					//at least log it
			
		} catch(OutOfMemoryError t) {
			String status = "dump of interesting objects before gc:"+
						"\ntcpConnections " +freenet.transport.tcpConnection.instances+
						"\nFnpLinks " +freenet.session.FnpLink.instances+
						"\nNIOOS " + freenet.support.io.NIOOutputStream.instances+
						"\nNIOIS " + freenet.support.io.NIOInputStream.instances+
						"\nCH " + freenet.ConnectionHandler.instances+
						"\nCHIS " +freenet.ConnectionHandler.CHISinstances+
						"\nCHOS " +freenet.ConnectionHandler.CHOSinstances+
						"\nRIS " +freenet.ConnectionHandler.RISinstances+
						"\nSOS " +freenet.ConnectionHandler.SOSinstances;
			System.err.println(status);
			Core.logger.log(this, status, Logger.ERROR);
			System.gc();
			System.runFinalization();
			System.gc();
			System.runFinalization();
			status = "dump of interesting objects after gc:"+
						"\ntcpConnections " +freenet.transport.tcpConnection.instances+
						"\nFnpLinks " +freenet.session.FnpLink.instances+
						"\nNIOOS " + freenet.support.io.NIOOutputStream.instances+
						"\nNIOIS " + freenet.support.io.NIOInputStream.instances+
						"\nCH " + freenet.ConnectionHandler.instances+
						"\nCHIS " +freenet.ConnectionHandler.CHISinstances+
						"\nCHOS " +freenet.ConnectionHandler.CHOSinstances+
						"\nRIS " +freenet.ConnectionHandler.RISinstances+
						"\nSOS " +freenet.ConnectionHandler.SOSinstances;
			System.err.println(status);
			Core.logger.log(this, status,Logger.ERROR);
			try {
				Core.logger.log(this, "Ran emergency GC in "+getClass().getName(),
								Logger.ERROR);
			} catch (Throwable any) {};
		} catch(Throwable t){
			try {
				Core.logger.log(this, 
								"Caught throwable in AbstractSelectorLoop!",
								t, Logger.ERROR);
				t.printStackTrace();
			} catch (Throwable x) {};
		}
		}
	}
	
	
	/**
	 * moved down from the RSL
	 */
	
	// queueClose(ClosePair) overridden by subclasses, so always use it
	protected void queueClose(ClosePair chan) {
		if(logDebug)
			Core.logger.log(this, "queueing close: "+chan+":"+
							chan.attachment, new Exception("debug"), 
							Logger.DEBUG);
		if(chan.attachment != null)
			((NIOCallback)(chan.attachment)).queuedClose();
		if(isWindows)
			synchronized(preCloseQueue) {
				preCloseQueue.addLast(chan);
			}
		else
			synchronized(closeQueue) {
				boolean wasEmpty = closeQueue.isEmpty();
				closeQueue.addLast(chan);
				if(wasEmpty)
					closeQueue.notify();
			}
	}

	public void queueClose(Connection conn, SocketChannel sc) {
		Socket s = null;
		try {
			s = ((tcpConnection)conn).getSocket();
		}catch (IOException e) { return;}
		if (!closeUniqueness.containsKey(conn))
			queueClose(new ClosePair(conn, null, sc));
	}
	
	public void queueClose(Connection conn, NIOCallback cb, SocketChannel sc) {
		Socket s = null;
		try {
			s = ((tcpConnection)conn).getSocket();
		}catch (IOException e) { return;}
		if (!closeUniqueness.containsKey(conn))
			queueClose(new ClosePair(conn, cb, sc));
	}
	
	public void queueClose(SocketChannel chan) {
		if(!chan.isConnected()) return;
		if(!chan.isOpen()) return;
		Connection c = tcpConnection.getConnectionForSocket(chan.socket());
		if(c == null) {
			if(!chan.isConnected()) return;
			if(!chan.isOpen()) return;
			throw new IllegalArgumentException("Fed socket not connected to a tcpConnection!: "+chan);
		}
		if (closeUniqueness.containsKey(c)) return;
		queueClose(new ClosePair(c, null, chan));
	}
	
	public void queueClose(SocketChannel chan, NIOCallback nc) {
		if(!chan.isConnected()) return;
		if(!chan.isOpen()) return;
		Connection c = tcpConnection.getConnectionForSocket(chan.socket());
		if(c == null) {
			if(!chan.isConnected()) return;
			if(!chan.isOpen()) return;
			throw new IllegalArgumentException("Fed socket not connected to a tcpConnection!: "+chan+","+nc);
		}
		if (closeUniqueness.containsKey(c)) return;
		queueClose(new ClosePair(c,nc,chan));
	}
	
	public void queueClose(ChannelAttachmentPair pair) {
		NIOCallback cb = null;
		if(pair.attachment instanceof NIOCallback)
			cb = (NIOCallback)(pair.attachment);
		queueClose((SocketChannel)(pair.channel),cb);
	}
	
	volatile boolean terminateCloseThread = false;
	
	protected class CloseThread extends Thread {
		CloseThread() {
			super("AbstractSelectorLoop background close() thread");
		}
		
		public void run() {
			ClosePair current = null;
			while(true) {
				try {
					synchronized(closeQueue) {
						while(closeQueue.isEmpty()) {
							try {
								closeQueue.wait(200);
							} catch (InterruptedException e) {};
							if(terminateCloseThread) return;
						}
						current = (ClosePair)(closeQueue.removeFirst());
					}
					if(current != null) {
						Connection c = current.conn;
						try {
							try {
								c.close(true);
							} catch (Throwable t) {
								Core.logger.log(this, "Caught "+t+" closing "+
												c, t, Logger.ERROR);
							}
							//notify the callback
							if (current.attachment !=null){
								if(current.attachment instanceof NIOCallback) {
									NIOCallback nc = (NIOCallback)current.attachment;
									nc.closed();
								}
								current.attachment = null;
							}
						} finally {
							closeUniqueness.remove(c);
						}
					}
					current = null;
				} catch (OutOfMemoryError e) {
					System.gc();
					System.runFinalization();
					System.gc();
					System.runFinalization();
					try {
						Core.logger.log(this, "Ran emergency GC in AbstractSelectorLoop closeThread", Logger.ERROR);
					} catch (Throwable any) {};
				} catch (Throwable t) {
					try {
						String s = "Caught Throwable "+t+
							" in background close thread!";
						Core.logger.log(this, s, t, Logger.ERROR);
						// FIXME
						// System.err, System.out may be files on disk
						// PrintWriter on a disk file behaviour is to block
						// until more disk space is available (tested, Sun 1.4.1)
						// so don't use them!
					} catch (Throwable anything) {};
				}
			}
		}
	}
	
	protected void closeCloseThread() {
	terminateCloseThread = true;
		synchronized(closeQueue) {
			closeQueue.notify();
		}
	}
 }
