package freenet.support;

import freenet.*;
import freenet.support.*;

import java.io.*;
import java.net.*;
import java.util.*;

/*
 *  This code is part of fproxy, an HTTP proxy server for Freenet.
 *  It is distributed under the GNU Public Licence (GPL) version 2.  See
 *  http://www.gnu.org/ for further details of the GPL.
 */
/**
 *  Temporary Bucket Factory
 *
 * @author     giannij
 * @created    August 28, 2002
 */

public class TempBucketFactory implements BucketFactory {
  private String tmpDir = null;

  private static class NOPHook implements TempBucketHook {
      public void enlargeFile(long curLength, long finalLength) throws IOException {}
      public void shrinkFile(long curLength, long finalLength) {}
      public void deleteFile(long curLength) {}
      public void createFile(long curLength) throws IOException {}
  }

  private final static TempBucketHook DONT_HOOK = new NOPHook();
  private static TempBucketHook hook = DONT_HOOK;
  
  public static long defaultIncrement = 4096;

  // Storage accounting disabled by default.   
  public TempBucketFactory(String temp) {
    tmpDir = temp;
    if(tmpDir == null) tmpDir = System.getProperty("java.io.tmpdir");
//     Core.logger.log(this, "Creating TempBucketFactory, tmpDir = "+
// 		    (tmpDir == null ? "(null)" : tmpDir),
// 		    new Exception("debug"), Core.logger.DEBUG);
  }

  public TempBucketFactory() {
    this(System.getProperty("java.io.tmpdir"));
    if(Core.logger.shouldLog(Core.logger.DEBUG))
       Core.logger.log(this, "Creating TempBucketFactory, tmpDir = "+
		      (tmpDir == null ? "(null)" : tmpDir),
		      new Exception("debug"), Core.logger.DEBUG);
  }

  public Bucket makeBucket(long size) throws IOException {
      return makeBucket(size, 1.25F, defaultIncrement);
  }

  public Bucket makeBucket(long size, float factor) throws IOException {
      return makeBucket(size, factor, defaultIncrement);
  }

  /**
   *  Create a temp bucket
   *
   * @param  size             Default size
   * @param  factor           Factor to increase size by when need more space
   * @return                  A temporary Bucket
   * @exception  IOException  If it is not possible to create a temp bucket due to an I/O error
   */
  public Bucket makeBucket(long size, float factor, long increment) 
      throws IOException {
    File f = null;
    do {
      if (tmpDir != null) {
        f = new File(tmpDir, "tbf_" +
            Long.toHexString(Math.abs(Core.randSource.nextInt())));
 	Core.logger.log(this, "Temp file in "+tmpDir,
 			Core.logger.DEBUG);
      } else {
        f = new File("tbf_" +
            Long.toHexString(Math.abs(Core.randSource.nextInt())));
 	Core.logger.log(this, "Temp file in pwd",
 			Core.logger.DEBUG);
      }
    } while (f.exists());
    
    //System.err.println("TEMP BUCKET CREATED: " + f.getAbsolutePath());
    //(new Exception("creating TempBucket")).printStackTrace();

    if(Core.logger.shouldLog(Core.logger.DEBUG))
 	Core.logger.log(this, "Temp bucket created: " + f.getAbsolutePath()+
			" with hook "+hook+" initial length "+size,
 			new Exception("creating TempBucket"), 
 			Core.logger.DEBUG);
    
    return new TempFileBucket(f, hook, size, increment, factor);
  }


  /**
   *  Free bucket
   *
   * @param  b                Description of the Parameter
   * @exception  IOException  Description of the Exception
   */
  public void freeBucket(Bucket b) throws IOException {
    if (b instanceof TempFileBucket) {
	if(Core.logger.shouldLog(Core.logger.DEBUG))
	    Core.logger.log(this, "Temp bucket released: " + 
			    ((TempFileBucket)b).getFile().getAbsolutePath(), 
			    new Exception("debug"), Logger.DEBUG);
	if(!((TempFileBucket) b).release()) {
	    System.err.println("Could not release temp bucket"+b);
	    Core.logger.log(this, "Could not release temp bucket "+b,
			    Core.logger.ERROR);
	}
    }
  }
  
  /**
   * Sets the storage accounting hook.
   * @param t  The hook object to use to keep track of the amount of storage used.
   *           It is legal for t to be null.  In this case storage accounting is
   *           disabled.
   **/
  public static void setHook(TempBucketHook t) {
      Core.logger.log(TempBucketFactory.class, "Set TempBucketHook to "+t,
		      Core.logger.DEBUG);
      hook = t;
      if (hook == null) {
          // Allow hooks to be disabled w/o sprinkling
          // if (hook != null) {/*blah blah */} calls
          // throughout the code.
          hook = DONT_HOOK;
          if(Core.logger.shouldLog(Core.logger.DEBUG)) {
              Core.logger.log(TempBucketHook.class, "TempBucketHook file usage management was disabled.", 
                              Core.logger.DEBUG);
          }
      }
  }
}


