/*
THIS CODE ADAPTED FROM GNU Classpath 0.02 -- java.util.TreeMap
--------------------------------------------------------------

   TreeMap.java -- a class providing a basic Red-Black Tree data structure,
   mapping Object --> Object
   Copyright (C) 1998, 1999, 2000 Free Software Foundation, Inc.

This file is part of GNU Classpath.

GNU Classpath is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU Classpath is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA.

As a special exception, if you link this library with other files to
produce an executable, this library does not by itself cause the
resulting executable to be covered by the GNU General Public License.
This exception does not however invalidate any other reasons why the
executable file might be covered by the GNU General Public License. */


package freenet.support;

import freenet.support.BinaryTree.Node;

/**
 * Originally adapted from GNU Classpath 0.02 -- java.util.TreeMap
 *
 * There is virtually no trace left of the original code,
 * but I did use it as a reference.
 *
 * @author tavin
 */
public class RedBlackTree extends AbstractBinaryTree {
    
    public static final int RED   = 0,
                            BLACK = 1;


    public static interface RBNode extends Node {
        int getColor();
        void setColor(int color);
    }

    public static abstract class AbstractRBNode extends AbstractNode
                                                implements RBNode {
        int color = BLACK;
        public final int getColor() {
            return color;
        }
        public final void setColor(int color) {
            this.color = color;
        }
        public String toString() {
            return super.toString() + " color: "
                   + (color == BLACK ? "BLACK" : "RED");
        }
    }

    public static class RBNodeImpl extends AbstractRBNode {
        
        private final Comparable obj;
        
        public RBNodeImpl(Comparable obj) {
            this.obj = obj;
        }
        
        public final Comparable getObject() {
            return obj;
        }
    }

    

    protected RBNode root = null;


    
    protected final Node treeRoot() {
        return root;
    }
  

    public final Node treeInsert(Node node, boolean replace) {
        return treeInsert((RBNode) node, replace);
    }

    public Node treeInsert(RBNode node, boolean replace) {
        if (root == null) {
            root = node;
            root.setColor(BLACK);
            return null;
        }
        else {
            RBNode collision = (RBNode) treeInsert(root, node);
            if (collision != null) {
                if (replace)
                    treeReplace(collision, node);
                return collision;
            }
        }

        node.setColor(RED);

        do {
            RBNode parent = (RBNode) node.getParent();
            if (parent.getColor() == BLACK)
                break;

            RBNode grandParent = (RBNode) parent.getParent();

            if (compare(parent, grandParent) < 0) {
                RBNode uncle = (RBNode) grandParent.getRightChild();
                if (uncle != null && uncle.getColor() == RED) {
                    parent.setColor(BLACK);
                    uncle.setColor(BLACK);
                    grandParent.setColor(RED);
                    node = grandParent;
                }
                else {
                    if (compare(node, parent) > 0) {
                        node = parent;
                        leftRotate(node);
                        parent = (RBNode) node.getParent();
                        grandParent = (RBNode) parent.getParent();
                    }
                    parent.setColor(BLACK);
                    grandParent.setColor(RED);
                    rightRotate(grandParent);
                }
            }
            else {
                RBNode uncle = (RBNode) grandParent.getLeftChild();
                if (uncle != null && uncle.getColor() == RED) {
                    parent.setColor(BLACK);
                    uncle.setColor(BLACK);
                    grandParent.setColor(RED);
                    node = grandParent;
                }
                else {
                    if (compare(node, parent) < 0) {
                        node = parent;
                        rightRotate(node);
                        parent = (RBNode) node.getParent();
                        grandParent = (RBNode) parent.getParent();
                    }
                    parent.setColor(BLACK);
                    grandParent.setColor(RED);
                    leftRotate(grandParent);
                }
            }
        }
        while (node != root);

        root.setColor(BLACK);
        
        return null;
    }
  
    protected final void treeReplace(RBNode rem, RBNode splice) {
        
        Node parent = rem.getParent();
        Node leftChild = rem.getLeftChild();
        Node rightChild = rem.getRightChild();
        
        rem.setParent(null);
        rem.setLeftChild(null);
        rem.setRightChild(null);
        
        splice.setColor(rem.getColor());
        splice.setParent(parent);
        splice.setLeftChild(leftChild);
        splice.setRightChild(rightChild);
        
        if (leftChild != null)
            leftChild.setParent(splice);
        if (rightChild != null)
            rightChild.setParent(splice);

        if (parent == null)
            root = splice;
        else if (compare(splice, parent) < 0)
            parent.setLeftChild(splice);
        else
            parent.setRightChild(splice);
    }

    public final boolean treeRemove(Node node) {
        return treeRemove((RBNode) node);
    }

    public boolean treeRemove(RBNode node) {

        if (node.getParent() == null && node != root)
            return false;

        RBNode parent;
        RBNode child;
        
        if (node.hasLeftChild() && node.hasRightChild()) {
            RBNode splice = (RBNode) node.getRightChild();
            if (splice.hasLeftChild()) {
                do {
                    parent = splice;
                    splice = (RBNode) splice.getLeftChild();
                }
                while (splice.hasLeftChild());
                
                child = (RBNode) splice.getRightChild();
                parent.setLeftChild(child);

                if (child != null) {
                    treeReplace(node, splice);
                    child.setParent(parent);
                    child.setColor(BLACK);
                }
                else if (splice.getColor() == BLACK) {
                    treeReplace(node, splice);
                    deleteFixup(parent);
                }
                else {
                    treeReplace(node, splice);
                }
            }
            else {
                parent = (RBNode) node.getParent();
                child = (RBNode) node.getLeftChild();
                
                node.setParent(null);
                node.setLeftChild(null);
                node.setRightChild(null);

                splice.setLeftChild(child);
                splice.setParent(parent);

                child.setParent(splice);

                if (parent == null)
                    root = splice;
                else if (compare(splice, parent) < 0)
                    parent.setLeftChild(splice);
                else
                    parent.setRightChild(splice);
                
                child = (RBNode) splice.getRightChild();
                if (child != null) {
                    splice.setColor(node.getColor());
                    child.setColor(BLACK);
                }
                else if (splice.getColor() == BLACK) {
                    splice.setColor(node.getColor());
                    deleteFixup(splice);
                }
                else {
                    splice.setColor(node.getColor());
                }
            }
        }
        else {
            parent = (RBNode) node.getParent();
            node.setParent(null);
            
            if (node.hasLeftChild()) {
                child = (RBNode) node.getLeftChild();
                node.setLeftChild(null);
            }
            else if (node.hasRightChild()) {
                child = (RBNode) node.getRightChild();
                node.setRightChild(null);
            }
            else child = null;
            
            if (parent == null)
                root = child;
            else if (compare(node, parent) < 0)
                parent.setLeftChild(child);
            else
                parent.setRightChild(child);
            
            if (child != null) {
                child.setParent(parent);
                child.setColor(BLACK);
            }
            else if (node.getColor() == BLACK) {
                deleteFixup(parent);
            }
        }
        return true;
    }

    protected final void deleteFixup(RBNode parent) {
        RBNode node = null;
        //int iter = 0;
        while (parent != null) {
            
            //++iter;
            //System.out.println("Iteration #"+iter);
            //System.out.println("Node: "+node);
            //System.out.println("Parent: "+parent);
            //System.out.println();
            
            // Invariant of this loop: oNode is a black node (or NIL)
            // oParent is its parent and the tree on the oNode side is
            // one black node too small.
            if (node == null && !parent.hasLeftChild()
                || node != null && compare(node, parent) < 0) {
                
                RBNode sibling = (RBNode) parent.getRightChild();
                if (sibling.getColor() == RED) {
                    sibling.setColor(BLACK);
                    parent.setColor(RED);
                    leftRotate(parent);
                    sibling = (RBNode) parent.getRightChild();
                }
                RBNode siblingLeft = (RBNode) sibling.getLeftChild();
                RBNode siblingRight = (RBNode) sibling.getRightChild();
                if ((siblingLeft == null || siblingLeft.getColor() == BLACK)
                    && (siblingRight == null || siblingRight.getColor() == BLACK)) {
                    sibling.setColor(RED);
                    node = parent;
                    parent = (RBNode) parent.getParent();
                }
                else {
                    if (siblingLeft != null && siblingLeft.getColor() == RED) {
                        siblingLeft.setColor(BLACK);
                        sibling.setColor(RED);
                        rightRotate(sibling);
                        sibling = (RBNode) parent.getRightChild();
                        siblingRight = (RBNode) sibling.getRightChild();
                    }
                    sibling.setColor(parent.getColor());
                    parent.setColor(BLACK);
                    if (siblingRight != null)
                        siblingRight.setColor(BLACK);
                    leftRotate(parent);
                    // Fixup completed
                    return;
                }
            }
            else {
                RBNode sibling = (RBNode) parent.getLeftChild();
                if (sibling.getColor() == RED) {
                    sibling.setColor(BLACK);
                    parent.setColor(RED);
                    rightRotate(parent);
                    sibling = (RBNode) parent.getLeftChild();
                }
                RBNode siblingLeft = (RBNode) sibling.getLeftChild();
                RBNode siblingRight = (RBNode) sibling.getRightChild();
                if ((siblingRight == null || siblingRight.getColor() == BLACK)
                    && (siblingLeft == null || siblingLeft.getColor() == BLACK)) {
                    sibling.setColor(RED);
                    node = parent;
                    parent = (RBNode) parent.getParent();
                }
                else {
                    if (siblingRight != null && siblingRight.getColor() == RED) {
                        siblingRight.setColor(BLACK);
                        sibling.setColor(RED);
                        leftRotate(sibling);
                        sibling = (RBNode) parent.getLeftChild();
                        siblingLeft = (RBNode) sibling.getLeftChild();
                    }
                    sibling.setColor(parent.getColor());
                    parent.setColor(BLACK);
                    if (siblingLeft != null)
                        siblingLeft.setColor(BLACK);
                    rightRotate(parent);
                    // Fixup completed
                    return;
                }
            }
            if (node.getColor() == RED) {
                node.setColor(BLACK);
                return;
            }
        }
    }



    protected void leftRotate(Node node) {

        Node child = node.getRightChild();

        Node childLeft = child.getLeftChild();
        
        node.setRightChild(childLeft);
        
        if (childLeft != null)
            childLeft.setParent(node);
  
        Node parent = node.getParent();
        child.setParent(parent);
        if (parent == null)
            root = (RBNode) child;
        else if (compare(node, parent) < 0)
            parent.setLeftChild(child);
        else
            parent.setRightChild(child);
  
        child.setLeftChild(node);
        node.setParent(child);
    }
  
    protected void rightRotate(Node node) {

        Node child = node.getLeftChild();
        
        Node childRight = child.getRightChild();
        
        node.setLeftChild(childRight);
  
        if (childRight != null)
            childRight.setParent(node);
  
        Node parent = node.getParent();
        child.setParent(parent);
        if (parent == null)
            root = (RBNode) child;
        else if (compare(node, parent) > 0)
            parent.setRightChild(child);
        else
            parent.setLeftChild(child);
  
        child.setRightChild(node);
        node.setParent(child);
    }


    


    //public static void main(String[] args) {
    //  do some timing tests..
    //}

}


