package freenet.support;
import freenet.*;
import java.io.*;
import java.util.StringTokenizer;
import java.util.Vector;
import java.util.Date;
import java.util.LinkedList;
import java.text.DateFormat;
import java.text.SimpleDateFormat;

/**
 * Converted the old StandardLogger to Ian's loggerhook interface.
 *
 * @author oskar
 */
public class FileLoggerHook extends LoggerHook {

    /** Verbosity types **/
    public static final int DATE       = 1,
                            CLASS      = 2,
                            HASHCODE   = 3,
                            THREAD     = 4,
                            PRIORITY   = 5,
                            MESSAGE    = 6,
                            UNAME      = 7;
    
    private volatile boolean closed = false;
    
    private static String uname;
    static {
        try {
	    java.net.InetAddress addr = null;
	    try {
		java.net.InetAddress.getLocalHost();
	    } catch (NullPointerException e) {
		// Blech, but Sun throws an NPE if it can't lookup localhost or 
		// some such crap
		addr = null;
	    }
	    if(addr == null) uname = "unknown";
	    else {
		uname = new StringTokenizer(addr.getHostName(),".").nextToken();
	    }
	} catch (java.net.UnknownHostException e) {
            uname = "unknown";
        }
    }
    
    private DateFormat df;
    private int[] fmt;
    private String[] str;
    /**
     * The printstream the Logger writes too.
     **/
    public PrintStream lout;
    
    protected OutputStream uout;
    
    /**
     * Something wierd happens when the disk gets full, also we don't want to block
     * So run the actual write on another thread
     */
    protected LinkedList list = new LinkedList();
    protected long listBytes = 0;
    
    protected static int MAX_LIST_SIZE=100000;
    protected static long MAX_LIST_BYTES=10*(1<<20);
    
    // FIXME: should reimplement LinkedList with minimal locking
    
    public static void setMaxListLength(int len) {
	MAX_LIST_SIZE = len;
    }
    
    public static void setMaxListBytes(long len) {
	MAX_LIST_BYTES = len;
    }
    
    class WriterThread extends Thread {
	WriterThread() {
	    super("Log File Writer Thread");
	}
	
	public void run() {
	    Object o = null;
	    while(true) {
		try {
		    if(uout != null) {
			myWrite(uout, null);
		    } else lout.flush();
		    synchronized(list) {
			while(list.size() == 0) {
			    if(closed) return;
			    try {
				list.wait(500);
			    } catch (InterruptedException e) {};
			}
			o = list.removeFirst();
			listBytes -= (((byte[])o).length+16);
		    }
		    if(uout != null) {
			myWrite(uout, ((byte[])o));
		    } else {
			lout.print(new String((byte[])o));
		    }
		} catch (OutOfMemoryError e) {
			String status = "dump of interesting objects before gc in FileLoggerHook:"+
						"\ntcpConnections " +freenet.transport.tcpConnection.instances+
						"\nFnpLinks " +freenet.session.FnpLink.instances+
						"\nNIOOS " + freenet.support.io.NIOOutputStream.instances+
						"\nNIOIS " + freenet.support.io.NIOInputStream.instances+
						"\nCH " + freenet.ConnectionHandler.instances+
						"\nCHIS " +freenet.ConnectionHandler.CHISinstances+
						"\nCHOS " +freenet.ConnectionHandler.CHOSinstances+
						"\nRIS " +freenet.ConnectionHandler.RISinstances+
						"\nSOS " +freenet.ConnectionHandler.SOSinstances;
			System.err.println(status);
			Core.logger.log(this, status, Logger.ERROR);
		    System.gc();
		    System.runFinalization();
		    System.gc();
		    System.runFinalization();
		     status = "dump of interesting objects after gc in FileLoggerHook:"+
						"\ntcpConnections " +freenet.transport.tcpConnection.instances+
						"\nFnpLinks " +freenet.session.FnpLink.instances+
						"\nNIOOS " + freenet.support.io.NIOOutputStream.instances+
						"\nNIOIS " + freenet.support.io.NIOInputStream.instances+
						"\nCH " + freenet.ConnectionHandler.instances+
						"\nCHIS " +freenet.ConnectionHandler.CHISinstances+
						"\nCHOS " +freenet.ConnectionHandler.CHOSinstances+
						"\nRIS " +freenet.ConnectionHandler.RISinstances+
						"\nSOS " +freenet.ConnectionHandler.SOSinstances;
			System.err.println(status);
			Core.logger.log(this, status, Logger.ERROR);
		} catch (Throwable t) {
		    System.err.println("FileLoggerHook log writer caught "+
				       t);
		    t.printStackTrace(System.err);
		}
	    }
	}
	
	/**
	 * @param b the bytes to write, null to flush
	 */
	protected void myWrite(OutputStream os, byte[] b) {
	    long sleepTime = 1000;
	    while(true) {
		boolean thrown = false;
		try {
		    if(b != null) uout.write(b);
		    else uout.flush();
		} catch (IOException e) {
		    System.err.println("Exception writing to log: "+e);
		    thrown = true;
		}
		if(thrown) {
		    try {
			Thread.sleep(sleepTime);
		    } catch (InterruptedException e) {};
		    sleepTime += sleepTime;
		} else
		    return;
	    }
	}
	
    }
    
    /**
     * Create a Logger to append to the given file. If the file does not
     * exist it will be created.
     *
     * @param filename  the name of the file to log to.
     * @param verb      The verbosity setting.
     * @param fmt       log message format string
     * @param dfmt      date format string
     * @param threshhold  Lowest logged priority
     * @param assumeWorking If false, check whether stderr and stdout are writable and if not, redirect them to the log file
     * @exception IOException if the file couldn't be opened for append.
     */
    public FileLoggerHook(String filename, String fmt, 
                          String dfmt, int threshold, boolean assumeWorking)
	throws IOException {
        this(new FileOutputStream(filename, true), fmt, dfmt, threshold);
	
        // Redirect System.err and System.out to the Logger Printstream
        // if they don't exist (like when running under javaw)
	if(!assumeWorking) {
	    System.out.print(" \b");
	    if(System.out.checkError()) System.setOut(lout);
	    System.err.print(" \b"); 
	    if(System.err.checkError()) System.setErr(lout);
	}
    }
    
    public FileLoggerHook(OutputStream os, String fmt, String dfmt,
			  int threshold) {
	this(new PrintStream(os), fmt, dfmt, threshold);
	uout = os;
    }
    
    /**
     * Create a Logger to send log output to the given PrintStream.
     *
     * @param stream  the PrintStream to send log output to.
     * @param fmt     log message format string
     * @param dfmt    date format string
     * @param threshhold  Lowest logged priority
     */
    public FileLoggerHook(PrintStream stream, String fmt, String dfmt,
                          int threshold) {
        lout = stream;
        this.threshold = threshold;
        if (dfmt != null && !dfmt.equals("")) {
            try {
                df = new SimpleDateFormat(dfmt);
            }
            catch (RuntimeException e) {
                df = DateFormat.getDateTimeInstance();
            }
        }
        else df = DateFormat.getDateTimeInstance();
        
        if (fmt == null || fmt.equals("")) fmt = "d:c:h:t:p:m";
        char[] f = fmt.toCharArray();
        
        Vector fmtVec = new Vector(),
               strVec = new Vector();
               
        StringBuffer sb = new StringBuffer();
        
        boolean comment = false;
        for (int i=0; i<f.length; ++i) {
            if (!comment && numberOf(f[i]) != 0) {
                if (sb.length() > 0) {
                    strVec.addElement(sb.toString());
                    fmtVec.addElement(new Integer(0));
                    sb.setLength(0);
                }
                fmtVec.addElement(new Integer(numberOf(f[i])));
            } else if (f[i] == '\\') {
                comment = true;
            } else {
                comment = false;
                sb.append(f[i]);
            }
        }
        if (sb.length() > 0) {
            strVec.addElement(sb.toString());
            fmtVec.addElement(new Integer(0));
            sb.setLength(0);
        }
        
        this.fmt = new int[fmtVec.size()];
        int size = fmtVec.size();
        for (int i=0; i < size; ++i)
            this.fmt[i] = ((Integer) fmtVec.elementAt(i)).intValue();

        this.str = new String[strVec.size()];
	str = (String[])strVec.toArray(str);
	WriterThread wt = new WriterThread();
	//wt.setDaemon(true);
	CloserThread ct = new CloserThread();
	Runtime.getRuntime().addShutdownHook(ct);
	wt.start();
    }
    
    public void log(Object o, Class c, String msg, Throwable e, int priority){
	if (!acceptPriority(priority)) return;
	
	if(closed) return;

        StringBuffer sb = new StringBuffer(512);
        int sctr = 0;
	
        for (int i=0; i<fmt.length; ++i) {
            switch (fmt[i]) {
            case 0:         sb.append(str[sctr++]); break;
            case DATE:      sb.append(df.format(new Date())); break;
            case CLASS:     sb.append(c == null ? "<none>" : c.getName());
                break;
            case HASHCODE:  sb.append(o == null ? "<none>"
                                      : Integer.toHexString(o.hashCode()));
            break;
            case THREAD:    sb.append(Thread.currentThread().getName()); break;
            case PRIORITY:  sb.append(Logger.priorityOf(priority)); break;
            case MESSAGE:   sb.append(msg); break;
            case UNAME:         sb.append(uname); break;
            }
        }
	sb.append('\n');
	synchronized(list) { // want the exception consecutive to the message
	    logString(sb.toString().getBytes());
	    if(e != null) {
		ByteArrayOutputStream bos = new ByteArrayOutputStream();
		PrintWriter bpw = new PrintWriter(bos);
		e.printStackTrace(bpw);
		bpw.flush();
		byte[] b = bos.toByteArray();
		logString(b);
	    }
	}
    }
    
    public void logString(byte[] b) {
	synchronized(list) {
	    int sz = list.size();
	    list.add(b);
	    listBytes += (b.length+16); /* total guess */
	    int x = 0;
	    if(list.size() > MAX_LIST_SIZE || listBytes > MAX_LIST_BYTES) {
		while(list.size() > (MAX_LIST_SIZE * 0.9F) || 
		      listBytes > (MAX_LIST_BYTES * 0.9F)) {
		    byte[] ss = (byte[])(list.removeFirst());
		    listBytes -= (ss.length+16);
		    x++;
		}
		String err = "GRRR: ERROR: Logging too fast, chopped "+x+
		    " lines, "+listBytes+" bytes in memory\n";
		list.add(0, err.getBytes());
	    }
	    if(sz == 0) list.notify();
	}
    }
    
    public long listBytes() {
	return listBytes;
    }

    public static int numberOf(char c) {
        switch (c) {
        case 'd':   return DATE;
        case 'c':   return CLASS;
        case 'h':   return HASHCODE;
        case 't':   return THREAD;
        case 'p':   return PRIORITY;
        case 'm':   return MESSAGE;
        case 'u':   return UNAME;    
        default:    return 0;
        }
    }

    public long minFlags()
    {
	return 0;
    }

    public long notFlags()
    {
	return INTERNAL;
    }

    public long anyFlags()
    {
	return ((2*ERROR)-1) & ~(threshold-1);
    }

    public PrintStream getStream()
    {
	return lout;
    }
    
    public void close() {
	closed = true;
    }
    
    class CloserThread extends Thread {
	public void run() {
	    closed = true;
	}
    }
}
