package freenet.support;

import java.io.*;
import java.util.Vector;


/**
 * Helper functions for working with Buckets. 
 *
 */
public class BucketTools {

    public final static void copy(Bucket src, Bucket dst) throws IOException {
        OutputStream out = dst.getOutputStream();
        byte[] buffer = new byte[16384];
        InputStream in = src.getInputStream();
        
        int nRead = in.read(buffer);
        while (nRead > 0) {
            out.write(buffer, 0, nRead);
            nRead = in.read(buffer);
        }
        
        in.close();
        out.close();
    }

    public final static void dumpBucket(Bucket b, String fileName) throws IOException {
        OutputStream out = new FileOutputStream(fileName);
        byte[] buffer = new byte[16384];
        InputStream in = b.getInputStream();
        
        int nRead = in.read(buffer);
        while (nRead > 0) {
            out.write(buffer, 0, nRead);
            nRead = in.read(buffer);
        }
        
        in.close();
        out.close();
    }

    public final static Bucket readBucket(String fileName, BucketFactory bf, long size) 
        throws IOException {

        InputStream in = null;
        OutputStream out = null;
        Bucket ret = null;
        byte[] buffer = new byte[16384];
        boolean groovy = false;
        try {
            ret = bf.makeBucket(size);
            out = ret.getOutputStream(); 
            in = new FileInputStream(fileName);

            int nRead = in.read(buffer);
            while (nRead > 0) {
                out.write(buffer, 0, nRead);
                nRead = in.read(buffer);
            }
            groovy = true;
        }
        finally {
            if (in != null) {
                try { in.close(); } catch (IOException ioe) {}
            }
            if (out != null) {
                try { out.close(); } catch (IOException ioe) {}
            }
            if (!groovy && ret != null) {
                try { bf.freeBucket(ret); } catch (IOException ioe) {}
            }
        }
        return ret;
    }

    public final static void zeroPad(Bucket b, long size)
        throws IOException {

        OutputStream out = b.getOutputStream();

        // Initialized to zero by default.
        byte[] buffer = new byte[16384];

        long count = 0;
        while (count < size ) {
            long nRequired = buffer.length;
            if (nRequired > size - count) {
                nRequired = size - count;
            }
            out.write(buffer, 0, (int)nRequired);
            count += nRequired;
        }
        
        out.close();
    }

    public final static void paddedCopy(Bucket from, Bucket to, long nBytes, int blockSize)
        throws IOException {

        if (nBytes > blockSize) {
            throw new IllegalArgumentException("nBytes > blockSize");
        }

        OutputStream out = to.getOutputStream();
        byte[] buffer = new byte[16384];
        InputStream in = from.getInputStream();
        
        long count = 0;
        for (;;) {
            long nRequired = nBytes - count;
            if (nRequired > buffer.length) {
                nRequired = buffer.length;
            }
            long nRead = in.read(buffer, 0, (int)nRequired);
            if (nRead == -1) {
                throw new IOException("Not enough data in source bucket.");
            }
            out.write(buffer, 0, (int)nRead);
            count += nRead;
            if (count == nBytes) {
                break;
            }
        }

        if (count < blockSize) {
            // hmmm... better to just allocate a new buffer
            // instead of explicitly zeroing the old one?
            // Zero pad to blockSize
            long padLength = buffer.length;
            if (padLength > blockSize - nBytes) {
                padLength = blockSize - nBytes;
            }
            for (int i = 0; i < padLength; i++) {
                buffer[i] = 0;
            }

            for (;;) {
                long nRequired = blockSize - count;
                if (blockSize - count > buffer.length) {
                    nRequired = buffer.length;
                }
                out.write(buffer, 0, (int)nRequired);
                count += nRequired;
                if (count == blockSize) {
                    break;
                }
            }
        }
        in.close();
        out.close();
    }

    public static class BucketFactoryWrapper implements BucketFactory {
        public BucketFactoryWrapper(BucketFactory bf) {
            BucketFactoryWrapper.this.bf = bf;
        }
        public Bucket makeBucket(long size) throws IOException {
            return bf.makeBucket(size);
        }

        public void freeBucket(Bucket b) throws IOException {
            if (b instanceof RandomAccessFileBucket) {
                ((RandomAccessFileBucket)b).release();
                return;
            }
            bf.freeBucket(b);
        }
        private BucketFactory bf = null;
    }

    public static Bucket[] makeBuckets(BucketFactory bf, int count, int size)
        throws IOException {
        Bucket[] ret = new Bucket[count];

        for (int i = 0; i < count; i++) {
            ret[i] = bf.makeBucket(size);
        }
        return ret;
    }

    public static void freeBuckets(BucketFactory bf, Bucket[] buckets) {
        if (buckets == null) {
            return;
        }

        for (int i = 0; i < buckets.length; i++) {
            // Make sure we free any temp buckets on exception
            try { 
                if (buckets[i] != null) {
                    bf.freeBucket(buckets[i]);
                } 
                buckets[i] = null;
            } catch (Exception e) {
                // REDFLAG: remove
                e.printStackTrace();
            }
        }
    }


    // Note: Not all buckets are allocated by the bf.
    //       You must use the BucketFactoryWrapper class above
    //       to free the returned buckets.
    //
    // Always returns blocks, blocks, even if it has to create
    // zero padded ones.
    public static Bucket[] splitFile(File file, int blockSize, 
                                     long offset, int blocks,  boolean readOnly,
                                     BucketFactoryWrapper bf) throws IOException {

        
        long len = file.length() - offset;
        if (len > blocks * blockSize) {
            len = blocks * blockSize;
        }

        long padBlocks = 0;
        if ((blocks * blockSize) - len >= blockSize) {
            padBlocks = ((blocks * blockSize) - len) / blockSize;
        }

        Bucket[] ret = new Bucket[blocks];
        Bucket[] rab = RandomAccessFileBucket.segment(file, blockSize, offset, 
                                                      (int)(blocks - padBlocks), true);
        System.arraycopy(rab, 0, ret, 0, rab.length);

        boolean groovy = false;
        try {
            if (len % blockSize != 0) {
                // Copy and zero pad final partial block
                Bucket partial = ret[rab.length - 1];
                ret[rab.length - 1] = bf.makeBucket(blockSize);
                paddedCopy(partial, ret[rab.length - 1], len % blockSize, blockSize);
            }

            // Trailing zero padded blocks
            for (int i = rab.length; i < ret.length; i++) {
                ret[i] = bf.makeBucket(blockSize);
                zeroPad(ret[i], blockSize);
            }
            groovy = true;
        }
        finally {
            if (!groovy) {
                freeBuckets(bf, ret);
            }
        }
        return ret;
    } 
    
    // Truncates last bucket if it exceeds length.
    public final static void concatinate(Bucket[] src, Bucket dest, long length, boolean append) 
        throws IOException {
        OutputStream out = null;
        try {
            if (!append) {
                dest.resetWrite();
            }
            out = dest.getOutputStream();

            byte[] buf = new byte[16384];
            int i=0;
            for (i = 0; i < src.length; i++) {
                InputStream in = null;
                try {
                    in = src[i].getInputStream();
                    int bucketLen = (int)src[i].size();
                    while ((bucketLen > 0) && (length > 0)) {
                        int nLimit = buf.length;
                        if (length < nLimit) {
                            nLimit = (int)length;
                        }
                        int nRead = in.read(buf, 0, nLimit);
                        if (nRead < 0) {
                            throw new IOException("read failed.");
                        }
                        out.write(buf, 0, nRead);
                        bucketLen -= nRead;
                        length -= nRead;
                    } 
                    in.close();
                    in = null;
                }
                catch (IOException ioe) {
                    if (in != null) {
                        try {in.close();} catch (IOException ioe1) {}
                    }
                    throw ioe;
                }
            }
        }
        finally {
            if (out != null) {
                try { out.close(); } catch (IOException ioe) {} 
            }
        }
    }


    public final static int[] nullIndices(Bucket[] array) {
        Vector list = new Vector();
        int i = 0;
        for (i = 0; i < array.length; i++) {
            if (array[i] == null) {
                list.addElement( new Integer(i) );
            }
        }

        int[] ret = new int[list.size()];
        for (i = 0; i < list.size(); i ++) {
            ret[i] = ((Integer)list.elementAt(i)).intValue();
        }
        return ret;
    }

    public final static int[] nonNullIndices(Bucket[] array) {
        Vector list = new Vector();
        int i = 0;
        for (i = 0; i < array.length; i++) {
            if (array[i] != null) {
                list.addElement( new Integer(i) );
            }
        }

        int[] ret = new int[list.size()];
        for (i = 0; i < list.size(); i ++) {
            ret[i] = ((Integer)list.elementAt(i)).intValue();
        }
        return ret;
    }


    public final static Bucket[] nonNullBuckets(Bucket[] array) {
        Vector list = new Vector();
        int i = 0;
        for (i = 0; i < array.length; i++) {
            if (array[i] != null) {
                list.addElement( array[i] );
            }
        }

        Bucket[] ret = new Bucket[list.size()];
        list.copyInto(ret);
        return ret;
    }
}







