package freenet.node.states.request;

import freenet.*;
import freenet.node.*;
import freenet.message.*;
import freenet.support.Logger;

/** After a request or insert has successfully completed transferring,
  * we will enter this state to wait for the StoreData (unless we were
  * the terminal node in the chain).
  * @author tavin
  */
public class AwaitingStoreData extends RequestState {

    private NoStoreData nosd;

    // he he .. and here we are again :)
    private TransferInsert xferIns;
    
    private boolean isInsert;
    
    /** New AwaitingStoreData with a definite timeout.
      * @param nosd  the MO that governs the timeout
      *              -- should be already scheduled on the ticker
      */
    AwaitingStoreData(Pending ancestor, NoStoreData nosd, boolean isInsert) {
        super(ancestor);
        this.nosd = nosd;
	this.isInsert = isInsert;
    }

    AwaitingStoreData(TransferInsert ancestor, NoStoreData nosd, boolean isInsert) {
        this((Pending) ancestor, nosd, isInsert);
        xferIns = ancestor;
	this.isInsert = isInsert;
    }

    public final String getName() {
        return "Awaiting StoreData";
    }
    
    // it could happen...
    public State receivedMessage(Node n, Accepted a) {
        return this;
    }

    // ignore.
    public State receivedMessage(Node n, QueryRestarted qr) {
        return this;
    }

    public State receivedMessage(Node n, QueryRejected qr) throws StateException {
        if (!fromLastPeer(qr)) {
            throw new BadStateException("QueryRejected from the wrong peer!");
        }
        // this is pretty much the same as if it timed out for a DataRequest,
        // but for an insert we need to go back to TransferInsertPending
        if (xferIns == null) {
            relayStoreData(n, null);
            return new RequestDone(this);
        }
        else {
            if (nosd != null) nosd.cancel();
            return xferIns.receivedMessage(n, qr);
        }
    }
    
    public State receivedMessage(Node n, NoStoreData nosd) throws BadStateException {
        if (this.nosd != nosd) {
            throw new BadStateException("Not my NoStoreData: "+nosd);
        }
        relayStoreData(n, null);
        return new RequestDone(this);
    }
    
    public State receivedMessage(Node n, StoreData sd) throws BadStateException {
        if (!fromLastPeer(sd)) {
            throw new BadStateException("StoreData from the wrong peer!");
        }
        // yay!
        routes.routeSucceeded(sd.source.isCached());
	
	// Update global network load estimate stats.			    
        n.loadStats.storeTraffic(sd.dataSource(), sd.requestsPerHour());
        relayStoreData(n, sd);

        return new RequestDone(this);
    }

    public final void lost(Node n) {
        Core.diagnostics.occurrenceCounting("lostAwaitingStoreData", 1);
        // just like timing out with the NoStoreData
        relayStoreData(n, null);
    }

    private void relayStoreData(Node n, StoreData sd) {
        if (nosd != null) {
            nosd.cancel();
        }
        NodeReference nr = (sd == null ? null : sd.dataSource());
        if (nr != null && !nr.checkAddresses(n.transports)) {
            if (n.logger.shouldLog(n.logger.MINOR))
                    n.logger.log(this,
                                 "Not referencing because addresses "
                                 + "of DataSource wrong: " 
                                 + nr.toString(), n.logger.MINOR);
            nr = null;
        }
        long rate = -1;
	if(sd != null)
	    n.diagnostics.occurrenceContinuous("incomingHopsSinceReset", 
					       sd.hopsSinceReset());
        if (n.shouldReference(nr, sd)) {
	    n.diagnostics.occurrenceCounting("prefAccepted", 1);
	    n.logger.log(this, "PCaching: Referencing "+searchKey, Logger.DEBUG);
            n.rt.reference(searchKey, nr);
            rate = sd.requestsPerHour();
        } else {
	    n.diagnostics.occurrenceCounting("prefRejected", 1);
	    n.logger.log(this, "PCaching: Not Referencing "+searchKey,
			 Logger.DEBUG);
	}
	if(n.shouldCache(sd)) {
	    n.logger.log(this, "PCaching: Keeping "+searchKey, Logger.DEBUG);
	    n.diagnostics.occurrenceCounting("pcacheAccepted", 1);
	    // Keep it
	} else {
	    // Try to delete it
	    n.logger.log(this, "PCaching: Removing "+searchKey, Logger.DEBUG);
	    n.diagnostics.occurrenceCounting("pcacheRejected", 1);
	    if(!n.ds.demote(searchKey))
		n.diagnostics.occurrenceCounting("pcacheFailedDelete", 1);
	}
        try {
	    n.diagnostics.occurrenceCounting("storeDataAwaitingStoreData",1);
            ft.storeData(n, nr, rate, sd == null ? 0 : sd.hopsSinceReset());
        }
        catch (CommunicationException e) {
            n.logger.log(this, "Failed to relay StoreData to peer "+e.peer,
                         e, Logger.MINOR);
        }
	if (origPeer != null) {
	    if (isInsert) {
	        if (n.successInsertDistribution != null) {
                    n.successInsertDistribution.add(searchKey.getVal());
	        }
	    } else {
	        if (n.successDataDistribution != null) {
                    n.successDataDistribution.add(searchKey.getVal());
		}
	    }
	    if (n.inboundRequests != null)
		    n.inboundRequests.incActive(origPeer.getAddress().toString());
	}
    }
}


