package freenet.node.rt;

import freenet.Key;
import freenet.Identity;
import freenet.Core;
import freenet.node.NodeReference;
import freenet.support.*;
import freenet.support.Comparable;
import freenet.support.BinaryTree.Node;
//import freenet.support.RedBlackTree.RBNodeImpl;
import freenet.support.Skiplist.SkipNodeImpl;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Random;

/**
 * The basic core of a routing table that routes by walking
 * through closest keys.
 * @author tavin
 */
public abstract class TreeRoutingTable implements RoutingTable {

    protected final RoutingStore routingStore;
    protected final int maxNodes,
                        maxRefsPerNode;

    protected long initialRefs;
    protected int curRefs;

    protected final BinaryTree refTree = new Skiplist(32, new Random()); //new RedBlackTree();


    public TreeRoutingTable(RoutingStore routingStore,
                            int maxNodes, int maxRefsPerNode) throws IOException {
        
        this.routingStore = routingStore;
        this.maxNodes = maxNodes;
        this.maxRefsPerNode = maxRefsPerNode;
	this.initialRefs = 0;
	this.curRefs = 0;
        
        // build binary tree of references
        Enumeration rme = routingStore.elements();
        while (rme.hasMoreElements()) {
            RoutingMemory mem = (RoutingMemory) rme.nextElement();
            ReferenceSet refSet = ReferenceSet.getProperty(mem, "keys");
            Enumeration refs = refSet.references();
            while (refs.hasMoreElements()) {
                Reference ref = (Reference) refs.nextElement();
                refTree.treeInsert(new SkipNodeImpl(ref), true);
		initialRefs++;
		curRefs++;
            }
        }
    }

    public long initialRefsCount() {
	return initialRefs;
    }
    
    public int getKeyCount() {
	return curRefs;
    }
    
    public synchronized void reference(Key k, NodeReference nr) {

        long now = System.currentTimeMillis();
        
        RoutingMemory mem = routingStore.getNode(nr.getIdentity());
        if (mem == null || nr.supersedes(mem.getNodeReference()))
            mem = routingStore.putNode(nr);

	if(k != null && mem != null) {

	    ReferenceSet refSet = ReferenceSet.getProperty(mem, "keys");
	    
	    Reference ref = new Reference(k, nr.getIdentity(), now);
	    
	    Node oldRef = refTree.treeInsert(new SkipNodeImpl(ref), false);
	    if (oldRef != null) {
		ref = (Reference) oldRef.getObject();
		ref.timestamp = now;
		refSet.remove(ref);
	    } else curRefs++;
	    
	    // enforce refs per node limit
	    if (refSet.size() >= maxRefsPerNode) {
		refTree.treeRemove(refSet.pop());
		curRefs--;
	    }
	    
	    refSet.append(ref);
	    
	    mem.setProperty("keys", refSet);
	    
	    // enforce max nodes limit
	    if (routingStore.size() > maxNodes)
		discard(routingStore.size() / 20);
	}
	long end = System.currentTimeMillis();
	long length = end-now;
	Core.logger.log(this, "reference("+k+","+nr+") took "+
			length, 
			length>500?Core.logger.MINOR:Core.logger.DEBUG);
    }
    
    public synchronized boolean references(Identity id) {
        return routingStore.contains(id);
    }
    
	public synchronized NodeReference getNodeReference(Identity id) {
		RoutingMemory r =routingStore.getNode(id);
		return r==null?null:r.getNodeReference();
	}
    
	

    public synchronized Routing route(Key k) {
	return route(k, false);
    }

    public synchronized Routing route(Key k, boolean doDesperate) {
	long x = System.currentTimeMillis();
	Core.logger.log(this, "Entered route("+k+","+doDesperate+
			") at "+x, new Exception("debug"),
			Core.logger.DEBUG);
	MetricWalk w = new MetricWalk(refTree, new Reference(k), false);
	long y = System.currentTimeMillis();
	Core.logger.log(this, "Creating MetricWalk for "+k+" took "+(y-x)+
			" at "+y, Core.logger.DEBUG);
	TreeRouting r = new TreeRouting(this, w, doDesperate);
	long length = y-x;
	Core.logger.log(this, "route("+k+","+doDesperate+") locked RT for "+
			length+" at "+y, 
			length > 500 ? Core.logger.MINOR : Core.logger.DEBUG);
	return r;
    }
    
    public final RoutingStore getRoutingStore() {
        return routingStore;
    }

    public final Object semaphore() {
        return this;
    }

    private final Object discardSync = new Object();
    
    /**
     * Discards information about some nodes to free up memory.  The
     * deletion algorithm is derived from the subclass implementation.
     * All that is required is a Comparable object for each node.
     * 
     * This routine requires building a heap, so it is called in a
     * high water / low water mark pattern.
     */
    protected void discard(int num) {
	synchronized(discardSync) {
	    int onum = num;
	    Core.logger.log(this, "Discarding "+num+" references", 
			    Core.logger.MINOR);
	    Heap heap = new Heap(routingStore.size());
	    long startTime, endTime;
	    synchronized(this) { // FIXME: can we improve on this?
		startTime = System.currentTimeMillis();
		Enumeration rme = routingStore.elements();
		while (rme.hasMoreElements()) {
		    RoutingMemory mem = (RoutingMemory) rme.nextElement();
		    Comparable c = getDiscardValue(mem);
		    heap.put(new DiscardSort(c, mem));
		}
		endTime = System.currentTimeMillis();
	    }
	    long length = endTime-startTime;
	    Core.logger.log(this, "discard("+onum+") locked RT for "+length,
			    length>500?Core.logger.MINOR:Core.logger.DEBUG);
	    DiscardSort d;
	    while (num-- > 0 && null != (d = (DiscardSort) heap.pop()))
		remove(d.mem, d.ident);
	    Core.logger.log(this, "Discarded "+onum+" references",
			    Core.logger.DEBUG);
	}
    }
    
    /**
     * Remove a node from the routing table. Call with lock held.
     * @param mem the RoutingMemory corresponding to the node. Must not be null.
     * @param ident the identity, if known. May be null.
     */
    protected synchronized void remove(RoutingMemory mem, Identity ident) {
	ReferenceSet refSet = ReferenceSet.getProperty(mem, "keys");
	Enumeration refs = refSet.references();
	while(refs.hasMoreElements()) {
	    Reference ref = (Reference) refs.nextElement();
	    refTree.treeRemove(ref);
	    curRefs--;
	}
	routingStore.remove(ident == null ? mem.getIdentity() : ident);
	// Save a syscall
    }
    
    private static final class DiscardSort implements Comparable {
        final Comparable cmp;
        final Identity ident;
	final RoutingMemory mem;
        DiscardSort(Comparable cmp, RoutingMemory mem) {
            this.cmp = cmp;
	    this.mem = mem;
	    this.ident = mem.getIdentity();
        }
        public final int compareTo(Object o) {
            return compareTo((DiscardSort) o);
        }
        public final int compareTo(DiscardSort o) {
            return cmp.compareTo(o.cmp);
        }
    }

    /**
     * @return  an object that can be used to decide what node to discard
     *          by picking the nodes with the highest sort-value according
     *          to the Comparable returned.
     */
    protected abstract Comparable getDiscardValue(RoutingMemory mem);
    
    
    public abstract RTDiagSnapshot getSnapshot();

    // For book keeping.
    //protected abstract void attempted(RoutingMemory mem);
    // This is not needed because either routeConnected()
    // or connectFailed() is always called.
    
    protected abstract boolean isRoutable(RoutingMemory mem, 
					  boolean desperate);

    protected synchronized boolean isRoutable(RoutingMemory mem) {
	return isRoutable(mem, false);
    }
    
    // always call one of these:
    
    protected abstract void routeConnected(RoutingMemory mem);

    protected abstract void connectFailed(RoutingMemory mem);
    

    // then this can be called at any time:

    protected abstract void timedOut(RoutingMemory mem);

    
    // then only one of these (or timedOut()):
    
    protected abstract void authFailed(RoutingMemory mem);

    protected abstract void routeAccepted(RoutingMemory mem);
    
    
    // then only one of these (or timedOut()):
    
    protected abstract void routeSucceeded(RoutingMemory mem, boolean cached);
    
    protected abstract void transferFailed(RoutingMemory mem);
    
    protected abstract void verityFailed(RoutingMemory mem);

    protected abstract void queryRejected(RoutingMemory mem, boolean cached, long attenuation);
}



