package freenet.node.rt;

import freenet.Identity;
import freenet.node.NodeReference;
import freenet.support.Walk;
import freenet.support.BinaryTree.Node;
import freenet.support.MetricWalk;
import freenet.Core;
import java.util.Hashtable;

/**
 * This iterates over a metric walk of Reference nodes in the binary tree.
 * Each time a unique Identity is encountered it becomes the next route,
 * if its RoutingMemory can pass the isRoutable() test.
 * @author tavin
 */
class TreeRouting implements Routing {

    protected final Hashtable steps = new Hashtable();

    protected final TreeRoutingTable rt;
    
    protected final MetricWalk refWalk;

    protected final MetricWalk backupWalk;

    protected boolean desperate;
    protected boolean doDesperate = false;

    protected RoutingMemory mem;
    
    protected int stepCount = 0;
    protected boolean logDEBUG = false;

    TreeRouting(TreeRoutingTable rt, MetricWalk refWalk, boolean doDesperate) {
        this.rt = rt;
        this.refWalk = refWalk;
	this.backupWalk = (MetricWalk)refWalk.clone();
	this.desperate = false;
	this.doDesperate = doDesperate;
	logDEBUG = Core.logger.shouldLog(Core.logger.DEBUG); // HACK
	Core.logger.log(this, "Created TreeRouting at "+System.currentTimeMillis()+
			" for "+refWalk, Core.logger.DEBUG);
    }
    
    public NodeReference getNextRoute() {
	while (true) {
	    if (stepCount++ > freenet.node.Node.maxRoutingSteps) {
		// Just give up (RNF) once we have
		// tried a reasonable number of references.
		//
		// The rationale is that at some point the specialization
		// of the references that we are trying is so far away
		// from the search key that we are no longer doing
		// a useful steepest ascent search by trying further
		// refs.
		//
		mem = null;
		if((!doDesperate) || desperate) {
		    if(logDEBUG)
			Core.logger.log(this, 
					"Terminating routing with desperate="+
					desperate+", doDesperate="+
					doDesperate+", stepCount="+stepCount+" for "+
					refWalk, Core.logger.DEBUG);
		    return null;
		} else {
		    if(logDEBUG)
			Core.logger.log(this, "Going into desperation mode routing after "+stepCount+" steps for "+refWalk, Core.logger.DEBUG);
		    desperate = true;
		    stepCount=freenet.node.Node.maxRoutingSteps/2;
		    continue;
		}
	    }
		
	    if(logDEBUG)
		Core.logger.log(this, "Trying to getNextRoute... step "+stepCount+
				(desperate ? "(desperate)" : ""), Core.logger.DEBUG);
		
	    long doneTrivialTime = System.currentTimeMillis();
	    Node n;
	    long enteredTime = -1;
	    synchronized(rt) {
		enteredTime = System.currentTimeMillis();
		long syncTime = enteredTime-doneTrivialTime;
		if(syncTime>500)
			if(Core.logger.shouldLog(Core.logger.MINOR)) Core.logger.log(this, "Got lock on RT in "+syncTime+" for "+refWalk, Core.logger.MINOR);
		else
			 if(logDEBUG) Core.logger.log(this, "Got lock on RT in "+syncTime+" for "+refWalk, Core.logger.DEBUG);
		
		
		
		/* BinaryTree's are NOT thread-safe, and the convention is to lock on the 
		   routingTable (it does not lock the tree). Locking the RT for the whole
		   of this section may be a bit excessive. */
		n = (Node) (desperate ? backupWalk.getNext() : 
			    refWalk.getNext());
	    }
	    
	    long gotNodeTime = System.currentTimeMillis();
	    long syncedTime = gotNodeTime - enteredTime;
	    if(syncedTime>500)
		if(Core.logger.shouldLog(Core.logger.MINOR)) Core.logger.log(this, "Got node (synchronized) from "+refWalk+ " in "+syncedTime,Core.logger.MINOR);
	    else
	    	if(logDEBUG) Core.logger.log(this, "Got node (synchronized) from "+refWalk+ " in "+syncedTime,Core.logger.DEBUG);
			    
	    
	    if (n == null) {
		mem = null;
		if(logDEBUG)
		    Core.logger.log(this, "refWalk returned null", Core.logger.DEBUG);
		if((!doDesperate) || desperate) {
		    if(logDEBUG)
			Core.logger.log(this, "... so quitting for "+refWalk,
					Core.logger.DEBUG);
		    return null;
		} else {
		    desperate = true;
		    stepCount--;
		    if(logDEBUG)
			Core.logger.log(this, "... going into desperation mode: stepCount="+
					stepCount+" for "+refWalk, Core.logger.DEBUG);
		    continue;
		}
	    };
	    if(logDEBUG)
		Core.logger.log(this, "Got a non-null node from refWalk ("+refWalk+
				")", Core.logger.DEBUG);
	    Reference ref = (Reference) n.getObject();
	    if (!steps.containsKey(ref.ident)) {
		if(logDEBUG)
		    Core.logger.log(this, "Node not in already tried nodes",
				    Core.logger.DEBUG);
		
		mem = rt.routingStore.getNode(ref.ident);
		// RoutingStore is threadsafe
		
		long gotFromRTTime = System.currentTimeMillis();
		if(logDEBUG) Core.logger.log(this, "Got mem from RT in "+(gotFromRTTime-gotNodeTime),
				Core.logger.DEBUG);
		if (mem == null) {  // race condition ?
		    if(logDEBUG)
			Core.logger.log(this, 
					"Node no longer in routing table!",
					Core.logger.DEBUG);
		    stepCount--;
		    continue;
		}
		if(logDEBUG)
		    Core.logger.log(this, 
				    "Still here; node is in routing table; memory is "+
				    mem+", at "+System.currentTimeMillis()+", for "+
				    refWalk, Core.logger.DEBUG);
		if (((CPAlgoRoutingTable)rt).isRoutable(mem, desperate, " for "+refWalk)) { // routing table is threadsafe
		    if(logDEBUG)
			Core.logger.log(this, "Node is routable",
					Core.logger.DEBUG);
		    long routableTime = System.currentTimeMillis();
			if(logDEBUG) Core.logger.log(this, "Is routable in "+(routableTime-gotFromRTTime),
				    Core.logger.DEBUG);
		    steps.put(ref.ident, ref.ident);
		    //rt.attempted(mem);
		    NodeReference r = mem.getNodeReference();
		    long gotNewRefTime = System.currentTimeMillis();
			if(logDEBUG) Core.logger.log(this, "Returning route in "+
				    (gotNewRefTime-routableTime)+" for "+refWalk, 
				    Core.logger.DEBUG);
		    return r;
		} else {
		    long notRoutableTime = System.currentTimeMillis();
		    if(logDEBUG)
			Core.logger.log(this, "Node is NOT routable in "+
					(notRoutableTime-gotFromRTTime)+" at "+
					notRoutableTime+" for "+refWalk, 
					Core.logger.DEBUG);
		    stepCount--; // steps must be actually tried
		}
	    } else {
		if(logDEBUG)
		    Core.logger.log(this, "Node is in already tried nodes",
				    Core.logger.DEBUG);
		stepCount--;
	    }
	}
    }
    
    public void ignoreRoute() {
	if(stepCount>0) stepCount--;
    }

    public final Identity getLastIdentity() {
        return mem == null ? null : mem.getIdentity();
    }

    public final void routeConnected() {
        rt.routeConnected(mem);
    }

    public final void routeAccepted() {
        rt.routeAccepted(mem);
    }

    public final void routeSucceeded(boolean cached) {
        rt.routeSucceeded(mem, cached);
    }

    public final void connectFailed() {
        rt.connectFailed(mem);
    }

    public final void authFailed() {
        rt.authFailed(mem);
    }

    public final void timedOut() {
        rt.timedOut(mem);
    }

    public final void transferFailed() {
        rt.transferFailed(mem);
    }

    public final void verityFailed() {
        rt.verityFailed(mem);
    }

    public final void queryRejected(boolean cached, long attenuation) {
        rt.queryRejected(mem, cached, attenuation);
    }
}


