package freenet.node.rt;

import freenet.Key;
import freenet.Identity;
import freenet.node.NodeReference;

/**
 * @author tavin
 */
public interface RoutingTable {

    /**
     * Associates a NodeReference with a key.
     *
     * @param k  The key to store the reference under
     * @param nr  The node to associate with the key
     */
    void reference(Key k, NodeReference nr);

    /**
     * @return  true, if the routing table references a node
     *          with the given Identity
     */
    boolean references(Identity id);
    
	/**
	 * @return  returns the NodeReference for the given Identity
	 * null if no such noderef is present
	 */
	NodeReference getNodeReference(Identity id);

    /**
     * Returns a Routing object which can be used to get
     * a sequence of NodeReference objects in order of
     * routing preference, and which accepts feedback
     * about the results of routing.
     *
     * @param k  the key to find routes for
     * @param force whether to try to contact backed off nodes
     * @return   the generated Routing object
     */
    Routing route(Key k, boolean force);

    /**
     * NOTE:  It is important to hold the sync-lock
     *        while using the RoutingStore.
     * @see semaphore()
     * @return  the underlying RoutingStore
     */
    RoutingStore getRoutingStore();

    /**
     * Actions on the routing table are naturally synchronized.
     * This returns the synchronization lock so that it can be
     * held between method calls.
     * @return  an object to synchronize on when doing multiple
     *          correlated operations on the routing table
     */
    Object semaphore();
    
    /**
     * Returns diagnostic information about the current
     * state of the RoutingTable implementation.
     *
     * @return The diagnostic info. This can be null.
     */
    RTDiagSnapshot getSnapshot();
    
    /**
     * Returns the total number of references in the
     * RoutingTable
     */
    int getKeyCount();

    /**
     * Returns the total number of references found when 
     * initializing
     */
    public long initialRefsCount();
}



