package freenet.node.rt;

import freenet.Identity;
import freenet.node.NodeReference;

/**
 * Supplies a sequence of NodeReferences in order of routing
 * preference.  Feeds information about the results of routing
 * back into the implementation for use in future routing decisions.
 *
 * The methods of the Routing object must be called in this
 * sequence (for each NodeReference obtained from getNextRoute()):
 *
 * 1) Either routeConnected() or connectFailed()
 * 2) Either authFailed(), routeAccepted(), or timedOut()
 * 3) Either routeSucceeded(), transferFailed(),
 *           verityFailed(), or timedOut()
 *
 * @author tavin
 */
public interface Routing {

    /**
     * @return  the next route to try, or null
     *          if we are out of possibilities
     */
    NodeReference getNextRoute();

    /**
     * Ignore the last route, for purposes of maxRouteSteps
     * Called when we try to route to the originator
     */
    void ignoreRoute();

    /**
     * @return  the Identity of the last returned NodeReference
     */
    Identity getLastIdentity();


    // First, call one of these two:
    
    /**
     * Called if the routed node is connected to
     * and authorized with successfully.
     */
    void routeConnected();

    /**
     * Called if communications with the routed node fail.
     */
    void connectFailed();


    // From here on out timedOut() may be called at any time:
    
    /**
     * Called if the routed node times out during message handling.
     */
    void timedOut();

    
    // Then, call only one of these (or timedOut()):
    
    /**
     * Called if the routed node fails authorization
     * (by being unable to prove its PK credentials).
     * Connection-related failures during authorization
     * should call connectFailed().
     */
    void authFailed();

    /**
     * Called when the Accepted is received from the routed node.
     */
    void routeAccepted();


    // Then call only one of these (or timedOut()):
    
    /**
     * Called if the routed node successfully handles
     * the message chain.
     * @param cached This should be set if the connection was cached.
     */
    void routeSucceeded(boolean cached);

    /**
     * Called if the routed node fails during file transfer.
     */
    void transferFailed();

    /**
     * Called if the routed node sends bad data.
     */
    void verityFailed();

    /**
     * Called if the routed node successfully
     * handles the message chain but the
     * request is answered with a QueryRejected.
     */
    void queryRejected(boolean cached, long attenuation);
}




