package freenet.node.rt;

import freenet.FieldSet;
import freenet.Identity;
import freenet.node.NodeReference;
import freenet.node.BadReferenceException;
import freenet.fs.dir.FileNumber;
import freenet.support.*;
import freenet.support.io.*;
import freenet.Core; // for logging
import java.io.*;

/**
 * Acts as a handle for a given node and a container for its node reference.
 * Named properties of the node are retrieved through this handle.
 * @author tavin
 */
class DataObjectRoutingMemory implements RoutingMemory, DataObject {

    
    private final DataObjectRoutingStore routingStore;
    
    NodeReference noderef;
    private final byte[] identfp;

    public String toString() {
	return "DataObjectRoutingMemory:"+noderef.toString()+
	    ":"+Fields.bytesToHex(identfp);
    }
    
    DataObjectRoutingMemory(DataObjectRoutingStore routingStore,
                            NodeReference noderef) {
        this.routingStore = routingStore;
        this.noderef = noderef;
        this.identfp = noderef.getIdentity().fingerprint();
    }

    
    DataObjectRoutingMemory(DataObjectRoutingStore routingStore,
                            DataObjectPending dop) throws BadReferenceException,
                                                          IOException {
        this.routingStore = routingStore;
        FieldSet fs = new FieldSet(new ReadInputStream(dop.getDataInputStream()));
        this.noderef = new NodeReference(fs);
        this.identfp = noderef.getIdentity().fingerprint();
        dop.resolve(this);
    }
    

    /**
     * @return  the node's identity
     */
    public final Identity getIdentity() {
        return noderef.getIdentity();
    }

    /**
     * @return  the node's node reference
     */
    public final NodeReference getNodeReference() {
        return noderef;
    }
    
    /**
     * @return  the named DataObject, or null if not found
     */
    public final DataObject getProperty(String name) 
	throws DataObjectUnloadedException {
	long startedTime = -1;
	Core.logger.log(this, "getProperty("+name+")", 
			Core.logger.DEBUG);
	FileNumber fn = makeFileNumber(name);
	try {
	    startedTime = System.currentTimeMillis();
	    DataObject o = routingStore.rtProps.get(fn);
	    long gotObjectTime = System.currentTimeMillis();
	    long len2 = gotObjectTime - startedTime;
	    if(Core.logger.shouldLog(Core.logger.DEBUG) || len2>500)
		Core.logger.log(this, "getProperty("+name+") getting DataObject took "+
				len2, len2>500?Core.logger.MINOR:Core.logger.DEBUG);
	    return o;
	} catch (DataObjectUnloadedException e) {
	    long thrownTime = System.currentTimeMillis();
	    long len3 = thrownTime - startedTime;
	    if(Core.logger.shouldLog(Core.logger.DEBUG) || len3>500)
		Core.logger.log(this, "getProperty("+name+
				") throwing DataObjectUnloadedException; get("+
				fn+") took "+len3,
				len3>500?Core.logger.MINOR:Core.logger.DEBUG);
	    throw e;
	}
    }
    
    /**
     * Schedules the named DataObject for saving to disk.
     * Could be better named!
     */
    public final void setProperty(String name, DataObject o) {
        routingStore.rtProps.set(makeFileNumber(name), o);
    }

    /**
     * Make a FileNumber from the byte array resulting from
     * concatenating the Identity fingerprint with the UTF-8
     * representation of ".<name>"
     */
    private final FileNumber makeFileNumber(String name) {
        byte[] nm = ('.'+name).getBytes();
        byte[] fn = new byte[identfp.length + nm.length];
        System.arraycopy(identfp, 0, fn, 0, identfp.length);
        System.arraycopy(nm, 0, fn, identfp.length, nm.length);
        return new FileNumber(fn);
    }
    

    // reuse instance for getDataLength() computations
    private static final ByteArrayOutputStream bout = new ByteArrayOutputStream();

    /**
     * Determine the field-set's byte length when written out.
     */
    public final int getDataLength() {
        synchronized (bout) {
            try {
                writeTo(new DataOutputStream(bout));
                return bout.size();
            }
            catch (IOException e) {
                return 0;
            }
            finally {
                bout.reset();
            }
        }
    }
    
    /**
     * Write the node ref's field-set.
     */
    public final void writeTo(DataOutputStream out) throws IOException {
        noderef.getFieldSet().writeFields(new WriteOutputStream(out));
    }
}




