package freenet.node.rt;

import freenet.BadAddressException;
import freenet.Address;
import freenet.Key;
import freenet.Identity;
import freenet.FieldSet;
import freenet.KeyException;
import freenet.client.FreenetURI;
import freenet.client.InternalClient;
import freenet.crypt.DSAPublicKey;
import freenet.node.NodeReference;
import freenet.node.Node;
import freenet.node.Main;
import freenet.client.AutoBackoffNodeRequester;
import freenet.node.BadReferenceException;
import freenet.node.states.maintenance.Checkpoint;
import freenet.support.Bucket;
import freenet.support.ArrayBucket;
import freenet.support.io.ReadInputStream;
import freenet.support.DataObject;
import freenet.support.DataObjectPending;
import freenet.support.DataObjectUnloadedException;
import freenet.support.StringMap;
import freenet.support.SimpleStringMap;
import freenet.support.sort.*;
import freenet.support.Comparable;
import freenet.support.Fields;
import freenet.transport.TCP;
import freenet.Transport;
import freenet.TransportHandler;
import freenet.transport.tcpAddress;
import freenet.Core;
import freenet.support.Logger;

import java.util.Random;
import java.util.Vector;
import java.util.Enumeration;
import java.util.HashSet;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;

/**
 * RoutingTable implementation which uses the tested CP
 * algorithm that was in place before Tavin ripped out 
 * the old routing code.
 *
 * @author giannij
 */
public class CPAlgoRoutingTable extends TreeRoutingTable {

    protected final Random rand;
    
    public CPAlgoRoutingTable(RoutingStore routingStore,
			      int maxNodes, int maxRefsPerNode,
			      int consecutiveFailuresLookupARK,
			      int minDowntimeARKLookup, float minCP,
			      int maxLookupThreads, Random rand)
	throws IOException {
        super(routingStore, maxNodes, maxRefsPerNode);
        this.rand = rand;
	this.node = null;
	this.consecutiveFailuresLookupARK = 
	    consecutiveFailuresLookupARK;
	this.minDowntimeARKLookup = minDowntimeARKLookup;
	this.minCP = minCP;
	this.maxLookupThreads = maxLookupThreads;
    }
    
    int consecutiveFailuresLookupARK;
    int minDowntimeARKLookup;
    float minCP;
    int maxLookupThreads;

    public RTDiagSnapshot getSnapshot() {
        long nowMs = System.currentTimeMillis();

        // Reset ref aggregate counters
        totalRefs = 0;
        contactedRefs = 0;
	requestingRefs = 0;

        // ^^^ the fact that you have to do this suggests these
        //     should be local variables not instance variables  -tc
        //
        // 6 of 1, 1/2 dozen of the other. makeRefInfo() updates
        // the counts as a side effect, if you were morally opposed
        // to side effects you could get rid of makeRefInfo
        // (decreasing readability).  --gj

        Vector nodeRefs = new Vector();
        Vector refInfos = new Vector();
	HashSet keys = new HashSet();

	boolean shouldLog = Core.logger.shouldLog(Core.logger.DEBUG);
	synchronized(this) {
	    if(shouldLog)
		Core.logger.log(this, "Starting getSnapshot()", Core.logger.DEBUG);
	    Enumeration memories = routingStore.elements();
	    while (memories.hasMoreElements()) {
		if(shouldLog)
		    Core.logger.log(this, "Next memory element", Core.logger.DEBUG);
		RoutingMemory memory = (RoutingMemory)memories.nextElement();
		NodeReference nodeRef = memory.getNodeReference();
		if (nodeRef == null) {
		    if(shouldLog)
			Core.logger.log(this, "ignored memory because nodeRef null",
					Core.logger.DEBUG);
		    continue;
		}
		
		ReferenceSet refSet = ReferenceSet.getProperty(memory, "keys");
		Enumeration refs = refSet.references();
		while (refs.hasMoreElements()) {
		    if(shouldLog)
			Core.logger.log(this, "Next ref element", Core.logger.DEBUG);
		    Reference ref = (Reference) refs.nextElement();
		    if (!keys.contains(ref.key)) {
			// REDFLAG: Change to only exclude "0000" keys once code is 
			//          fixed to mark all non-data keys with "0000"
			byte[] b = ref.key.getVal();
			if(b.length>=2 && 
			   ((b[b.length-1] == 2 && b[b.length-2] == 3) ||
			    (b[b.length-2] == 2 && b[b.length-1] == 3))) {
			    // Only keys ending in 0x0302 or 0x0203 are valid data
			    keys.add(ref.key);
			    if(shouldLog)
				Core.logger.log(this, "Got new key " + 
						ref.key.toString(),Core.logger.DEBUG);
			} else {
			    if(shouldLog)
				Core.logger.log(this, "Ignored nonstandard key ending",
						Core.logger.DEBUG);
			} 
		    } else {
			if(shouldLog)
			    Core.logger.log(this, "Ignored duplicate key", 
					    Core.logger.DEBUG);
		    }
		}
		if (!nodeRefs.contains(nodeRef)) {
		    final StringMap refInfo = makeRefInfo(nodeRef, memory, nowMs); 
		    refInfos.addElement(refInfo);
		    nodeRefs.addElement(nodeRef);
		}
	    }
	}
	
        CPSortableStringMap[] refs = new CPSortableStringMap[refInfos.size()];
        refInfos.copyInto(refs);
	
        Object[] props = new Object[TABLE_PROPERTIES.length];
        props[0] = new Integer(totalRefs);
        props[1] = new Integer(contactedRefs);
        props[2] = new Integer(requestingRefs);
        props[3] = new Integer(totalAttempts);
        props[4] = new Integer(totalSuccesses);
        props[5] = getClass().getName();
        
	Key[] keyList = (Key[])keys.toArray(new Key[keys.size()]);
	if(shouldLog)
	    Core.logger.log(this,"got " + keyList.length + " keys in getSnapshot",
			    Core.logger.DEBUG);
	
	int countedKeys = getKeyCount();
	if(keyList.length != countedKeys)
	    Core.logger.log(this, "Supposed to have "+countedKeys+" refs but actually "+
			    "have "+keyList.length, Logger.DEBUG);
	
        HeapSorter.heapSort(new ArraySorter(refs));
	
        return new SimpleRTDiagSnapshot(new SimpleStringMap(TABLE_PROPERTIES, props),
                                        refs, keyList);
    }
    
    private final static String RMKEY = "cpdata";
 
    
    protected final Comparable getDiscardValue(RoutingMemory mem) {
        CPAlgoData cpd = getProperty(mem, RMKEY, node);
        return new DiscardValue(cpd.lastSuccessfulContact);
    }

    private static final class DiscardValue implements Comparable {
        final long time;
        DiscardValue(long time) {
            this.time = time;
        }
        public final int compareTo(Object o) {
            return compareTo((DiscardValue) o);
        }
        public final int compareTo(DiscardValue o) {
            return time == o.time ? 0 : (time < o.time ? 1 : -1);
        }
    }
    

    private final synchronized void incTrials(RoutingMemory mem) {
        CPAlgoData cpd = getProperty(mem, RMKEY, node);
        cpd.trials++;
        // We don't call mem.setProperty() because trials isn't
        // persisted.
    }
    
    private final synchronized void fail(RoutingMemory mem) {
	fail(mem, false);
    }
    
    private final synchronized void fail(RoutingMemory mem, boolean weak) {
        CPAlgoData cpd = getProperty(mem, RMKEY, node);
        if (cpd.decreaseContactProbability(weak)) {
	    remove(mem, null);
            return;
        }
        mem.setProperty(RMKEY, cpd);
    }
    
    protected final synchronized void remove(Identity i) {
	// HACK!
	remove((RoutingMemory)(routingStore.getNode(i)), i);
    }

    private final synchronized void succeed(RoutingMemory mem, boolean cached,
					    boolean attenuated, boolean weak) {
        CPAlgoData cpd = getProperty(mem, RMKEY, node);
        cpd.successes++;
        if (!cached) {
            // Uncached connections don't count.
            cpd.increaseContactProbability(weak);
        }
        mem.setProperty(RMKEY, cpd);
    }

    /**
     * Succeed and/or fail on weak or strong in a single step
     * @param strongSuccess whether we strongly (connection level) succeded
     * @param weakSuccess whether we weakly (query level) succeeded
     * @param ignoreStrong whether to leave the strong component unchanged
     * - generally set when we are dealing with a cached connection
     */
    protected final synchronized void succeedFail(RoutingMemory mem,
						  boolean strongSuccess,
						  boolean weakSuccess,
						  boolean ignoreStrong) {
	CPAlgoData cpd = getProperty(mem, RMKEY, node);
	if(!ignoreStrong) {
	    if(!strongSuccess) {
		if(cpd.decreaseContactProbability(false)) {
		    remove(mem, null);
		    return;
		}
	    } else {
		cpd.increaseContactProbability(false);
	    }
	}
	if(weakSuccess) {
	    cpd.increaseContactProbability(true);
	} else {
	    if(cpd.decreaseContactProbability(true)) {
		remove(mem, null);
		return;
	    }
	}
	if(weakSuccess || strongSuccess)
	    cpd.successes++;
	mem.setProperty(RMKEY, cpd);
    }
    
    protected boolean isRoutable(RoutingMemory mem,
				 boolean desperate) {
	return isRoutable(mem, desperate, "");
    }
    
    protected boolean isRoutable(RoutingMemory mem, 
				 boolean desperate, String addToLog) {
	// Move all var decls up to the top to work around bugs in sun 1.4.1.
	long startTime = System.currentTimeMillis();
	boolean shouldLog = Core.logger.shouldLog(Core.logger.DEBUG);
	long gotFloatTime;
	long syncedTime = -1;
	long len, gotPropertyTime, gotCPTime, unsyncedTime, length;
	float y;
	CPAlgoData cpd;
	if(shouldLog)
	    Core.logger.log(this, "isRoutable("+mem+","+desperate+") at "+
			    startTime+addToLog, Core.logger.DEBUG);
	float f = rand.nextFloat();
	gotFloatTime = System.currentTimeMillis();
	if(shouldLog)
	    Core.logger.log(this, "Got float for isRoutable("+mem+","+desperate+
			    ") at "+gotFloatTime+" in "+(gotFloatTime-startTime), 
			    Logger.DEBUG);
	long returningTime = -1;
	try {
	    synchronized(this) {
		syncedTime = System.currentTimeMillis();
		len = syncedTime - gotFloatTime;
		if(shouldLog || len > 500)
		    Core.logger.log(this, "isRoutable("+mem+","+desperate+
				    ") synchronized up in "+len+" at "+
				    syncedTime+addToLog, 
				    len>500?Core.logger.MINOR:Core.logger.DEBUG);
		cpd = getProperty(mem, RMKEY, node);
		gotPropertyTime = System.currentTimeMillis();
		if(shouldLog || (gotPropertyTime - syncedTime) > 500)
		    Core.logger.log(this, "Got property for "+mem+" in "+
				    (gotPropertyTime - syncedTime)+" at "+
				    gotPropertyTime+addToLog, 
				    new Exception("debug"),
				    ((gotPropertyTime - syncedTime)>500)
				    ? Core.logger.MINOR : Core.logger.DEBUG);
		if(desperate) {
		    if(shouldLog)
			Core.logger.log(this, "Returning because desperate for " +
					"isRoutable("+mem+") at "+
					System.currentTimeMillis()+addToLog,
					Core.logger.DEBUG);
		    returningTime = System.currentTimeMillis();
		    return true;
		} else {
		    y = cpd.contactProbability();
		    gotCPTime = System.currentTimeMillis();
		    if(shouldLog || (gotCPTime - gotPropertyTime)>500)
			Core.logger.log(this, "Got contact probability for "+mem+
					" in "+(gotCPTime - gotPropertyTime)+addToLog+
					" at "+gotCPTime,
					((gotCPTime - gotPropertyTime)>500)
					? Core.logger.MINOR : Core.logger.DEBUG);
		    returningTime = System.currentTimeMillis();
		    return f <= y;
		}
	    }
	} finally {
	    unsyncedTime = System.currentTimeMillis();
	    if(shouldLog) {
		Core.logger.log(this, "Returning through finally took "+
				(unsyncedTime-returningTime)+", returning was at "+
				returningTime+" for isRoutable("+mem+","+desperate,
				Logger.DEBUG);
	    }
	    length = unsyncedTime - syncedTime;
	    if(shouldLog || length > 500)
		Core.logger.log(this, "isRoutable("+mem+","+desperate+
				") locked RT for "+length+addToLog+", totally took "+
				(unsyncedTime-startTime)+" at "+unsyncedTime,
				length>500 ? Core.logger.MINOR : Core.logger.DEBUG);
	    if(shouldLog)
		Core.logger.log(this, "Returning through finally took "+
				(unsyncedTime-returningTime)+", returning was at "+
				returningTime+" for isRoutable("+mem+","+desperate,
				Logger.DEBUG);
	}
    }
    
    protected synchronized void routeConnected(RoutingMemory mem) {
        incTrials(mem);
        totalAttempts++;
    }

    protected synchronized void routeAccepted(RoutingMemory mem) {
        // hmm.. what?
    }

    protected synchronized void routeSucceeded(RoutingMemory mem, boolean cached) {
        // Don't reward cached connections.
        succeedFail(mem, true, true, cached);
        totalSuccesses++;
    }

    protected synchronized void connectFailed(RoutingMemory mem) {
        incTrials(mem);
        totalAttempts++;
        fail(mem);
    }

    protected synchronized void authFailed(RoutingMemory mem) {
	fail(mem);
	// don't deref it completely in case this is corruption or something
	// don't ignore it because we want to clear out nodes that change identity reasonably quickly
    }

    protected synchronized void timedOut(RoutingMemory mem) {
        fail(mem);
    }

    protected synchronized void transferFailed(RoutingMemory mem) {
        fail(mem);
    }

    // this is not a tyop!
    protected synchronized void verityFailed(RoutingMemory mem) {
        // dereference node
        remove(mem, null);
    }

    private final synchronized boolean deleteReferences(ReferenceSet refSet, 
							long number) {
	if(Core.logger.shouldLog(Core.logger.DEBUG))
	    Core.logger.log(this, "Removing "+number+" references.", 
			    Core.logger.DEBUG);
	long startTime = System.currentTimeMillis();
        if (refSet.size() == 0) {
            return false;
        }
        
        while (number > 0 ) {
            Reference ref = refSet.pop();
            if (ref == null) {
                break;
            }
            refTree.treeRemove(ref);
            number--;
        }
	
	long endTime = System.currentTimeMillis();
	long length = endTime-startTime;
	if(Core.logger.shouldLog(Core.logger.DEBUG) || length > 500)
	    Core.logger.log(this, "deleteReferences held RT lock for "+length,
			    length>500 ? Core.logger.MINOR : Core.logger.DEBUG);
        return true;
    }
    
    // Reduce the chances of routing to nodes that always 
    // replies with QueryRejected.
    protected synchronized void queryRejected(RoutingMemory mem, boolean cached, long attenuation) {
        
        // I would have preferred to make queryRejected() in addition
        // to routeSucceeded() to keep transport and application
        // levels separate. --gj
        
        // Counter-intutitive.  The routing has succeeded at the 
        // transport layer,  but we got a QueryRejected 
        // back at the application layer.
        //
	succeedFail(mem, true, false, cached);
        totalSuccesses++;

        // Disable routing attenuation. 20020421 --gj
        //         
        //          ReferenceSet refSet = ReferenceSet.getProperty(mem, "keys");
        //          int refCount = refSet.size();
        
        //          if (refCount < 4) {
        //              // Never remove the very last three refs.
        //              return;
        //          }
        
        //          if (refCount <= attenuation) {
        //              attenuation = refCount - 1;
        //          }
        
        //          if (attenuation < 1) {
        //              return;
        //          }
        
        //          // We want to remove refs more aggressively when 
        //          // there are many.
        //          if (rand.nextFloat() < (1.0f - (1.0f / refCount))) {
        //              if (deleteReferences(refSet, attenuation)) {
        //                  mem.setProperty("keys", refSet);
        //              }
        //          }
    }

    ////////////////////////////////////////////////////////////
    // Helper functions used to implement diagnostic snapshots.
    private final static String[] REF_PROPERTIES 
        = {"Address", "Contact Probability", "Consecutive Failures",
           "Connection Attempts", "Successful Connections", 
           "Last Attempt", "NodeReference", "Node Version", "Key Count",
	   "ARK version", "Fetching ARK", "ARK URL"};

    private final static String[] TABLE_PROPERTIES 
        = {"Number of node references", "Contacted node references",
           "Node references requesting ARKs", "Total Trials", 
	   "Total Successes", "Implementation" };

    // ^^^ i don't understand why you are doing this here
    //     instead of just passing the NodeReference back  -tc
    //
    // The point is to present data that the client
    // (i.e. the caller of getSnapShot()) can immediately 
    // display without doing any more work, or even having
    // to know the type of the value.  
    // NodeReference.toString() doesn't give a value that 
    // makes sense to an end user. --gj


    private final static Long lastAttemptSecs(CPAlgoData cpd, long nowMs, long thenMs) {
        long ret = -1; // never
        if (cpd.trials > 0) {
	    long x = cpd.lastRetryMs;
	    if(x > 0) {
		// trials++ does not immediately imply lastRetryMs
		// lastRetryMs is changed only when we have a result - success or failure
		ret = (nowMs - cpd.lastRetryMs) / 1000;
		if (ret < 1) {
		    ret = 1;
		}
	    }
	}
        return new Long(ret);
    }
    
    // ARK utility method
    // REDFLAG: move to NodeReference?
    
    private int totalAttempts = 0;
    private int totalSuccesses = 0;

    // Diagnostic stats set by makeRefInfo
    private int totalRefs = 0;
    private int requestingRefs = 0;
    private int contactedRefs = 0;

    private final Long LONG_MINUS_ONE = new Long(-1);
    private final Integer INT_ZERO = new Integer(0);
    private final Long LONG_ZERO = new Long(0);

    ////////////////////////////////////////////////////////////
    static class CPSortableStringMap extends SimpleStringMap implements Comparable {
        public CPSortableStringMap(String[] keys, Object[] objs) {
            super(keys, objs);
        }

        public int compareTo(Object o) {
            if (!(o instanceof CPSortableStringMap)) {
                throw new IllegalArgumentException("Not a CPSortableStringMap"); 
            }

            final float cpA = ((Float)objs[1]).floatValue();
            final float cpB = ((Float)((CPSortableStringMap)o).objs[1]).floatValue();

            final float delta = cpB - cpA;

            if (delta > 0.0f) {
                return 1;
            }
            else if (delta < 0.0f) {
                return -1;
            }
            else {
                return 0;
            }
        }
    }

    ////////////////////////////////////////////////////////////
    private final CPSortableStringMap makeRefInfo(NodeReference ref, RoutingMemory mem, long nowMs) {
        CPAlgoData cpd = getProperty(mem, RMKEY, node);
        Object[] values = new Object[REF_PROPERTIES.length];
        values[0] = ref.firstPhysicalToString();
        // Don't phreak out cluebies with 0.0 cp during backoff.
        values[1] = new Float(cpd.effectiveCP(nowMs));
	values[2] = new Long(cpd.consecutiveFailures);
        ///values[2] = new Integer(cpd.failureIntervals);
        values[3] = new Long(cpd.trials);
        values[4] = new Long(cpd.successes);
        ///values[5] = backedOffSecs(cpd, nowMs);
        values[5] = lastAttemptSecs(cpd, nowMs, cpd.lastRetryMs);
        values[6] = ref; // refs are immutable
        values[7] = ref.getVersion();
        ///values[9] = new Boolean(cpd.routingBackedOff());
        ReferenceSet refSet = ReferenceSet.getProperty(mem, "keys");
        values[8] = new Long(refSet.size());
	values[9] = new Long(ref.revision());
	values[10] = new Boolean(cpd.fetchingARK());
	try {
	    FreenetURI uri = ref.getARKURI(ref.revision());
	    if(uri != null)
		values[11] = uri.toString(false);
	    else values[11] = null;
	} catch (KeyException e) {
	    Core.logger.log(this, "KeyException thrown getting ARK URI in makeRefInfo: "+
			    e, e, Logger.ERROR);
	    values[11] = null;
	}
	
        // Update aggregate stats
        totalRefs++;
	if (cpd.fetchingARK()) {
	    requestingRefs++;
	}

	if (cpd.successes > 0) {
            contactedRefs++;
        }

        return new CPSortableStringMap(REF_PROPERTIES, values); 
    }

    protected Node node = null;

    public void setNode(Node node) {
	if(node == null) throw new IllegalStateException("Tried to set node to null!");
	this.node = node;
    }
    
    final CPAlgoData getProperty(RoutingMemory mem, String name,
			   Node node) {
	CPAlgoData ret;
	long startTime = System.currentTimeMillis();
	boolean shouldLog = Core.logger.shouldLog(Core.logger.DEBUG);
	if(shouldLog)
	    Core.logger.log(this, "getProperty("+mem+","+name+",node) at "+startTime,
			    Core.logger.DEBUG);
	try {
	    ret = (CPAlgoData) mem.getProperty(name);
	}
	catch (DataObjectUnloadedException e) {
	    long thrownTime = System.currentTimeMillis();
	    if(shouldLog || (thrownTime - startTime) > 500)
		Core.logger.log(this, "getProperty got DataObjectUnloadedException "+
				"after "+(thrownTime - startTime),
				((thrownTime - startTime) > 500)?
				Core.logger.MINOR:Core.logger.DEBUG);
	    ret = new CPAlgoData(e, mem.getNodeReference());
	    long constTime = System.currentTimeMillis();
	    long constLength = constTime-thrownTime;
	    if(shouldLog || constLength > 500)
		Core.logger.log(this, "Constructed CPAlgoData from "+
				"DataObjectUnloadedException in "+constLength, 
				constLength>500 ? 
				Core.logger.MINOR : Core.logger.DEBUG);
	    mem.setProperty(name, ret);
	    long setTime = System.currentTimeMillis();
	    long setLength = setTime - constTime;
	    if(shouldLog)
		Core.logger.log(this, "Set property "+name+" to mem ("+mem+
				") in "+setLength, Core.logger.DEBUG);
	}
	long gotTime = System.currentTimeMillis();
	long gotLength = gotTime - startTime;
	if(shouldLog || gotLength>500)
	    Core.logger.log(this, "Got property from mem ("+mem+") in "+gotLength,
			    gotLength>500?Core.logger.MINOR:Core.logger.DEBUG);
	if (ret == null) {
	    ret = new CPAlgoData(mem.getNodeReference());
	    mem.setProperty(name, ret);
	    long setTime = System.currentTimeMillis();
	    long setLength = setTime - gotTime;
	    if(shouldLog)
		Core.logger.log(this, "Set property "+name+" to mem ("+mem+
				") in "+setLength, Core.logger.DEBUG);
	}
	return ret;
    }
    
    int lookupThreads = 0;
    Object lookupThreadsLock = new Object();
    
    ////////////////////////////////////////////////////////////
    /**
     * Class to encapsulate the data used to implement the tested CP
     * algorithm that was in place before Tavin ripped out the old
     * routing code.
     * <p>
     *
     **/
    class CPAlgoData implements DataObject {
	
	// persisted.
	float contactProbability = 1.0f;
	
	// Diagnostic info
	long successes = 0;
	long trials = 0;
	
	// Number of consecutive failures
	long consecutiveFailures = 0;
	
	// FIXME
// 	static final long CONSECUTIVEFAILURESLOOKUPARK = 20;
// 	static final long MINOLDMILLISLOOKUPARK = 900*1000; // 15 minutes
// 	static final float MINCP = 0.01F;
	
	// Time of last successful contact
	long lastSuccessfulContact = -1;
	
	long lastRetryMs = -1;

	// ARK fetcher
	LookupARK lookup = null;
	Object lookupLock = new Object();
	boolean justFetchedARK = false;

	NodeReference ref;
	
	
	////////////////////////////////////////////////////////////
	
	private CPAlgoData(NodeReference ref) {
	    this.ref = ref;
	}

	protected void finalize() {
	    synchronized(lookupLock) {
		if(lookup != null) {
		    Core.logger.log(this, "Killing ARK lookup in finalizer",
				    Core.logger.DEBUG);
		    lookup.kill();
		    lookup = null;
		}
	    }
	}
	
	////////////////////////////////////////////////////////////
	// DataObject implementation
	//
	// REDFLAG: double check 
	private CPAlgoData(DataObjectPending dop, NodeReference ref) {
	    this.ref = ref;
	    if (dop.getDataLength() > 0) {
		try {
		    DataInputStream din = dop.getDataInputStream();
		    contactProbability = din.readFloat();
		}
		catch (IOException e) {
		    Core.logger.log(this, "IOException reading CP!",
				    e, Core.logger.ERROR);
		}
		//wtf? Tavin, If this is ok there should be a comment explaining why.
		//
		// GJ, there were a few fine points I kind of glossed over..  but
		// it's not unreasonable to do this, the object just ends up being
		// reset to initial values from the application layer's point of view.
		//
		// Besides, if we get IOExceptions here there is something very wrong
		// with the data store that will probably bring down the whole node
		// anyway.
	    }
	    dop.resolve(this);
	}
	
	public final int getDataLength() {
	    // 1 float, 1 * 4    4
	    return 4;
	}
	
	public final void writeTo(DataOutputStream out) throws IOException {
	    out.writeFloat(contactProbability);
	}
	
	////////////////////////////////////////////////////////////
	// 9/12/2002 amphibian
	//
	// Removed all backoff code.
	//
	
	// Connection limits are inexact because of the time lag
	// between connection attempts and the CP update.  One
	// could achieve more exact enforcement at the cost of
	// exposing connection management to the routing code.
	// It doesn't look like that's necessary.  There is
	// already a hack in the OCM that keeps too many 
	// connections to the same ref from blocking at
	// the socket layer.
	//
	
	// Weight the CP based on the time since the last connection
	// attempt and the number of failures.  
	final float effectiveCP(long nowMs) {
	    return contactProbability();
	}
	
	final float contactProbability() {
	    if(contactProbability < minCP) {
		contactProbability = minCP;
		return minCP;
	    }
	    return contactProbability;
	}
	
	// returns true if the node should be dereferenced.
	final boolean decreaseContactProbability(boolean weak) {
	    contactProbability = 
		(float)(weak ? ((96.0 * contactProbability) / 100.0) :
			(((4.0 * contactProbability) /* + 0.0 */) / 5.0));
	    // Decrease by 4% for weak, 20% for strong
	    if(contactProbability < minCP) contactProbability = minCP;
	    consecutiveFailures++;
	    // When consecutiveFailures reaches some value, start an ARK lookup
	    if(consecutiveFailures >= consecutiveFailuresLookupARK &&
	       lastSuccessfulContact + minDowntimeARKLookup <
	       System.currentTimeMillis()) {
		synchronized(lookupLock) {
		    if(node == null) throw new IllegalStateException
					 ("creating LookupARK with null node");
		    try {
			if(lookup == null && getARKURI() != null) {
			    synchronized(lookupThreadsLock) {
				if(lookupThreads < maxLookupThreads)
				    lookup = new LookupARK(getARKURI());
			    }
			    // FIXME: Kill the LookupARK whose ref is least recently used
			}
		    } catch (KeyException e) {
			Core.logger.log(this, "Cannot start ARK lookup: "+e, e,
					Logger.ERROR);
		    }
		}
	    }
	    lastRetryMs = System.currentTimeMillis();
	    return false;
	}
	
	final void increaseContactProbability(boolean weak) {
	    if(justFetchedARK) {
		node.diagnostics.occurrenceCounting("successLookupARK",1);
		justFetchedARK = false;
	    }
	    if(contactProbability < minCP) contactProbability = minCP;
	    consecutiveFailures = 0;
	    float f = (float)
		(weak ? (((19 * contactProbability) + 1.0) / 20.0) :
		 (((3.0 * contactProbability) + 1.0) / 4.0));
	    // Move closer to 1.0 by 5% for weak
	    // 25% for strong
	    if(f>1) f = 1;
	    contactProbability = f;
	    lastRetryMs = lastSuccessfulContact = System.currentTimeMillis();
	    synchronized(lookupLock) {
		if(lookup != null) {
		    lookup.kill();
		    lookup = null;
		}
	    }
	}
	
	final boolean fetchingARK() {
	    return lookup != null;
	}
	
	protected final FreenetURI getARKURI() throws KeyException {
	    return ref.getARKURI(ref.revision()+1);
	}

	protected final FreenetURI getARKURI(long version) throws KeyException {
	    return ref.getARKURI(version);
	}

	protected class LookupARK extends AutoBackoffNodeRequester {
	    long version;
	    long firstVersion;
	    long lastVersion;

	    public LookupARK(FreenetURI uri) throws KeyException {
		super(new InternalClient(node), uri, false, 
		      new ArrayBucket(), node.maxHopsToLive);
		version = ref.revision()+1;
		lastVersion = firstVersion = version;
		sleepTimeMultiplier = 1.0F; // deal with it manually
		// be a bit more aggressive than the others
		if(Core.logger.shouldLog(Core.logger.DEBUG))
		    Core.logger.log(this, "Scheduling ARK fetch for "+getURI(),
				    Core.logger.DEBUG);
		new Checkpoint(this).schedule(node);
		node.diagnostics.occurrenceCounting("startedLookupARK",1);
		synchronized(lookupThreadsLock) {
		    lookupThreads++;
		    if(Core.logger.shouldLog(Logger.DEBUG))
			Core.logger.log(this, "Added new ARK lookup - now up to "+
					lookupThreads, Logger.DEBUG);
		}
	    }
	    
	    public String getCheckpointName() {
		return "Fetching ARK "+getURI();
	    }
	    
	    public FreenetURI getURI() {
		FreenetURI uri;
		try {
		    uri = getARKURI(version);
		} catch (KeyException e) {
		    Core.logger.log(this, "KeyException getting URI: "+e,
				    e, Logger.ERROR);
		    finished = true;
		    return null;
		}
		return uri;
	    }
	    
	    protected Bucket getBucket() {
		return bucket;
	    }
	    
	    protected void onFinish() {
		synchronized(lookupThreadsLock) {
		    if(Core.logger.shouldLog(Logger.DEBUG))
			Core.logger.log(this, "Terminated ARK lookup - now up to "+
					lookupThreads, Logger.DEBUG);
		    lookupThreads--;
		}
	    }
	    
	    public boolean internalRun() {
		node.diagnostics.occurrenceCounting("lookupARKattempts",1);
		return super.internalRun();
	    }
	    
	    protected boolean success() {
		String s = ((ArrayBucket)bucket).toString();
		Core.logger.log(this, "Fetched ARK "+getURI()+": "+s,
				Core.logger.MINOR);
		// REDFLAG/FIXME: CHECK LOCKING HERE

                // hmmm... Dodgy. Need cast because
                // Bucket.getInputStream() throws IOException but 
                // ArrayBucket.getInputStream() doesn't. -gj
                
		InputStream in = ((ArrayBucket)bucket).getInputStream();
		FieldSet fs;
		node.diagnostics.occurrenceCounting("fetchedLookupARK",1);
		try {
		    fs = new FieldSet(new ReadInputStream(in));
		} catch (IOException e) {
		    Core.logger.log(this, 
				    "IOException reading FieldSet from ARK! ("+
				    getURI()+")", 
				    e, Core.logger.ERROR);
		    uri=getURI();
		    version++;
		    sleepTime = initialSleepTime; // probably not their fault
		    if(firstVersion < version) firstVersion = version;
		    if(lastVersion < version) lastVersion = version;
		    return false; // ?
		}
		NodeReference newRef;
		try {
		    newRef = new NodeReference(fs, true);
		} catch (BadReferenceException e) {
		    // Might just be old-format; regardless, lets try the next one
		    version++;
		    
		    sleepTime = sleepTime/4;
		    if(sleepTime <= initialSleepTime) 
			sleepTime = initialSleepTime;
		    sleepTime *= 1.1; 
		    // so a series of bad references will gradually slow down
		    
		    if(firstVersion < version) firstVersion = version;
		    if(lastVersion < version) lastVersion = version;

		    Core.logger.log(this, 
				    "Invalid ARK - fetching next version: "
				    +getURI(), e, Core.logger.MINOR);
		    uri = getURI();
		    return false;
		}
		if(!newRef.getIdentity().equals(ref.getIdentity())) {
		    version++;
		    
		    sleepTime = sleepTime/4;
		    if(sleepTime <= initialSleepTime) 
			sleepTime = initialSleepTime;
		    sleepTime *= 1.5; // bad identity worse than bad reference
		    // so a series of bad references will gradually slow down
		    
		    if(firstVersion < version) firstVersion = version;
		    if(lastVersion < version) lastVersion = version;

		    Core.logger.log(this,
				    "Invalid ARK (tried to change identity) - fetching next version: " + getURI(), Core.logger.MINOR);
		    uri = getURI();
		    return false;
		}
		synchronized(CPAlgoRoutingTable.this) {
		    routingStore.putNode(newRef);
		    // Don't touch the CP. Nodes should not persist just by inserting lots of ARKs.
		}
		node.diagnostics.occurrenceCounting("validLookupARK", 1);
		justFetchedARK = true;
		synchronized(lookupLock) {
		    lookup = null; // we're done
		}
		finished = true;
		return true;
	    }
	    
	    int seq = 0;
	    
	    protected boolean failedInvalidData() {
		Core.logger.log(this, "failed: invalid data: "+this, Logger.DEBUG);
		version++;
		firstVersion = version;
		if(lastVersion < firstVersion) lastVersion = firstVersion;
		Core.logger.log(this, "failed: invalid data: "+this+": retrying with higher version number", Logger.DEBUG);
		return false; // retry
	    }
	    
	    protected boolean failure() {
		boolean shouldLog = Core.logger.shouldLog(Core.logger.DEBUG);
		if(shouldLog)
		    Core.logger.log(this, "LookupARK Failure: version="+version+
				    ", firstVersion="+firstVersion+
				    ", lastVersion="+lastVersion+" for "+
				    getURI(), Core.logger.DEBUG);
		if(dnf != null) {
		    seq++; // used by both branches
		    if (version == lastVersion) {
			if(shouldLog)
			    Core.logger.log(this, "At last version",
					    Core.logger.DEBUG);
			// exponential backoff, double up on every full pass
			sleepTime *= 2;
			
			if((seq & 1) == 0) {
			    lastVersion++;
			    if(shouldLog)
				Core.logger.log(this, 
						"Incrementing lastVersion to "+
						lastVersion+" for "+
						getURI(), Core.logger.DEBUG);
			}
			version = firstVersion;
		    } else {
			if((seq & 1) == 0) {
			    version++;
			    if(shouldLog)
				Core.logger.log(this, "Incrementing version to "+
						version+" for "+
						getURI(), Core.logger.DEBUG);
			}
			sleepTime *= 1.5;
		    }
		} else {
		    if(shouldLog)
			Core.logger.log(this, "ARK failure not a DNF for "+
					getURI(), Core.logger.DEBUG);
		    sleepTime *= 2;
		    // Standard response to RNFs: double the sleepTime
		}
		if(shouldLog)
		    Core.logger.log(this, "Retrying ARK fetch with version "+
				    version+" for "+getURI()+" (delay "+
				    sleepTime+")", Core.logger.MINOR);
		return false; // failed, retry exponentially
	    }
	    
	}
    }
    
}







