package freenet.node.ds;

import freenet.Key;
import freenet.KeyException;
import freenet.node.Node;
import freenet.node.ds.KeyInputStream;
import freenet.fs.dir.*;
import freenet.fs.acct.Fragment;
import freenet.support.Fields;
import freenet.support.io.WriteOutputStream;
import javax.servlet.*;
import javax.servlet.http.*;
import java.io.*;
import java.util.Enumeration;
import java.util.Date;
import java.text.DateFormat;

/**
 * Debug console for the FSDataStore.
 * @author tavin
 */
public class DSConsole extends HttpServlet {
    
    Node node;
    
    public final void init() {
        this.node = (Node) getServletContext().getAttribute("freenet.node.Node");
    }
    
    public void doGet(HttpServletRequest req, HttpServletResponse resp)
                                                    throws IOException {
        String key = req.getPathInfo();
        if (key != null) {
            // discard the leading slash
	    try {
		Key k = Key.readKey(key.substring(1, key.length()));
		dumpKey(k, req, resp);
	    } catch (KeyException e) {
		resp.sendError(resp.SC_INTERNAL_SERVER_ERROR, "Invalid key");
	    }
        } else {
            listKeys(req, resp);
        }
    }

    public void doPost(HttpServletRequest req, HttpServletResponse resp)
                                                    throws IOException {
        if (req.getParameter("reset") != null) {
            FSDataStore ds = (FSDataStore) node.ds;
            synchronized (ds.dir.semaphore()) {
                Enumeration keys = ds.dir.keys(true);
                while (keys.hasMoreElements()) {
                    FileNumber fn = (FileNumber) keys.nextElement();
                    ds.dir.delete(fn, false);
                }
            }
        } else if (req.getParameter("delete") != null) {
            String[] keys = req.getParameterValues("keys");
            if (keys != null) {
                for (int i=0; i<keys.length; ++i) {
		    try {
			Key k = Key.readKey(keys[i]);
			node.ds.remove(k, false);
		    } catch (KeyException e) {
			// Just ignore it
		    }
		}
            }
        }
        listKeys(req, resp);
    }

    private void dumpKey(Key key, HttpServletRequest req, HttpServletResponse resp)
                                                                throws IOException {
        resp.setContentType("text/plain");
        KeyInputStream kin = node.ds.getData(key);
        if (kin == null) {
            resp.sendError(resp.SC_NOT_FOUND, "Key gone from store");
            return;
        }
        try {
            ServletOutputStream out = resp.getOutputStream();
            kin.getStorables().writeFields(new WriteOutputStream(out));
            resp.flushBuffer();
        }
        finally {
            kin.close();
        }
    }

    private void listKeys(HttpServletRequest req, HttpServletResponse resp)
                                                    throws IOException {
    
        resp.addHeader("Cache-control", "no-cache");
        resp.addHeader("Pragma", "no-cache");
        resp.addHeader("Expires", "Thu, 01 Jan 1970 00:00:00 GMT");
        resp.setContentType("text/html");
        
        PrintWriter out = resp.getWriter();
        
        out.println("<!DOCTYPE html");
        out.println("  PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\"");
        out.println("  \"DTD/xhtml1-frameset.dtd\">");
        out.println("<html>");
        out.println("<head><title>Data Store Console: " + (new Date())
                                                        + "</title></head>");
        out.println("<body>");

        out.println("<form method=\"post\">");
        
        FSDataStore ds = (FSDataStore) node.ds;
        synchronized (ds.dir.semaphore()) {
            out.println("<p><code>Free space: " + ds.dir.available()
                                                + "</code></p>");
            listCommittedKeys(ds, req, resp);
            listLRUKeys(ds, req, resp);
        }
        
        out.println("<p>");
        out.println("<input type=\"submit\" name=\"delete\" value=\"delete keys\" />");
        out.println("&nbsp; &nbsp; &nbsp;");
        out.println("<input type=\"submit\" name=\"reset\" value=\"reset datastore\" />");
        out.println("</p>");
        out.println("</form>");
        
        out.println("</body>");
        out.println("</html>");

        resp.flushBuffer();
    }

    private static void printKeyLink(byte[] val,
                                     HttpServletRequest req,
                                     HttpServletResponse resp) throws IOException {
        String hex = Fields.bytesToHex(val);
        String uri = req.getContextPath() + req.getServletPath() + '/' + hex;
        PrintWriter out = resp.getWriter();
        out.println("<td><input type=\"checkbox\" name=\"keys\" value=\""
                    + hex + "\" /></td>");
        out.println("<td><code><a href=\"" + uri + "\">"
                    + hex + "</a></code></td>");
    }

    private static void listCommittedKeys(FSDataStore ds,
                                          HttpServletRequest req,
                                          HttpServletResponse resp) throws IOException {
        PrintWriter out = resp.getWriter();
        out.println("<table>");
        out.print("<tr style=\"background-color: grey\"><th colspan=\"4\">");
        out.print("Committed keys");
        out.println("</th></tr>");
        Enumeration keys = ds.dir.keys(true);
        while (keys.hasMoreElements()) {
            FileNumber fn = (FileNumber) keys.nextElement();
            Buffer buffer = ds.dir.fetch(fn);
            try {
                out.println("<tr>");
                printKeyLink(fn.getByteArray(), req, resp);
                out.println("<td><code>@</code></td>");
                out.println("<td><code>");
                out.println(Fragment.rangeList(buffer.ticket().ranges));
                out.println("</code></td>");
                out.println("</tr>");
            }
            finally {
                buffer.release();
            }
        }
        out.println("</table>");
    }
    
    private static void listLRUKeys(FSDataStore ds,
                                    HttpServletRequest req,
                                    HttpServletResponse resp) throws IOException {
        PrintWriter out = resp.getWriter();
        DateFormat df = DateFormat.getDateTimeInstance();
        out.println("<table>");
        out.print("<tr style=\"background-color: grey\"><th colspan=\"4\">");
        out.print("LRU keys (oldest last)");
        out.println("</th></tr>");
        Enumeration keys = ds.dir.lruKeys(false);
        while (keys.hasMoreElements()) {
            FileNumber fn = (FileNumber) keys.nextElement();
            Buffer buffer = ds.dir.fetch(fn);
            try {
                out.println("<tr>");
                printKeyLink(fn.getByteArray(), req, resp);
                out.println("<td><code>@</code></td>");
                out.println("<td><code>");
                out.println(df.format(new Date(buffer.ticket().timestamp)));
                out.println("</code></td>");
                out.println("</tr>");
            }
            finally {
                buffer.release();
            }
        }
        out.println("</table>");
    }
}



