package freenet.node;

import freenet.*;
import freenet.client.FreenetURI;
import freenet.client.Base64;
import freenet.support.*;
import freenet.crypt.*;
import freenet.session.LinkManager;
import freenet.transport.VoidAddress;
import freenet.keys.SVK;
import freenet.client.ClientSSK;
import java.util.Enumeration;
import java.math.BigInteger;
import java.net.MalformedURLException;

/**
 * References contains names from which Address objects can be resolved.
 * How this is done is transport dependant.
 *
 * It is expected that at some later time References will be more complex
 * structures contain the names for several different transports and internal
 * lookup and authorization information.
 *
 * @author oskar
 */

public class NodeReference {

    private static final String[] stringSignature = {"signature"};

    public final String[] physical;
    private long[] sessions;
    private long[] presentations;

    private final Identity identity;    // node's public key
    private boolean hasARK = false;     // whether we have an ARK
    private long ARKrevision;           // ARK revision number
    private byte[] ARKcrypt;            // ARK data encryption key
    private DSASignature signature;
    private String version;

    // kill me quickly
    //private boolean signatureHack = false;


    /**
     * Makes a deep copy of a NodeReference.
     */
    public NodeReference(NodeReference nr) {
        
        physical = new String[nr.physical.length];
        System.arraycopy(nr.physical, 0, physical, 0, physical.length);
        sessions = new long[nr.sessions.length];
        System.arraycopy(nr.sessions, 0, sessions, 0, sessions.length);
        presentations = new long[nr.presentations.length];
        System.arraycopy(nr.presentations, 0, presentations, 0, presentations.length);
        
        identity = nr.identity;
        hasARK = nr.hasARK;
        ARKrevision = nr.ARKrevision;
        ARKcrypt = nr.ARKcrypt;
        signature = nr.signature;
        version = nr.version;
    }

    /**
     * Creates a new nodereference from its FieldSet, verifying the
     * signature.
     * @param ref A FieldSet containing the full Node Reference information.
     * @exception BadReferenceException - if the data is malformed.
     */
    public NodeReference(FieldSet ref) throws BadReferenceException {
        this(ref, true, null);
    }

    /**
     * Creates a new nodereference from its FieldSet.
     * @param ref A FieldSet containing the full Node Reference information.
     * @param verify  true mandates the existance of valid signature.
     * @exception BadReferenceException - if the data is malformed.
     */
    public NodeReference(FieldSet ref, boolean verify) throws BadReferenceException {
        this(ref, true, null);

    }

    /**
     * Creates a new NodeReference from us, updating the ARKVersion and
     * the physical addresses and recalculating the signature
     */
    NodeReference newVersion(DSAAuthentity auth, String[] physical, 
			     long ARKversion) {
	NodeReference n = new NodeReference(identity, physical, 
					    sessions, presentations, version,
					    ARKversion, ARKcrypt);
	n.addSignature(auth);
	return n;
    }

    /**
     * Creates a new nodereference from its FieldSet, adding an externally
     * obtained identity.
     * @param ref A FieldSet containing the full Node Reference information.
     * @param verify  true mandates the existance of valid signature.
     * @param ident  The identity to assign the fieldset to. The method
     *               will look for a field called "identityFP" in the 
     *               and match that against the fingerprint of this identity.
     * @exception BadReferenceException - if the data is malformed.
     */
    public NodeReference(FieldSet ref, boolean verify, Identity ident) 
        throws BadReferenceException {

        if (!ref.isSet("identity") && ident != null) {
            // It's sort of ugly, but we'll actually insert the identity
            // into the fieldset.
            String s = ref.get("identityFP");
            if (s == null || !s.equals(ident.fingerprintToString()))
                throw new BadReferenceException("Provided identity did not " +
                                                "match fingerprint.");
            ref.remove("identityFP");
            ref.add("identity", ident.getFieldSet());
        }
        // Read physical addresses

        FieldSet phy = ref.getSet("physical");
        //if (phy == null || phy.isEmpty())
        //    throw new BadReferenceException("Malformed ref: " + 
        //                                       "no physical addresses.");

        if (phy != null && !phy.isEmpty()) {
            String trans, addr;
            int i = 0;
            physical = new String[phy.size() * 2];
            for (Enumeration e = phy.keys() ; e.hasMoreElements() ; i += 2) {
                trans = (String) e.nextElement();
                addr = phy.get(trans);
                if (addr == null) { // safety, it could be a FieldSet, which is bad
                    throw new BadReferenceException("Malformed ref: " + 
                                                    "bad physical address.");
                }
                physical[i] = trans;
                physical[i + 1] = addr;
            }
        }
        else physical = new String[0];

        // read Contact info

        try {
            String ss = ref.get("sessions");
            if (ss == null)
                throw new NumberFormatException("No session field");
            sessions = Fields.numberList(ss);


            String ps = ref.get("presentations");

            if (ps == null)
                throw new NumberFormatException("No presentations field");

            presentations = Fields.numberList(ps);

            //FieldSet ident = ref.getSet("identity");
            // temporary compatibility measure
            //if (ident == null) {
            //    String pks = ref.get("identity");
            //    if (pks == null)
            //        throw new NumberFormatException("No node identity");
            //    identity = new DSAIdentity(ref.get("identityGroup"), pks);
            //}
            //else {
            //    identity = new DSAIdentity(ident);
            //}

            identity = new DSAIdentity(ref.getSet("identity"));
            
            version = ref.get("version");

            
        } catch (NumberFormatException e) {
            throw new BadReferenceException("Malformed ref: " + 
                                               e.getMessage());
        }
        
        // read ARK info

        FieldSet ARK = ref.getSet("ARK");
        if (ARK != null) {
            try {
                if (!ARK.isString("revision"))
                    throw new NumberFormatException();
                ARKrevision = Fields.stringToLong(ARK.get("revision"));
                String crypts = ARK.get("encryption");
                if (crypts != null ) {
                    ARKcrypt = new byte[crypts.length() / 2];
                    Fields.hexToBytes(crypts, ARKcrypt, 0);
                }
                hasARK = true;
            } catch (NumberFormatException e) {
                Core.logger.log(this, 
                                "Malformed ARK entry in Reference ignored",
                                Logger.MINOR);
            }
        }

        if (verify) {
            // last but not least, read and verify signature
            
            String signS = ref.get("signature");
            if (signS == null)
                throw 
                    new BadReferenceException("NodeReference must be signed");
            try {
                signature = new DSASignature(signS);
                Digest d = SHA1.getInstance();
                ref.hashUpdate(d, stringSignature);
                //try {
                if (!identity.verify(signature, 
                                     new BigInteger(1, d.digest()))) {
                    throw new BadReferenceException("NodeReference self " +
                                                    "signature check failed.");
                }
                //}
                // oh what a brutal hack .. but it's temporary
                //catch (BadReferenceException e) {
                //    ref.put("identity", ((DSAPublicKey) identity).writeAsField());
                //    // they were using group C anyway..
                //    Digest d2 = new SHA1();
                //    ref.hashUpdate(d2, stringSignature);
                //    if (!identity.verify(signature, new BigInteger(1, d2.digest())))
                //        throw e;
                //    signatureHack = true;
                //}
            } catch (NumberFormatException e) {
                throw new BadReferenceException("Signature field not correct");
            }
        }

        if (version != null)
            Version.seenVersion(version);
    }

    /**
     * Constructs a new NodeReference to this address, with ARK as read
     * from the URI.
     * @param addr  An Address object describing a physical address and
     *              identity of the node reference.
     * @param ARK   An ARK URI. The subspace value is ignored, it is assumed
     *              to be the fingerprint of the identity. May be null.
     */
    public NodeReference(Peer peer, String version, FreenetURI ARK) {
        this(
            peer.getIdentity(),
            new Address[] { peer.getAddress() },
            new long[]    { peer.getLinkManager().designatorNum() },
            new long[]    { peer.getPresentation().designatorNum() },
            version,
            ARK
        );

        if (version != null)
            Version.seenVersion(version);
    }

    public NodeReference( Identity identity, Address[] addr,
			  long[] sessions, long[] presentations,
			  String version, FreenetURI ARK ) {
	this(identity, parsePhysical(addr), sessions, 
		      presentations, version);
        if (ARK != null) {
            try {
                if (ARK.getGuessableKey() == null)
                    throw new NumberFormatException();
                ARKrevision = Fields.stringToLong(ARK.getGuessableKey());
                ARKcrypt = ARK.getCryptoKey(); // may be null	
                hasARK = true;
            }
            catch (NumberFormatException e) {
                Core.logger.log(this, 
                                "Malformed ARK entry in Reference ignored",
                                Logger.MINOR);
            }
        }

    }

    protected static String[] parsePhysical( Address[] addr ) {
	String[] physical = new String[2*addr.length];
        for (int i=0; i<addr.length; ++i) {
            physical[2*i]   = addr[i].transport().getName();
            physical[2*i+1] = addr[i].getValString();
        }
	return physical;
    }
    
    protected NodeReference( Identity identity, String[] physical,
			     long[] sessions, long[] presentations,
			     String version ) {

        this.identity      = identity;
        this.sessions      = sessions;
        this.presentations = presentations;
        this.version       = version;
	this.physical      = physical;
        
        if (version != null)
            Version.seenVersion(version);
    }
    
    protected NodeReference ( Identity identity, String[] addr,
			      long[] sessions, long[] presentations,
			      String version, long ARKrevision,
			      byte[] ARKcrypt ) {

	this(identity, addr, sessions, presentations, version);
	hasARK = true;
	this.ARKrevision = ARKrevision;
	this.ARKcrypt = ARKcrypt;
    }

    public NodeReference ( Identity identity, Address[] addr,
			   long[] sessions, long[] presentations,
			   String version, long ARKrevision,
			   byte[] ARKcrypt ) {
	this(identity, parsePhysical(addr), sessions, presentations, version);
	hasARK = true;
	this.ARKrevision = ARKrevision;
	this.ARKcrypt = ARKcrypt;
    }
    
    /**
     * Returns true if this NodeReference and the given indicate that they can
     * talk to one another. That is they share at least of each of physical,
     * session, and presentation protocols.
     */
    public boolean intersects(NodeReference nr) {
        boolean found = false;
        // O(n^2) - but n is quite limited...
        for (int i = 0 ; i < physical.length && !found; i++) {
            for (int j = 0 ; j < nr.physical.length && !found; j++) {
                if (physical[i].equals(nr.physical[j]))
                    found = true;
            }
        }
        if (!found)
            return false;
        else
            found = false;

        for (int i = 0 ; i < sessions.length && !found; i++) {
            for (int j = 0 ; j < nr.sessions.length && !found; j++) {
                if (sessions[i] == nr.sessions[j])
                    found = true;
            }
        }
        if (!found)
            return false;
        else
            found = false;

        for (int i = 0 ; i < presentations.length && !found; i++) {
            for (int j = 0 ; j < nr.presentations.length && !found; j++) {
                if (presentations[i] == nr.presentations[j])
                    found = true;
            }
        }

        return found;
    }


    /**
     * Returns an address pointing at this reference.
     * @param   t The transport type of the address.
     * @return  The address if one with the correct transport
     *           is found, otherwise null.
     * @exception  If the address stored for this transport is broken.
     */
    public Address getAddress(Transport t)
        throws BadAddressException {
        for (int i = 0 ; i < physical.length ; i += 2) {
            if (t.getName().equals(physical[i])) {
		long startTime = System.currentTimeMillis();
		Address addr = t.getAddress(physical[i+1]);
		long endTime = System.currentTimeMillis();
		long time = endTime - startTime;
		Core.logger.log(this, "t.getAddress("+physical[i+1]+") took "+
				time, 
				time>1000 ? Core.logger.NORMAL : Core.logger.DEBUG);
		return addr;
            }
        }
        return null;
    }
    
    // Caveat: tcpAddress's are already-looked-up. Don't use this to update
    // to a new node _NAME_.
    static public String[] setPhysicalAddress(String[] physical, Transport t, 
					      Address a) {
	for (int i = 0 ; i < physical.length ; i += 2) {
	    if (t.getName().equals(physical[i])) {
		physical[i+1] = a.toString();
		return physical;
	    }
	}
	String[] newPhys = new String[physical.length+2];
	System.arraycopy(physical, 0, newPhys, 0, physical.length);
	newPhys[physical.length] = t.getName();
	newPhys[physical.length+1] = a.toString();
	return newPhys;
    }
    
    /**
     * Create public URI for ARK
     * @param version the sequence number to use
     */
    public FreenetURI getARKURI(long version) throws KeyException {
	DSAPublicKey pk = (DSAPublicKey)getIdentity();
	if(pk == null) return null;
	byte[] ckey = cryptoKey();
	if(ckey == null) return null;
	SVK svk = new SVK(pk, Fields.longToHex(version), SVK.SVK_LOG2_MAXSIZE);
	ClientSSK ssk = 
	    new ClientSSK(svk, ckey, Fields.longToHex(version));
	
	return ssk.getURI();
    }
    
    /**
     * Checks that all addresses for transports that we support are 
     * correct.
     */
    public boolean checkAddresses(TransportHandler th) {
        for (int i = 0 ; i < physical.length ; i += 2) {
            if (!th.checkAddress(physical[i], physical[i+1]))
                return false;
        }
        return true;
    }

    /**
     * Returns true if the node supports the presentation protocol
     * designated by the object.
     */
    public boolean supports(Presentation p) {
        int n = p.designatorNum();

        for (int i = 0 ; i < presentations.length; i++) {
            if (presentations[i] == n) {
                return true;
            }
        }

        return false;
    }

    /**
     * Returns true if the node supports the session protocol
     * designated by the object.
     */
    public boolean supports(LinkManager s) {
        int n = s.designatorNum();

        for (int i = 0 ; i < sessions.length; i++) {
            if (sessions[i] == n) {
                return true;
            }
        }

        return false;
    }

    /**
     * Returns the most prefered of the addresses suppported by both
     * the TransportHandler provided and this NodeReference.
     */
    public Peer getPeer(TransportHandler th, SessionHandler sh, 
                        PresentationHandler ph) {
        Presentation p = null;
        for (Enumeration e = ph.getPresentations() ; e.hasMoreElements() ;) {
            p = (Presentation) e.nextElement();
            if (supports(p)) {
                break;
            }
        }
        if (p == null) {
            Core.logger.log(this,
                "Failed to find supported presentation for peer.",
                Logger.DEBUG);
            return null;
        }
        LinkManager lm = null;
        for (Enumeration e = sh.getLinkManagers() ; e.hasMoreElements() ; ) {
            lm = (LinkManager) e.nextElement();
            if (supports(lm)) {
                break;
            }
        }
        if (lm == null) {
            Core.logger.log(this,
                "Failed to find supported session for peer.",
                Logger.DEBUG);
            return null;
        }

        Address r = null;
        for (Enumeration e = th.getTransports() ; 
             e.hasMoreElements() && r == null;) {
            try {
                r = getAddress((Transport) e.nextElement());
            } catch (BadAddressException bae) {
		Core.logger.log(this, "BadAddressException in getPeer",
				Core.logger.DEBUG);
            }
        }
	
	// Expanded out for easy augmentation
        if (r == null)
            r = new VoidAddress();
	
	Peer peer = new Peer(identity, r, lm, p);
        return peer;
    }
    
    /**
     * Returns the public key identity of the node reference.
     * @return The node referenced public key.
     */
    public final Identity getIdentity() {
        // maybe I should return a copy. If identity starts getting nuked
        // that will be the problem...
        return identity;
    }

    /**
     * Returns the version string of the NodeReference.
     * @return The version string of the node referenced.
     */
    public final String getVersion() {
        return version;
    }

    /**
     * Returns the ARK (Address resolution key) for this node reference. 
     * Null if ARK is missing.
     */
    public final FreenetURI getARK() {
        return !hasARK ? null : 
            new FreenetURI("SSK",
                           Fields.longToString(ARKrevision),
                           identity.fingerprint(), 
                           ARKcrypt);
    }


    /**
     * Returns true if the other NodeReference is to the same node, but
     * with a new ARK revision value.
     */
    public final boolean supersedes(NodeReference nr) {
	if(!identity.equals(nr.identity)) return false;
	if(ARKrevision < nr.ARKrevision) return false;
	if(noPhysical()) return false;
	if(ARKrevision > nr.ARKrevision) return true;
	if(nr.noPhysical()) return true;
	if(!version.equals(nr.version)) return true;
	return false;
    }
    
    /**
     * Returns true if we have no physical addresses
     */
    public final boolean noPhysical() {
	return physical == null || physical.length == 0;
    }
    
    /**
     * Returns the ARK revision value for this NodeReference
     */
    public final long revision() {
        return ARKrevision;
    }

    public final byte[] cryptoKey() {
	return ARKcrypt;
    }
    
    /**
     * Returns a FieldSet represenation of this NodeReference for
     * serialization.	
     */
    public FieldSet getFieldSet() {
        return getFieldSet(true);
    }

    /**
     * Returns a FieldSet representation of this NodeReference for
     * serialization, optionally omitting the identity.
     * @param addIdentity Whether to add the identity public key to the 
     *                    fieldset. If not, a field called "identityFP" will
     *                    contain the identity fingerprint instead. Note that
     *                    WE WILL KILL BOB IF YOU DO NOT GIVE US ONE MILLION
     *                    DOLLARS IN UNMARKED BILLS the signature field will
     *                    still contain the signature of the entire 
     *                    NodeReference - that is the fieldset that contains
     *                    the identity rather than the one generated.
     */
    public FieldSet getFieldSet(boolean addIdentity) {
        FieldSet fs = new FieldSet();

        if (addIdentity) {
            // add identity
            fs.put("identity", identity.getFieldSet());
        } else {
            fs.put("identityFP", identity.fingerprintToString());
        }

        // add physical addresses
        if (physical.length > 0) {
            FieldSet phys = new FieldSet();
            for (int i = 0 ; i < physical.length ; i += 2)
                phys.put(physical[i], physical[i+1]);
            fs.put("physical", phys);
        }

        fs.put("sessions", Fields.numberList(sessions));
        
        fs.put("presentations",Fields.numberList(presentations));

        if (version != null) fs.put("version", version);
        
        // add ARK info
        if (hasARK) {
            FieldSet ARK = new FieldSet();
            ARK.put("revision",Fields.longToString(ARKrevision));
            if (ARKcrypt != null) {
                ARK.put("encryption",Fields.bytesToHex(ARKcrypt, 0, 
                                                       ARKcrypt.length));
            }
            fs.put("ARK",ARK);
        }

        // add signature if we have it
        if (signature != null) fs.put("signature", signature.writeAsField());

        return fs;
    }

    
    
    private static final Digest ctx = SHA1.getInstance();
    
    /**
     * Signs the FieldSet with the provided authentity and adds the
     * signature value in the field "signature".
     * @param auth  The private key to sign with.
     */
    public void addSignature(DSAAuthentity auth) {

        byte[] b;
        FieldSet fs = getFieldSet();
        //fs.remove("signature");
        synchronized(ctx) {
            fs.hashUpdate(ctx, new String[] {"signature"});
            b = ctx.digest();
        }
        
        signature = (DSASignature) auth.sign(b);
    }        

    /**
     * Returns true if o is a NodeReference, has the same identity,
     * and the same revision as this.
     */
    public final boolean equals(Object o) {
        return o instanceof NodeReference && equals((NodeReference) o);
    }

    public final boolean equals(NodeReference nr) {
        return identity.equals(nr.identity) && ARKrevision == nr.ARKrevision;
    }

    public final int hashCode() {
        return identity.hashCode() ^ 
            (int) (ARKrevision * 1000000007); // large prime!
    }

    /**
     * @return  The first physical address as a string. They are often
     *          easier on the eye then the identity.
     */
    public String firstPhysicalToString() {
        if (physical.length < 2)
            return "void/void";
        else
            return physical[0] + "/" + physical[1];
    }
    
    public String toString() {
        //return getFieldSet().toString();
        StringBuffer sb = new StringBuffer(256);
        for (int i=0; i<physical.length; i+=2) {
            sb.append(physical[i]+'/'+physical[i+1]+", ");
        }
        sb.append("sessions="+Fields.numberList(sessions)+", ");
        sb.append("presentations="+Fields.numberList(presentations)+", ");
        sb.append("ID="+identity);
        return sb.toString();
    }

}




