package freenet.node;

import freenet.*;
import freenet.diagnostics.*;
import freenet.thread.*;
import freenet.interfaces.*;
import freenet.interfaces.servlet.*;
import freenet.client.events.*;
import freenet.client.FreenetURI;
import freenet.client.InsertURI;
import freenet.client.InternalClient;
import freenet.config.*;
import freenet.crypt.*;
import freenet.fs.*;
import freenet.fs.FileSystem; // java.io has one too
import freenet.fs.dir.*;
import freenet.support.*;
import freenet.support.io.*;
import freenet.node.ds.*;
import freenet.node.rt.*;
import freenet.node.states.maintenance.*;
import freenet.node.states.announcing.Announcing;
import freenet.message.*;
import freenet.message.client.*;
import freenet.message.client.FEC.*;
import freenet.client.FECTools;
import freenet.crypt.Global;
import freenet.presentation.*;
import freenet.session.*;
import freenet.transport.TCP;
import freenet.transport.tcpTransport;
import freenet.transport.tcpConnection;
import freenet.transport.tcpAddress;
import freenet.transport.ReadSelectorLoop;
import freenet.transport.WriteSelectorLoop;
import java.lang.reflect.Method;
import java.util.*;
import java.net.*;
import java.io.*;
import java.math.BigInteger;
import javax.servlet.Servlet;
import javax.servlet.http.HttpServlet;
import java.security.Security;
import freenet.client.AutoBackoffNodeRequester;

/**
 * This contains the execution code for starting a node. Moved from Node.java
 */

public class Main {

    public static WatchMe watchme = null;
    public static boolean publicNode = false;
    public static Node node = null; // everything else is static
    public static Params params = null;
    static byte[] cipherKey = null;
    static byte[] baseIV = null;
    static String oldTCPAddressAndPort = null; //Contains the address of the node in a a.b.c.d:port or hostname:port style
    static BlockCipher cipher;
    static Authentity privateKey;
    static long ARKversion;
    public /*FIXME*/static long initialARKversion = 0;
    static byte[] ARKcrypt = null;
    static public TransportHandler th = null;
    static NodeReference myRef = null;
    static InsertARK ARKinserter = null;
    static Object ARKInserterLock = new Object();
    static IPAddressDetector ipDetector = null;
    static File tempDir = null;
    static FileLoggerHook loggerHook = null;
    static LossyDirectory dsDir = null;
    static long storeSize = -1; // Node.storeSize is per store. Of course we only have one store now...
    public static String paramFile;
    static NodeConfigUpdater configUpdater = null;
    
    static public FileLoggerHook loggerHook() {
	return loggerHook;
    }
    
    static public long storeSize() {
	return storeSize;
    }
    
    public static NodeConfigUpdater getConfigUpdater() {
	return configUpdater;
    }
    
    public static final String[] defaultRCfiles =
        new String[] { "freenet.conf", "freenet.ini", ".freenetrc" };

    private static final Config switches = new Config();

    public static boolean doRequestTriageByDelay = true;

    static {
	// If this is moved further down the file, it may run after
	// InetAddress initialization. If you move it without checking
	// first that it does not cause InetAddress to ignore the setting
	// and cache everything forever, I will revoke your CVS perms!
	//    - amphibian (thanks to Pascal)
	//
	// Make java not cache DNS indefinitely. This is a big problem to
	// users with dyndns addresses... spoofability is also a problem,
	// but not relevant to freenet because the identity check will fail
	Security.setProperty("networkaddress.cache.ttl", "300");
	Security.setProperty("sun.net.inetaddr.ttl", "300");
	// Five minutes is probably not too much.
	// We don't do any of our own caching, and it's on the routing path,
	// so it is arguably necessary
	
        Node.class.toString(); // Force Node's static to run first
        
        String dc = defaultRCfiles[0];
        
        switches.addOption("help",       'h', 0, null, 10);
        switches.addOption("system",          0, null, 11);
        switches.addOption("version",    'v', 0, null, 12);
        switches.addOption("manual",          0, null, 13);
        switches.addOption("export",     'x', 1, "-",  20);
        switches.addOption("seed",       's', 1, "-",  21);
        switches.addOption("onTheFly",        1, "-",  30);
        switches.addOption("config",     'c', 1, dc,   40);
        switches.addOption("paramFile",  'p', 1, null, 41);
 
        switches.shortDesc ("help", "prints this help message");
        switches.shortDesc ("system", "prints JVM properties");
        switches.shortDesc ("version", "prints out version info");
        switches.shortDesc ("manual", "prints a manual in HTML");

        switches.argDesc   ("export", "<file>|-");
        switches.shortDesc ("export", "exports a signed NodeReference");

        switches.argDesc   ("seed", "<file>|-");
        switches.shortDesc ("seed", "seeds routing table with refs");

        switches.argDesc   ("onTheFly", "<file>|-");
        switches.shortDesc ("onTheFly", "writes a comma delimited list of config options that can be changed on the fly");

        switches.argDesc   ("config", "<file>");
        switches.shortDesc ("config", "generates or updates config file");

        switches.argDesc   ("paramFile", "<file>");
        switches.shortDesc ("paramFile", "path to a config file in a non-default location");
    }

    /**
     * Start Fred on his journey to world dominatrix^H^H^Hion
     */
    public static void main(String[] args) throws Throwable {
        try {
            // process command line
            Params sw = new Params(switches.getOptions());
            sw.readArgs(args);
            if (sw.getParam("help") != null) {
                usage();
                return;
            }
            if (sw.getParam("system") != null) {
                System.getProperties().list(System.out);
                return;
            }
            if (sw.getParam("version") != null) {
                version();
                return;
            }
            if (sw.getParam("manual") != null) {
                manual();
                return;
            }

            if (sw.getParam("onTheFly") != null) {
                String file = sw.getString("onTheFly");
                PrintStream out = System.out;
                if (!file.equals("-")) try {
                    out = new PrintStream(new BufferedOutputStream(new FileOutputStream(file)));
                } catch (FileNotFoundException e) {
                    System.err.println("Error creating file " + file);
                    return;
                }
                Method[] options = NodeConfigUpdater.ConfigOptions.class.getDeclaredMethods();
                out.print(options[0].getName());
                for (int x = 1 ; x < options.length ; x++)
                    out.print("," + options[x].getName());
                out.close();
                return;
            }

            args = sw.getArgs();  // remove switches recognized so far
	    
            params = new Params(Node.config.getOptions());
            
            // attempt to load config file
            paramFile = sw.getParam("paramFile");
            try {
                if (paramFile == null)
                    params.readParams(defaultRCfiles);
                else
                    params.readParams(paramFile);
            }
            catch (FileNotFoundException e) {
                if (sw.getParam("config") == null) {
                    if (paramFile == null) {
                        System.err.println("Couldn't find any of the following configuration files:");
                        System.err.println("    " + Fields.commaList(defaultRCfiles));
                        System.err.println("If you have just installed this node, use --config with no");
                        System.err.println("arguments to create a config file in the current directory,");
                        System.err.println("or --config <file> to create one at a specific location.");
                    }
                    else {
                        System.err.println("Couldn't find configuration file: "+paramFile);
                    }
                    return;
                }
            }
            params.readArgs(args);
            // I want config after readArgs, which must be after readParams
            // which mandates the hack in the catch block above.
            if (sw.getParam("config") != null) {
                try {
                    Setup set = new Setup(System.in, System.out, 
                                          new File(sw.getString("config")),
                                          false, params);
                    set.dumpConfig();
                } catch (IOException e) {
                    System.err.println("Error while creating config: " + e);
                }
                return;
            }


            // bail out if there were unrecognized switches
            // (we take no args)
            if (params.getNumArgs() > 0) {
                usage();
                return;
            }
	    
	    // note if we're a public node (and shouldn't run certain things)
	    // can be removed once we come up with a better method using some
	    // kind of user permissions
	    if (params.getBoolean("publicNode")) {
		Main.publicNode = true;
		Node.config.addOption("mainport.params.servlet.7.params.sfDisableWriteToDisk", 1, true, 4181);
	    }
	    
            // set up runtime logging
            int thresh = Logger.priorityOf(params.getString("logLevel"));
            Core.logger.setThreshold(thresh);

            String fname     = params.getString("logFile");
            String logFormat = params.getString("logFormat");
            String logDate   = params.getString("logDate");
	    
	    int logMaxLinesCached = params.getInt("logMaxLinesCached");
	    long logMaxBytesCached = params.getLong("logMaxBytesCached");
	    
	    FileLoggerHook.setMaxListLength(logMaxLinesCached);
	    FileLoggerHook.setMaxListBytes(logMaxBytesCached);
	    
            try {
		boolean dontCheck = (sw.getParam("export") != null);
                if (!fname.equalsIgnoreCase("NO")) {
                    loggerHook = new FileLoggerHook(fname, 
					   logFormat, 
					   logDate,
					   thresh,
					   dontCheck);
		} else {
		    loggerHook = new FileLoggerHook(System.err,
					   logFormat, 
					   logDate,
					   thresh);
		}
		Core.logger.addHook(loggerHook);
		Core.logStream = loggerHook.getStream();
	    }
	    catch (IOException e) {
                System.err.println("Opening log file failed!");
            }
            // Add a buffering hook for the last couple of entries as well
            Core.logger.addHook(new BufferLoggerHook(20));

            // Wait until after the log is set up so we can use it.
            removeObsoleteParameters(params);
            
            // If we are in watchme mode, initialize watchme and change the protocol
            // version to prevent spy nodes from polluting the real network
            if (params.getBoolean("watchme"))  {
                Main.watchme = new WatchMe();
                watchme.init(params.getInt("watchmeRetries"));
                Version.protocolVersion = Version.protocolVersion+"wm";
                freenet.session.FnpLink.AUTH_LAYER_VERSION=0x05;
            }
	    
	    //runMiscTests();

            // NOTE:  we are slowly migrating stuff related to setting up
            //        the runtime environment for the node into the Main
            //        class (and out of Core and Node)
            //
            //        this includes stuff like registering different
            //        message types, key types, diagnostics counters, etc.,
            //        and anything that involves the config params
            

            // set up static parameters in Core and Node
	    Core.logger.log(Main.class, "Starting Freenet ("+Version.nodeName+
			    ") "+Version.nodeVersion+" node, build #"+
			    Version.buildNumber+" on JVM "+
			    System.getProperty("java.vm.vendor")+":"+
			    System.getProperty("java.vm.name")+":"+
			    System.getProperty("java.vm.version"), 
			    Core.logger.MINOR);
	    
            Node.init(params);
	    
            // enable diagnostics
            //if (params.getBoolean("doDiagnostics"))
            initDiagnostics(params, Core.logger);
	    
            
            // load node secret keys file
            
            Core.logger.log(Main.class, "loading node keys: "+Node.nodeFile, Logger.NORMAL);

            cipher = (Node.storeCipherName == null
		      ? null
		      : Util.getCipherByName(Node.storeCipherName,
					     Node.storeCipherWidth));
            
	    loadNodeFile();

            // are we transient?
            boolean isTransient = params.getBoolean("transient");

            boolean strictAddresses = !params.getBoolean("localIsOK");

	    Node.maxThreads = params.getInt("maximumThreads");

            int negotiationLimit = Math.abs(params.getInt("maximumThreads")) 
		/ 4;
            if (negotiationLimit == 0) {
                negotiationLimit = Integer.MAX_VALUE;
            }

            // set up stuff for handling the node's transports,
            // sessions, and presentations
            th    = new TransportHandler();
            SessionHandler sh      = new SessionHandler();
            PresentationHandler ph = new PresentationHandler();
        
            th.register(new TCP(100, strictAddresses));
            sh.register(new FnpLinkManager(negotiationLimit), 100);
            ph.register(new FreenetProtocol(), 100);
	    
	    ipDetector = new IPAddressDetector();
	    
            // get the node's physical addresses
            Address[] addr;
            try {
            	String preferedAddress = null;
            	if(oldTCPAddressAndPort != null) //A new/transient node doesn't have an old address
            	{
					int colon = oldTCPAddressAndPort.indexOf(':');
					if (colon != -1) {
						preferedAddress =oldTCPAddressAndPort.substring(0, colon); 
					}
            	}
				addr = getAddresses(th, params, ipDetector.getAddress(preferedAddress));
            }
            catch (BadAddressException e) {
		Node.badAddress = true;
                if (!isTransient) {
		    String error = "There was an error determining this node's physical address(es).\nPlease make sure <ipAddress> and <listenPort> are correctly set.\nNote that you may put a host name in the <ipAddress> field if you have a dynamic IP and are using a dynamic DNS service.";
		    System.err.print(error);
		    Core.logger.log(Main.class, error, e, Core.logger.ERROR);
                    //throw e;
		    isTransient = true;
		    addr = new Address[0];
                }
                else addr = new Address[0];  // we're transient, screw it..
            }
	    
            // we have enough to generate the node reference now
            myRef = Node.makeNodeRef(privateKey, addr, sh, ph,
				     ARKversion, ARKcrypt);

	    Core.logger.log(Main.class, "Old address: "+oldTCPAddressAndPort, 
			    Core.logger.DEBUG);
	    tcpAddress a = getTcpAddress();
	    Core.logger.log(Main.class, 
			    "New address: "+((a==null)?"(null)":a.toString()),
			    Core.logger.DEBUG);
	    
            if (a != null &&
		!(a.toString().equals(oldTCPAddressAndPort))) {
		Core.logger.log(Main.class, 
				"Address has changed since last run",
				Core.logger.MINOR);
		newTcpAddress(a, th.get("tcp"), true);
	    }
	    
            // --export
            // (no need to load up the full node)
            if (sw.getParam("export") != null) {
                writeFieldSet(myRef.getFieldSet(),
                              sw.getString("export"));
		loggerHook.close();
                return;
            }
	    
            
            // load up the FS
            
	    boolean firstTime = true;

            Core.logger.log(Main.class, "starting filesystem", Logger.NORMAL);
	    File[] storeFiles = new File[Node.storeFile.length];
	    for (int i=0; i<storeFiles.length; ++i) {
		storeFiles[i] = new File(Node.storeFile[i]);
		if(storeFiles[i].exists()) firstTime = false;
	    }
	    Directory dir = null;
	    storeSize = Node.storeSize;

            if ((Node.routingDir == null || Node.routingDir.toString().equals(""))
		&& storeFiles[0] != null && !storeFiles[0].equals(""))
                Node.routingDir = storeFiles[0].
		    getAbsoluteFile().getParentFile();
	    
	    Node.storeType = params.getString("storeType");
	    if(Node.storeType.equals("convert")) {
		if(storeFiles[0].isDirectory()) {
		    Node.storeType = "native";
		}
		storeSize *= storeFiles.length;
	    }
	    if (!(Node.storeType.equals("monolithic") || 
		  Node.storeType.equals("native") ||
		  Node.storeType.equals("convert"))) {
		// Auto-select
		if(storeFiles[0].isFile()) {
		    Node.storeType="monolithic";
		} else {
		    Node.storeType="native";
		}
	    }
	    
	    if(Node.storeType.equals("monolithic") || Node.storeType.equals("convert")) {
                Storage storage = new RAFStorage(storeFiles, Node.storeSize);
                FileSystem fs;
                if (cipher == null) {
                    fs = new FileSystem(storage);
                }
                else {
                    cipher.initialize(cipherKey);
                    fs = new EncryptedFileSystem(storage, cipher, baseIV);
                }
                
                // FIXME:  there need to be some sanity checks on the
                //         ranges for the directory root, accounting tables,
                //         and files, to see if any of them were truncated.
                if (fs.truncation() < fs.size()) {
                    FSInitializer fsini = new FSInitializer(fs);
                    synchronized (fsini) {
                        try {
                            fsini.start();
                            fsini.wait();
                        }
                        catch (InterruptedException e) {}
                    }
                    if (fsini.delayedThrow != null)
                        throw fsini.delayedThrow; //.fillInStackTrace();
                }
                //
                // FIXME: replace literals with config values
                //
                dir = new FSDirectory(fs, new SHA1(), 10000, 10000, 1000, 100);
                storeSize = fs.size();
		if (!Node.storeType.equals("convert")) {
		    Core.logger.log(Main.class, "Monolithic datastore no longer supported. Converting to native. You had better have enough disk space for the conversion. If you don't, remove the old storefile.", Core.logger.ERROR);
		    Node.storeType = "convert";
		}
		String name = storeFiles[0].toString();
		if(name.charAt(name.length()-1) == File.separatorChar)
		    name = name.substring(0,name.length()-1);
		name += "-temp";
		Directory newDir;
		try {
		    newDir = new 
			NativeFSDirectory(new File(name),
					  storeSize,
					  Node.storeBlockSize,
					  Node.useDSIndex,
					  Node.storeMaxTempFraction);
		    Enumeration e = dir.keys(true);
		    Core.logger.log(Main.class, "Trying to convert datastore",
				    Core.logger.NORMAL);
		    long num = 0;
		    long bytes = 0;
		    while(e.hasMoreElements()) {
			num++;
			FileNumber fn = (FileNumber)e.nextElement();
			Buffer bIn = dir.fetch(fn);
			Core.logger.log(Main.class, "Converted "+num+" files "+
					bytes + " bytes.", Core.logger.NORMAL);
			bytes += bIn.length();
			Buffer bOut = newDir.store(bIn.length(), fn);
			InputStream i = bIn.getInputStream();
			OutputStream o = bOut.getOutputStream();
			byte[] b = new byte[4096];
			int x;
			while((x = i.read(b)) > 0) {
			    o.write(b, 0, x);
			};
			bOut.commit();
			bIn.release();
			bOut.release();
		    }
		    Core.logger.log(Main.class, "Conversion nearly done", 
				    Core.logger.NORMAL);
		} finally {
		    dir = null;
		    newDir = null;
		    // Try to close both of them
		    System.gc();
		    System.runFinalization();
		    System.gc();
		    System.runFinalization();
		    // Delete old files
		    for(int x=0;x<storeFiles.length;x++)
			storeFiles[x].delete();
		    new File(name).renameTo(storeFiles[0]);
		    dir = new NativeFSDirectory
			(storeFiles[0], Node.storeSize, Node.storeBlockSize,
			 Node.useDSIndex, Node.storeMaxTempFraction);
		    Core.logger.log(Main.class, "Conversion done", 
				    Core.logger.NORMAL);
		}            
	    } else {
                dir = new NativeFSDirectory(storeFiles[0], Node.storeSize, 
					    Node.storeBlockSize,
					    Node.useDSIndex,
					    Node.storeMaxTempFraction);
		Node.storeFile = new String[] { Node.storeFile[0] };
	    }
	    
	    Enumeration en = dir.lruKeys(false);
	    long dirLastModified = -1;
	    if(en.hasMoreElements())
		dirLastModified = ((NativeFSDirectory)dir). // FIXME?
		    mostRecentlyUsedTime();

	    // load DS
            Core.logger.log(Main.class, "loading data store", Logger.NORMAL);
            
            dsDir = new LossyDirectory(0x0001, dir);
            dsDir.forceFlush();
            DataStore ds = new FSDataStore(dsDir, storeSize / 100);
	    
            // load RT
            Core.logger.log(Main.class, "loading routing table", 
			    Logger.NORMAL);
	    
	    SimpleDataObjectStore rtNodes = null;
	    SimpleDataObjectStore rtProps = null;

	    
	    String rtn = "rtnodes_";
	    File[] nodesFile = {new File(Node.routingDir, "rtnodes_a"),
				new File(Node.routingDir, "rtnodes_b")};
	    
	    
	    
    	    rtn = "rtnodes_" + params.getInt("listenPort");
	    File[] altNodesFile = {new File(Node.routingDir, rtn + "a"),
				   new File(Node.routingDir, rtn + "b")};
	    
	    for(int x=0;x<2;x++) {
		if((!nodesFile[x].exists()) && altNodesFile[x].exists()) {
		    if(!altNodesFile[x].renameTo(nodesFile[x])) {
			if(!(nodesFile[x].delete() && 
			     altNodesFile[x].renameTo(nodesFile[x]))) {
			    Core.logger.log(Main.class, "Cannot rename "+
					    altNodesFile[x]+" to "+nodesFile[x]+
					    ". This would be useful"+
					    " because then you could change the port"+
					    " without causing the routing table to"+
					    " disappear.", Core.logger.NORMAL);
			    nodesFile[x] = altNodesFile[x];
			}
		    }
		}
	    }
	    
	    String rtp = "rtprops_" + params.getInt("listenPort");
	    
	    File[] propsFile = {new File(Node.routingDir, "rtprops_a"),
				new File(Node.routingDir, "rtprops_b")};
	    
	    File[] altPropsFile = {new File(Node.routingDir, rtp + "a"),
				   new File(Node.routingDir, rtp + "b")};
	    
	    for(int x=0;x<2;x++) {
		if((!propsFile[x].exists()) && altPropsFile[x].exists()) {
		    if(!altPropsFile[x].renameTo(propsFile[x])) {
			if(!(propsFile[x].delete() && 
			     altPropsFile[x].renameTo(propsFile[x]))) {
			    Core.logger.log(Main.class, "Cannot rename "+
					    altPropsFile[x]+" to "+propsFile[x]+
					    ". This would be useful"+
					    " because then you could change the port"+
					    " without causing the routing table to"+
					    " disappear.", Core.logger.NORMAL);
			    propsFile[x] = altPropsFile[x];
			}
		    }
		}
	    }
	    
	    // swap if b is newer
	    if (propsFile[1].lastModified() > propsFile[0].lastModified()){
		nodesFile = new File[] {nodesFile[1], nodesFile[0]};
		propsFile = new File[] {propsFile[1], propsFile[0]};
	    }
	    
	    for (int i = 0 ; i < nodesFile.length ; i++) {
		try {
		    if(rtNodes != null) {
			if((!rtNodes.truncated) && (!rtProps.truncated))
			    continue;
		    }
		    rtNodes = new SimpleDataObjectStore(nodesFile, i);
		    rtProps = new SimpleDataObjectStore(propsFile, i);
		} catch (IOException e) {
		    Core.logger.log(Main.class, 
				    "One of the routing table files was " +
				    "corrupt.", e, Logger.MINOR);
		    rtNodes = rtProps = null;
		}
	    }
	    
	    if (rtNodes == null || rtProps == null) {
		Core.logger.log(Main.class, 
				"Routing table corrupt! Trying to reseed.",
				Core.logger.ERROR);
		for(int x=0;x<nodesFile.length;x++) {
		    nodesFile[x].delete();
		    propsFile[x].delete();
		}
		rtNodes = new SimpleDataObjectStore(nodesFile, 0);
		rtProps = new SimpleDataObjectStore(propsFile, 0);
	    }
	    
	    if (rtNodes.truncated || rtProps.truncated)
		Core.logger.log(Main.class,
				"One of the routing table files was truncated",
				Core.logger.NORMAL);
	    
	    try {
		rtNodes.flush();
		rtProps.flush();
	    } catch (IOException e) {
		Core.logger.log(Main.class, "Flushing routing table failed!",
				e, Logger.ERROR);
	    }
	    
            DataObjectRoutingStore routingStore =
                new DataObjectRoutingStore(rtNodes, rtProps);
            
	    int maxARKLookups = 
		(int)(Node.maxThreads*((double)Node.maxARKThreadsFraction));
	    if(Node.maxThreads == 0) maxARKLookups = Node.rtMaxRefs;
	    if(maxARKLookups<0) maxARKLookups = -maxARKLookups;

	    CPAlgoRoutingTable origRT = 
		new CPAlgoRoutingTable(routingStore,
				       Node.rtMaxNodes,
				       Node.rtMaxRefs,
				       Node.failuresLookupARK,
				       Node.minARKDelay,
				       Node.minCP,
				       maxARKLookups,
				       Core.randSource);
	    
            RoutingTable rt = origRT;
	    
            //              RoutingTable rt = new ProbabilityRoutingTable(routingStore,
            //                                                            Node.rtMaxNodes,
            //                                                            Node.rtMaxRefs,
            //                                                            Core.randSource);

            rt = new FilterRoutingTable(rt, privateKey.getIdentity());

            // load FT

            FailureTable ft = 
                new FailureTable(params.getInt("failureTableSize"),
                                 params.getLong("failureTableTime"));

            

            // load LoadStats
	    
	    File[] lsFile = {new File(Node.routingDir, "lsnodes_a"),
			     new File(Node.routingDir, "lsnodes_b")};
	    
            String lsn = "lsnodes_" + params.getInt("listenPort");
	    File[] altLsFile = {new File(Node.routingDir, lsn + "a"),
				new File(Node.routingDir, lsn + "b")};
	    
	    for(int x=0;x<2;x++) {
		if((!lsFile[x].exists()) && altLsFile[x].exists()) {
		    if(!altLsFile[x].renameTo(lsFile[x])) {
			if(!(lsFile[x].delete() && altLsFile[x].renameTo(lsFile[x]))) {
			    Core.logger.log(Main.class, "Cannot rename "+altLsFile[x]+
					    " to "+lsFile[x]+". This would be useful"+
					    " because then you could change the port"+
					    " without causing the load stats data to"+
					    " disappear.", Core.logger.NORMAL);
			    lsFile[x] = altLsFile[x];
			}
		    }
		}
	    }
	    
	    // swap if b is newer
	    if (lsFile[1].lastModified() > lsFile[0].lastModified()){
		lsFile = new File[] {lsFile[1], lsFile[0]};
	    }
	    
            SimpleDataObjectStore lsNodes = null;
	    for (int i = 0 ; i < lsFile.length ; i++) {
		try {
		    if (lsNodes != null && !lsNodes.truncated) {
                        continue;
		    }
		    lsNodes = new SimpleDataObjectStore(lsFile, i);
		} catch (IOException e) {
		    Core.logger.log(Main.class, 
				    "One of the load stats files was " +
				    "corrupt.", e, Logger.MINOR);
		    lsNodes = null;
		    if(lsFile[i].delete()) {
			lsNodes = new SimpleDataObjectStore(lsFile, i);
		    } else {
			String err = "Can't delete corrupt loadstats file "+
			    lsFile[i].getAbsolutePath()+"!";
			System.err.println(err);
			Core.logger.log(Main.class, err, Core.logger.ERROR);
		    }
		}
	    }

	    if(lsNodes == null) {
		String err = "Could not load or create a valid loadstats file!";
		System.err.println(err);
		Core.logger.log(Main.class, err, Core.logger.ERROR);
		System.exit(27);
	    }
	    
            LoadStats loadStats = new LoadStats(lsNodes, 100, Core.diagnostics,
                                                null, 
						Node.defaultResetProbability);
	    
	    BucketFactory bf = loadTempBucketFactory(params, storeFiles[0],
						     Node.routingDir);
	    
	    if(rt.initialRefsCount() != 0)
		firstTime = false;

            try {
                // --seed
                // (now that we have the routing table)
                if (sw.getParam("seed") != null) {
                    NodeReference[] seedNodes = readSeedNodes(sw.getString("seed"));
		    Core.logger.log(Main.class, "read seed nodes", Logger.NORMAL);
                    seedRoutingTable(rt, seedNodes, true);
		    Core.logger.log(Main.class, "seeded routing table", Logger.NORMAL);
                }
                
                // run node
                else {
            
                    // load seed nodes
                    NodeReference[] seedNodes = null;
                    String seedNodesFile = params.getString("seedFile");
		    Core.logger.log(Main.class, "otherpath: seedFile="+
				    seedNodesFile, Core.logger.DEBUG);
                    if (seedNodesFile != null && !seedNodesFile.equals("")) {
                        try {
                            seedNodes = readSeedNodes(seedNodesFile);
			    Core.logger.log(Main.class, "read seed nodes",
					    Core.logger.NORMAL);
			    boolean seed = false;
			    Enumeration e = routingStore.elements();
			    if(!e.hasMoreElements()) {
				Core.logger.log(Main.class, "Reseeding because no keys in rt",
						Core.logger.MINOR);
				seed = true;
			    } else {
				long tt = (new File(seedNodesFile)).
				    lastModified();
				if(tt > dirLastModified) {
				    Core.logger.log(Main.class, 
						    "Reseeding because seedNodesFile more up "+
						    "to date than store: "+tt+" vs "+
						    dirLastModified, Core.logger.MINOR);
				    seed = true;
				}
			    }
			    Core.logger.log(Main.class, "Initial refs count: "+
					    rt.initialRefsCount(),
					    Core.logger.NORMAL);
			    if(rt.initialRefsCount() == 0)
				seed = true;
			    
			    if(seed) {
				seedRoutingTable(rt, seedNodes, false);
				Core.logger.log(Main.class, 
						"seeded routing table",
						Core.logger.NORMAL);
			    } else {
				Core.logger.log(Main.class, 
						"not seeding routing table", 
						Core.logger.NORMAL);
			    }
                            routingStore.checkpoint();  // save data
			    Core.logger.log(Main.class, "saved routing table",
					    Core.logger.NORMAL);
                        }
                        catch (FileNotFoundException e) {
                            Core.logger.log(Main.class,
                                            "Seed file not found: " + 
                                            seedNodesFile, Logger.NORMAL);
                        }
                    }

		    
                    Core.logger.log(Main.class, "starting node", 
                                    Logger.NORMAL);
                    
                    node = new Node(privateKey, myRef, dir, bf, ds, rt,
                                         ft, th, sh, ph, loadStats, 
                                         isTransient);

		    origRT.setNode(node);
		    
		    // write the new IP address into the node file
		    if(addr.length > 0)
			writeNodeFile();

                    // Initialize FCP FEC support
                    try {
			int cacheSize = params.getInt("FECInstanceCacheSize");
			// Worst case memory high water mark is 
			// 24MB * maxCodecs. Don't make it too big.
			int maxCodecs = params.getInt("FECMaxConcurrentCodecs");
                        Node.fecTools =  new FECTools(params, node.bf, 
						      cacheSize, maxCodecs);
                    }
                    catch (Exception e) {
			Core.logger.log(Main.class, 
					"Couldn't initialize FEC support!", 
					Core.logger.ERROR);
                        System.err.println("Couldn't initialize FEC support!");
                    }
		    
		    // Must create it for stuff in mainport dependant on it
		    configUpdater = new 
			NodeConfigUpdater(node.configUpdateInterval);
		    
                    startNode(addr, params);  // run Core
		    
                    // Handle watchme
                    if (params.getBoolean("watchme")) {
                        new Checkpoint(watchme).schedule(node);
                    }

		    // schedule ARK insertion
		    try {
			synchronized(ARKInserterLock) {
			    if(ARKversion > initialARKversion) {
				ARKinserter = new InsertARK();
				Core.logger.log(Main.class, "Physical address changed over restart, scheduling ARK insertion", Core.logger.MINOR);
				new Checkpoint(ARKinserter).schedule(node);
			    }
			}
		    } catch (KeyException e) {
			String err = "KeyException starting ARK insert! Report to devl@freenetproject.org: "+e;
			Core.logger.log(Main.class, err, e, Logger.ERROR);
			System.err.println(err);
			e.printStackTrace(System.err);
		    }
		    
                    if(node.configUpdateInterval != 0) try {
                        new Checkpoint(configUpdater).schedule(node);
                    } catch (Throwable e) {
                        Core.logger.log(Main.class, "On-the-fly config updater unable to start.", Core.logger.ERROR);
                    }
		    
                    if(node.aggressiveGC != 0)
			new Checkpoint(new GarbageCollectionCheckpointed()).schedule(node);
		    
                    // schedule checkpoints for link cleanup,
                    // routing table saves, and diagnostics aggregation

                    DiagnosticsCheckpoint diagnosticsCheckpoint =
                        new DiagnosticsCheckpoint(node.autoPoll, node.diagnostics);
    
                    new Checkpoint(sh).schedule(node);
                    new Checkpoint(routingStore).schedule(node);
                    new Checkpoint(diagnosticsCheckpoint).schedule(node);
                    new Checkpoint(ft).schedule(node);
                    new Checkpoint(loadStats).schedule(node);
		    new Checkpoint(ipDetector).schedule(node);
                    if (Node.useDSIndex)
                        new Checkpoint((NativeFSDirectory)dir).schedule(node);

                    // schedule announcements
                    if (params.getBoolean("doAnnounce")) {
                        boolean useRT = params.getBoolean("announcementUseRT");
                        int htl = params.getInt("announcementHTL");
                        int interval = params.getInt("announcementPollInterval");
                        if (node.isTransient()) {
                            node.logger.log(Main.class,
                                "Not announcing because I am transient.",
                                Logger.NORMAL);
                        } else if ((seedNodes == null || seedNodes.length == 0)
				   && !useRT) {
                            node.logger.log(Main.class,
					    "Cannot announce with no seed nodes!",
					    Logger.ERROR);
                        } else {
                            
			    int delay = firstTime ? 
				params.getInt("announcementFirstDelay") : interval;
			    
			    Announcing.placeAnnouncing(node, htl,
						       seedNodes, useRT,
                                                       delay);
			    node.logger.log(Main.class, "Will announce at HTL "+
					    htl+" with interval "+interval,
					    Logger.DEBUG);
                        }
                    }
		    
		    
		    Node.firstTime = firstTime;
                    node.join();  // join all interface threads
                }
            } finally {
                // save the latest state information
                routingStore.checkpoint();
                dsDir.forceFlush();
            }
        }
        catch (DiagnosticsException e) {
            System.err.println("ERROR: "+e.getMessage());
            System.err.println("The internal diagnostics could not be initialized.");
            //System.err.println("Set doDiagnostics=no in the configuration to disable them.");
            System.exit(1);
        }
        catch (ListenException e) {
            System.err.println("ERROR: "+e.getMessage());
	    e.printStackTrace(System.err);
            System.err.println("Could not bind to listening port(s)"
                               +" - maybe another node is running?");
            System.err.println("Or, you might not have privileges"
                               +" to bind to a port < 1024.");
            System.exit(1);
        }
	catch (Exception e) {
	    ByteArrayOutputStream s = new ByteArrayOutputStream();
	    PrintStream ps = new PrintStream(s);
	    e.printStackTrace(ps);
	    ps.flush();
	    Core.logger.log(Main.class, "Unexpected Exception: "
			    +e.getClass().getName()+"\n"+s.toString(),
			    Logger.ERROR);
	    throw e;
	}
        finally {
            Core.randSource.close();
        }
    }

    protected static BucketFactory loadTempBucketFactory(Params params,
							 File storeFile,
							 File routingDir) {
	// load temp bucket factory
	Core.logger.log(Main.class, "loading temp bucket factory", 
			Logger.NORMAL);
	
	String tempName;
	
	tempName = params.getString("tempDir");
	if(tempName == null || tempName.equals(""))
	    tempName = params.getString("FECTempDir");
	
	if(params.getBoolean("tempInStore")) {
	    Core.logger.log(Main.class, "Activating tempInStore", Logger.DEBUG);
	    TempBucketFactory.setHook(new NativeFSTempBucketHook());
	}
	
	if(tempName == null || tempName.equals("")) {
	    tempDir = new File(storeFile, "temp");
	}
	
	if(tempDir == null) {
	    if(tempName == null || tempName.equals("")) {
		tempDir = new File(routingDir, "client-temp");
	    } else {
		if(!tempName.endsWith(File.separator))
		    tempName += File.separator;
		tempDir = new File(tempName);
	    }
	}
	
	if(!tempDir.exists()) {
	    if(!tempDir.mkdir()) {
		String error = "WARNING: FECTempDir does not exist, and "+
		    "could not create";
		System.err.println(error);
		Core.logger.log(Main.class, error + " (it is "+
				tempDir+")", Logger.ERROR);
		System.exit(1);
	    }
	} else if(!tempDir.isDirectory()) {
	    String error = 
		"WARNING: FECTempDir is occupied by a file, cannot use";
	    System.err.println(error);
	    Core.logger.log(Main.class, error, Logger.ERROR);
	    System.exit(1);
	}
	
	Node.tempDir = tempDir;
	Core.logger.log(Main.class, "Node temp dir: " + tempDir,
			Core.logger.MINOR);
	System.setProperty("java.io.tmpdir", tempDir.toString());
	
	// Clean out
	File[] d = tempDir.listFiles();
	for(int x=0;x<d.length;x++) {
	    if(!d[x].isDirectory()) d[x].delete();
	}
	
	BucketFactory bf = new TempBucketFactory(tempDir.toString());
	
	Core.logger.log(Main.class, "loaded temp bucket factory", 
			Logger.NORMAL);
	
	if(Node.storeBlockSize >= 128) // sanity check
	    TempBucketFactory.defaultIncrement = Node.storeBlockSize;
	
	return bf;
    }
    
    // Finds the set or creates it if it doesn't exist.
    private static Params getSet(Params parent, String fullNameWithDots) {
         StringTokenizer st = new StringTokenizer(fullNameWithDots, ".");
         Params prev = parent;
         while (st.hasMoreTokens()) {
             String setName = st.nextToken().trim();
             if (setName.equals("")) {
                 throw new IllegalArgumentException("Can't parse set name: " + fullNameWithDots);
             }
             if (prev.isSet(setName)) {
                 // Use the existing set.
                 prev = (Params)prev.getSet(setName);
             }
             else if (prev.get(setName) != null) {
                 throw new IllegalArgumentException("Can't create set because non-set parameter with "
                                                    + "the same name already exists: " + fullNameWithDots);
             }
             else {
                 // Create a new set and add it.
                 Params newSet = new Params();
                 prev.put(setName, newSet);
                 prev = newSet;
             }
         }

         return prev;
    }

    // This is a hack to get the windows installer limping 
    // along. It should evenutally be removed. --gj 20021005
    private final static void removeObsoleteParameters(Params params) {
        if (params.isSet("fproxy")) {
            params.remove("fproxy");
            Core.logger.log(Main.class, 
                            "WORKAROUND: Ignoring obsolete fproxy.* lines in freenet.conf/ini. " + 
                            "You can remove them.",
                            Logger.NORMAL);
        }

        if (params.isSet("nodeinfo")) {
            params.remove("nodeinfo");
            Core.logger.log(Main.class, 
                            "WORKAROUND: Ignoring obsolete nodeinfo.* lines in freenet.conf/ini. " + 
                            "You can remove them.",
                            Logger.NORMAL);

        }

        if (params.isSet("nodestatus")) {
            params.remove("nodestatus");
            Core.logger.log(Main.class, 
                            "WORKAROUND: Ignoring obsolete nodestatus.* lines in freenet.conf/ini. " + 
                            "You can remove them.",
                            Logger.NORMAL);
        }
	
        if (params.getString("services") != null) {
	    String[] list = Fields.commaList(params.getString("services"));
            boolean sawMainPort = false;
            int count = 0;
            int i = 0;
            for (i = 0; i < list.length; i++) {
                if (list[i].equals("mainport")) {
                    sawMainPort = true;
                }
                else if (list[i].equals("fproxy")) {
                    list[i] = null;
                    count++;
                    Core.logger.log(Main.class, 
                                    "WORKAROUND: Ignoring obsolete fproxy entry from services list " +
                                    "in freenet.conf/ini.  You can remove it.",
                                    Logger.NORMAL);
                    //continue;
                }
                else if (list[i].equals("nodeinfo")) {
                    list[i] = null;
                    count++;
                    Core.logger.log(Main.class, 
                                    "WORKAROUND: Ignoring obsolete nodeinfo entry from services list " +
                                    "in freenet.conf/ini.  You can remove it.",
                                    Logger.NORMAL);
                }
                else if (list[i].equals("nodestatus")) {
                    list[i] = null;
                    count++;
                    Core.logger.log(Main.class, 
                                    "WORKAROUND: Ignoring obsolete nodestatus entry from services list " +
                                    "in freenet.conf/ini.  You can remove it.",
                                    Logger.NORMAL);
                }
            }

            if (count > 0 || (!sawMainPort)) {
                int length = list.length - count;
                if (!sawMainPort) {
                    length++;
                }
                String[] newList = new String[length];
                int index = 0;
                if (!sawMainPort) {
                    newList[index] = "mainport";
                    index++;
                    // Hmmm... do we really want to make it impossible to turn off mainport?
                    Core.logger.log(Main.class, 
                                    "WORKAROUND: mainport is missing from the services list " +
                                    "in freenet.conf/ini.  You should add it.",
                                    Logger.NORMAL);
                }
                for (i = 0; i < list.length; i++) {
                    if (list[i] != null) {
                        newList[index] = list[i];
                        index++;
                    }
                }
                params.put("services", Fields.commaList(newList));
            }
        }
    }

    private static final class FSInitializer extends Thread {

        final FileSystem fs;
        Throwable delayedThrow = null;

        FSInitializer(FileSystem fs) {
            super("store-initializer");
            setDaemon(true);
            this.fs = fs;
        }

        public void run() {
            Node.logger.log(this,
                            "initializing data store ("+fs.size()+" bytes)",
                            Logger.NORMAL);
            try {
                long t = System.currentTimeMillis();
                fs.initialize(Node.randSource, this);
                t = System.currentTimeMillis() - t;
                Node.logger.log(this, "finished initializing store ("+(t/1000)+" sec)",
                                Logger.NORMAL);
            }
            catch (IOException e) {
                delayedThrow = e;
                Node.logger.log(this, "I/O error while initializing datastore",
                                e, Logger.ERROR);
            }
            catch (Throwable e) {
                delayedThrow = e;
                Node.logger.log(this, "Unhandled error while initializing datastore",
                                e, Logger.ERROR);
            }
            finally {
                // in case it wasn't reached by fs.initialize()
                synchronized (this) {
                    notify();
                }
            }
        }
    }

    /**
     * @return  an array of physical addresses this node will listen on
     */
    private static Address[] getAddresses(TransportHandler th, Params params,
					  InetAddress detected)
	throws BadAddressException {
        // for now, we are going to be sloppy and just assume we can get
        // "tcp" out of the TransportHandler and look for "ipAddress" and
        // "listenPort" in the config params.  what we should be doing is
        // pulling a generalized list of addresses out of the config params
        // and resolving them against the TransportHandler based on transport
        // name and transport-specific address string
        String ipAddress = params.getString("ipAddress");
	if(ipAddress == null) ipAddress = "";
        int listenPort   = params.getInt("listenPort");

        TCP tcp = (TCP) th.get("tcp");
        if (tcp == null) {
            throw new BadAddressException("TCP not supported, but this code " +
                                          "only supports TCP...");
        }
        String addr = ipAddress+':'+listenPort;
        if (ipAddress.length()==0) {
	    if(detected != null)
		addr = detected.getHostAddress() + ':' + listenPort;
	    else addr = null;
	}
	
	if(addr == null || (!tcp.checkAddress(addr))) {
	    throw new BadAddressException("Address seemed incorrect. " + 
					  "To use Freenet you must have " +
					  "a globally addressable Internet " +
					  "address set correctly.");
	}
	
        return new Address[] { tcp.getAddress(addr) };
    }

    /**
     * Actually sets up the interfaces and starts running the Core.
     * @throws IOException          for an I/O error reading seed nodes or
     *                              config files
     * @throws BadAddressException  if there was a problem getting listening
     *                              addresses or allowed host addresses
     * @throws ListenException      if one of the interfaces couldn't listen
     */
    private static void startNode(Address[] addr, Params params)
                    throws IOException, BadAddressException, ListenException {
        // set up a thread group, the threadpool, and the OCM

        tcpConnection.startSelectorLoops();

        ThreadGroup tg = new ThreadGroup(node.toString());
        ThreadFactory tf = (Node.maxThreads > 0
                            ? (ThreadFactory) new QThreadFactory(tg, 
                                                                 Node.maxThreads)
                            : (ThreadFactory) new FastThreadFactory(tg,
                                                                    Node.maxThreads 
                                                                    * -1));

	if (Node.maxThreads < 0) Node.maxThreads = -Node.maxThreads;

        // Keep a hold of the the ThreadManager so that we can use
        // it for rate limiting in Node.acceptRequest().
        node.threadFactory = tf;

        // Use at most half as many connections as we have threads.
        int maxConn = params.getInt("maxNodeConnections");
	
	if(Node.isWin9X) {
	    Core.logger.log(Main.class, "Detected Windows 95/98/ME. Limiting connections accordingly. To get rid of this message, use a proper operating system - sorry", Logger.NORMAL);
	    if(maxConn > 40) maxConn = 40;
	} else if (Node.isWinCE) {
	    Core.logger.log(Main.class, "You are crazy (os.name = "+
			    Node.sysName+")", Logger.NORMAL);
	}
        OpenConnectionManager ocm = new OpenConnectionManager(tf, maxConn);
                
        // set up interfaces
        Vector iv = new Vector();
         
        // we start an FNP interface for each of our external addresses
        ConnectionRunner fnp = new FreenetConnectionRunner(node, node.sessions,
                                                           node.presentations, ocm, 3);
        for (int i=0; i<addr.length; ++i) {
            ContactCounter contactCounter = null;
            if (params.getBoolean("logInboundContacts") && 
                (Core.inboundContacts == null)) {
                // REDFLAG: This works because getAddresses() 
                //          only returns one address which is
                //          assumed to be tcp. If getAddresses() 
                //          is ever fixed, this code may break.
                //
                // Enable monitoring of inbound contact addresses
                // via NodeStatusServlet.
                Core.inboundContacts = new ContactCounter();
                contactCounter = Core.inboundContacts;
            }

            iv.addElement(new PublicNIOInterface(addr[i].listenPart(true), // see below
                                              node, tf, ocm, fnp, 
                                              contactCounter));
	    // We don't throttle negotiations; 
	    // FnpLink will add a limiter wrapper to the streams
        }

        if (params.getBoolean("logOutboundContacts")) {
            // Enable monitoring of outbound contact addresses
            // via NodeStatusServlet.
            Core.outboundContacts = new ContactCounter();
        }

        if (params.getBoolean("logInboundRequests")) {
            // Enable monitoring of per host inbound  
            // request stats via NodeStatusServlet
            Core.inboundRequests = new ContactCounter();
        }

        if (params.getBoolean("logOutboundRequests")) {
            // Enable monitoring of per host outbound  
            // request stats via NodeStatusServlet
            Core.outboundRequests = new ContactCounter();
        }

	// Distribution of inbound DataRequests
	// Non-optional because we want to use it for request triage
	Core.requestDataDistribution = new KeyHistogram();
	
        if (params.getBoolean("logInboundInsertRequestDist")) {
            // Enable monitoring of the distribution of  
            // incoming InsertRequests via NodeStatusServlet
            Core.requestInsertDistribution = new KeyHistogram();
        }

	// Distribution of successful inbound DataRequests
	// Non-optional because we want to use it for request triage
	Core.successDataDistribution = new KeyHistogram();

	if (params.getBoolean("logSuccessfulInsertRequestDist")) {
	    // Enable monitoring of the distribution of
	    // successful incoming InsertRequests via NodeStatusServlet
	    Core.successInsertDistribution = new KeyHistogram();
	}

	doRequestTriageByDelay = params.getBoolean("doRequestTriageByDelay");
        
        // Not here, this get's called after
        // Node.init()
	//overloadLow = params.getFloat("overloadLow");
	//overloadHigh = params.getFloat("overloadHigh");
        //System.err.println("This happens later");

        // the FCP interface
        int negotiationLimit = params.getInt("negotiationLimit");
        SessionHandler clientSh = new SessionHandler();
        clientSh.register(new FnpLinkManager(negotiationLimit), 10);
        clientSh.register(new PlainLinkManager(), 20);
        
        PresentationHandler clientPh = new PresentationHandler();
        clientPh.register(new ClientProtocol(), 100);
        
        String fcpHosts = params.getString("fcpHosts");
        TCP ltcp;

        // if fcpHosts is specified, we'll listen on all interfaces
        // otherwise we listen only on loopback
        if (fcpHosts == null || fcpHosts.trim().equals("")) {
            ltcp = new TCP(InetAddress.getByName("127.0.0.1"), 1, false);
        }
        else {
            ltcp = new TCP(1, false);
        }

        ConnectionRunner fcp =
            new FreenetConnectionRunner(node, clientSh, clientPh, ocm, 1);
        
        ListeningAddress la =
            ltcp.getListeningAddress(""+params.getInt("clientPort"), 
				     Node.dontLimitClients);
        
        iv.addElement(new LocalNIOInterface(la, tf, fcp, fcpHosts,
					 -1, -1));
		
	// Test HTTP interface
// 	try {
// 	    MultipleHttpServletContainer container = new MultipleHttpServletContainer(node, 
// 										      false);
// 	    // FIXME: hardcoded
// 	    Params mainportParams = (Params)(params.getSet("mainport"));
// 	    mainportParams = (Params)(mainportParams.getSet("params"));
	    
// 	    container.init(mainportParams, "mainport.params");
// 	    LocalHTTPInterface testInterface = 
// 		new LocalHTTPInterface(ltcp.getListeningAddress(9999, Node.dontLimitClients), rsl, wsl, container, "127.0.0.0/8", Node.maxThreads/5, Node.maxThreads/3);
// 	    iv.addElement(testInterface);
// 	} catch (ServiceException e) {
// 	    Core.logger.log(Main.class, "Could not initialize service: "+e, e, 
// 			    Logger.ERROR);
// 	}
        // plus services
        String[] services = params.getList("services");
        if (services != null && !services[0].equals("none")) {
            for (int i=0; i<services.length; ++i) {
		node.logger.log(Main.class, "loading service: " + services[i],
                                Logger.NORMAL);
                Params fs = (Params)params.getSet(services[i]);
                if (fs == null) {
                    fs = new Params();
		    Core.logger.log(Main.class, "No config params for "+
				    services[i], Core.logger.DEBUG);
		}
                /* {
                    node.logger.log(Main.class,
                        "No configuration parameters found for: "+services[i],
                        Logger.ERROR);
                    continue;
                    }*/
                try {
		    Core.logger.log(Main.class, "Dumping fs: "+fs.toString(),
				    Core.logger.DEBUG);
                    Service svc = loadService(fs,services[i]);
                    iv.addElement(LocalNIOInterface.make(fs, tf, svc, 

						      Node.dontLimitClients,
						      Node.maxThreads/5,
						      Node.maxThreads/3));
		    // FIXME: totally arbitrary limits
                }
                //catch (Exception e) {
                // Need to catch link errors.
                catch (Throwable e) {
                    node.logger.log(Main.class,
                        "Failed to load service: "+services[i],
                        e, Logger.ERROR);
                }
            }
        }
        
        
        // start the Core
        
        NIOInterface[] interfaces = new NIOInterface[iv.size()];
        iv.copyInto(interfaces);

        MessageHandler mh =
            new StandardMessageHandler(node, 
                                       params.getInt("messageStoreSize"));
        initMessageHandler(mh, params);

        Ticker ticker = new Ticker(mh, tf);

        node.begin(tg, ticker, ocm, interfaces, false);
    }

    private static Service loadService(Params fs, String serviceName)
                            throws IOException, ServiceException {
        Service service;
        
        String className = fs.getString("class");
        if (className == null || className.trim().equals(""))
            throw new ServiceException("No class given");
        Class cls;
        try {
	    if(Core.logger.shouldLog(Core.logger.DEBUG))
		Core.logger.log(Main.class, "Trying to load "+className,
				Core.logger.DEBUG);
            cls = Class.forName(className.trim());
	    if(Core.logger.shouldLog(Core.logger.DEBUG))
		Core.logger.log(Main.class, "Loading "+className,
				Core.logger.DEBUG);
        }
        catch (ClassNotFoundException e) {
            throw new ServiceException(""+e);
        }

        if (Servlet.class.isAssignableFrom(cls)) {
            if (HttpServlet.class.isAssignableFrom(cls))
                service = new SingleHttpServletContainer(node, cls, true);
            else
                throw new ServiceException("I'm too dumb for: "+cls);
        }
        else {
	    Core.logger.log(Main.class, "Loading class "+cls, 
			    Core.logger.DEBUG);
            service = ServiceLoader.load(cls, node, true);
        }

        Config serviceConf = service.getConfig();
// 	Core.logger.log(Main.class, "Service config: "+serviceConf,
// 			Core.logger.DEBUG);
        if (serviceConf != null) {
            Params params;
            if (fs.getSet("params") != null) {  // read from Params
                params = new Params(serviceConf.getOptions(),
                                    (Params)fs.getSet("params"));
            }
            else if (fs.get("params") != null) {  // or external file
                params = new Params(serviceConf.getOptions());
                params.readParams(fs.get("params"));
            }
            else {
                params = new Params(serviceConf.getOptions());
            }
            service.init(params, serviceName);
        }

        return service;
    }
    
    
    private static void initDiagnostics(Params params, Logger logger)
                                        throws DiagnosticsException {

        // FIXME:  we now have the capability to store
        //         diagnostics data in the datastore

        // set up diagnostics
        String statsDir = params.getString("diagnosticsPath");

        Diagnostics d = new StandardDiagnostics(logger, statsDir);

        // Categories
        DiagnosticsCategory connections =
            d.addCategory("Connections", "Data regarding the connections.",
                          null);
        DiagnosticsCategory outConn =
            d.addCategory("Outbound", "Connections established.",
                          connections);
        DiagnosticsCategory inConn = 
            d.addCategory("Inbound", "Connections accepted.",
                          connections);

        DiagnosticsCategory messages = 
            d.addCategory("Messages", "Data regarding the routing of messages",
                          null);

        DiagnosticsCategory threading = 
            d.addCategory("Threading",
                          "Data regarding the thread pool, job scheduling, " +
                          "and job execution", null);
	
	DiagnosticsCategory client = 
	    d.addCategory("Client",
			  "Data regarding the client level", null);
	
        if (params.getBoolean("logOutputBytes")) {                          
            // if something else shows up in this category, move this line 
            // out of the if block
            DiagnosticsCategory transport =
                d.addCategory("Transport",
                              "Data regarding the transports (i.e., TCP)",
                              null);

            d.registerCounting("outputBytes", d.MINUTE,
                               "The number of bytes written via TCP, " + 
                               "excluding connections to localhost.", transport);
                               
            tcpConnection.logBytes = true;
        }
        
        // connections
        
        d.registerContinuous("connectionLifeTime", d.HOUR, 
                             "The amount of time that connections stay open."
                             + " In ms.", connections);
        d.registerBinomial("connectionTimedout", d.MINUTE, 
                           "The number of connections closed when " + 
                           "we timed them out. Success marks waiting " +
                           "until the end of the configured connectionTimeout,"
                           + "otherwise it was forced by overflow.",
                           connections);
        d.registerCounting("peerClosed", d.MINUTE,
                           "The number of connections that were closed by " +
                           "the peer.", connections);
        d.registerCounting("liveConnections", d.MINUTE,
                           "The number of connections established minus " +
                           "the number closed.", connections);
        d.registerCounting("liveLinks", d.MINUTE,
                           "The number of cached links established minus " +
                           "the number forgotten.", connections);
        d.registerContinuous("connectionMessages", d.MINUTE, 
                             "The number of messages sent over each open " +
                             "connection. In ms.", connections);
        d.registerCounting("readLockedConnections", d.MINUTE,
                           "The number of connections that start waiting for "+
                           "the trailing fields to be read, minus those that "+
                           "finish.", connections);
	d.registerCounting("connectionResetByPeer", d.MINUTE,
			   "The number of times we got Connection reset by "+
			   "peer errors", connections);
        
        
        // outbound connections

        d.registerContinuous("connectingTime", d.MINUTE,
                             "The amount of time it takes to establish a " +
                             "new connection fully. In ms.", outConn);
        d.registerContinuous("socketTime", d.MINUTE,
                             "The amount of time it takes to open a socket " +
                             "to other nodes. In ms.", outConn);
        d.registerContinuous("authorizeTime", d.MINUTE, 
                             "The amount of time it takes to authorize new " +
                             "connections. In ms.", outConn);
        d.registerBinomial("connectionRatio", d.MINUTE,
                           "The successrate of new connections.", outConn);
        d.registerBinomial("outboundRestartRatio", d.MINUTE,
                           "The number of outbound connections that " + 
                           "restart an excisting session (key).",outConn);

        // inbound connections

        d.registerCounting("incomingConnections", d.MINUTE,
                           "The number of incoming connections.", inConn);
        d.registerCounting("inboundConnectionsDispatched", d.MINUTE,
                           "The number of inbound connection dispatch attempts"
                           + " via PublicInterfaces.",
                           inConn);
        d.registerCounting("inboundConnectionsConnLimitRej", d.MINUTE,
                           "The number of inbound connection's rejected " + 
                           "because of connection limit.", inConn);
        d.registerCounting("inboundConnectionsThreadLimitRej", d.MINUTE,
                           "The  number of inbound connection's rejected " + 
                           "because of thread limit.", inConn);
        d.registerCounting("inboundConnectionsAccepted", d.MINUTE,
                           "The  number of inbound connection's accepted.",
                           inConn);
        d.registerBinomial("inboundRestartRatio", d.MINUTE,
                           "The number of inbound connections that " + 
                           "restart an excisting session (key).", inConn);
	d.registerCounting("readinessSelectionScrewed", d.MINUTE,
			   "The number of times NIO makes a zero byte "+
			   "read after the JVM told it that the socket "+
			   "was readable.", inConn);
        // messages
	
        d.registerContinuous("hopTime", d.MINUTE,
                             "The time taken per hop on requests that " +
                             "reach termination. Note that the values are " +
                             "means of the remaining hops, so deviation will "
                             + "be incorrect. In ms.", messages);
        // FIXME:  need to think about this:
        //         is there a good way to capture statistics on this?
        //d.registerContinuous("storeDataTime", d.HOUR,
        //                     "The amount time each piece of data is kept in" 
        //                     + " the cache.");
        //
        // ^^^ this was supposed to be the amount of time to receive the
        //     StoreData message after the data is received

        d.registerCounting("liveChains", d.HOUR,
                           "The number of request chains started minus " +
                           "the number completed.", messages);
        d.registerCounting("incomingRequests", d.MINUTE,
                           "The number of DataRequest queries received.", messages);
        d.registerCounting("incomingInserts", d.MINUTE,
                           "The number of InsertRequest queries received.",
                           messages);
        d.registerCounting("lostRequestState", d.HOUR,
                           "The number of live, pending requests that " +
                           "were dropped due to resource limits, not " +
                           "counting those awaiting the final StoreData",
                           messages);
        d.registerCounting("lostAwaitingStoreData", d.HOUR,
                           "The number of live, pending requests that were " +
                           "dropped due to resource limits, but were in " +
                           "the final phase of waiting for the StoreData",
                           messages);
	d.registerCounting("lookupARKattempts", d.MINUTE,
			   "The number of requests attempting to fetch ARKs",
			   messages);
	d.registerCounting("startedLookupARK", d.HOUR,
			   "The number of times we start to request an ARK",
			   messages);
	d.registerCounting("fetchedLookupARK", d.HOUR,
			   "The number of times we fetched an ARK",
			   messages);
	d.registerCounting("validLookupARK", d.HOUR,
			   "The number of times we fetched a *valid* ARK",
			   messages);
	d.registerCounting("successLookupARK", d.HOUR,
			   "The number of times we fetched a valid ARK that "+
			   "actually lead to successful connections", messages);
	d.registerCounting("queuedBackgroundInsert", d.HOUR,
			   "The number of times we queued a background insert",
			   messages);
	d.registerCounting("successBackgroundInsert", d.HOUR,
			   "The number of times a background insert completed "+
			   "successfully", messages);
	d.registerCounting("failureBackgroundInsert", d.HOUR,
			   "The number of times a background insert failed",
			   messages);
	d.registerCounting("pcacheAccepted", d.MINUTE,
			   "Keys accepted into the datastore by the "+
			   "probabilistic caching mechanism", messages);
	d.registerCounting("pcacheRejected", d.MINUTE,
			   "Keys rejected by the probabilistic caching "+
			   "mechanism; node will attempt to remove them", 
			   messages);
	d.registerCounting("pcacheFailedDelete", d.MINUTE,
			   "Keys rejected by the probabilistic caching "+
			   "mechanism, but not deleted because of being used "+
			   "since commit", messages);
	d.registerCounting("prefAccepted", d.MINUTE,
			   "Keys accepted into routing table by the "+
			   "probabilistic reference mechanism", messages);
	d.registerCounting("prefRejected", d.MINUTE,
			   "Keys not referenced in routing table because "+
			   "rejected by the probabilistic routing mechanism",
			   messages);
	d.registerCounting("storeDataAwaitingStoreData", d.MINUTE,
			   "StoreData sent while in AwaitingStoreData", messages);
	d.registerCounting("storeDataReceivingInsert", d.MINUTE,
			   "StoreData sent while in ReceivingInsert", messages);
	d.registerCounting("storeDataSendingReply", d.MINUTE,
			   "StoreData sent while in SendingReply", messages);
	d.registerContinuous("incomingHopsSinceReset", d.MINUTE,
			     "The hopsSinceReset value on incoming messages", 
			     messages);
        d.registerContinuous("timeBetweenFailedRequests", d.MINUTE,
                             "The time between hits to the same entry " +
                             "in the table of recently failed keys.",
                             messages);
        d.registerContinuous("failureTableBlocks", d.MINUTE,
                             "The number of requests that are blocked by " +
                             "each entry on the failure table.",
                             messages);
	d.registerContinuous("sendingReplyHTL", d.MINUTE,
			     "The remaining Hops To Live on non-local "+
			     "queries served from the datastore.", messages);
	
        // FIXME:  use a Binomial
        d.registerCounting("inboundAggregateRequests", d.MINUTE,
                           "The number of incoming queries of all types.",
                           messages);
        d.registerCounting("inboundAggregateRequestsHandled", d.MINUTE,
                           "The number of incoming queries that are not " + 
                           "rejected", messages);
        
        d.registerBinomial("outboundAggregateRequests", d.MINUTE,
                           "The number of outbound queries of all types.",
                           messages);

        d.registerCounting("inboundClientRequests", d.MINUTE,
                          "The number of client (FCP, InternalClient for fproxy) " +
                          "data and insert requests." ,
                           messages);

        d.registerCounting("announcedTo", d.HOUR,
                           "The number of other nodes that the node was successfully "
                           + "announced to.", messages);

        d.registerCounting("restartedRequests", d.HOUR,
                           "The number of (Insert/Data)requests that were "
                           + "restarted at this node do responce timeout. " +
                           "This is no longer used, see " + 
                           "restartedRequestAccepted.",
                           messages);
        
        d.registerBinomial("restartedRequestAccepted", d.HOUR,
                           "Counts instances when requests are restarted " +
                           "locally because of a response timeout. If the " +
                           "request had already been Accepted, it counts as " +
                           "a success.", messages);

        d.registerContinuous("routingTime", d.MINUTE,
                             "The amount of time between when requests are " +
                             "dispatched (client call or message received) " +
                             "and when the first route is found. Note that " +
			     "this may include network activity such as DNS " +
			     "name resolution.", messages);
	
	d.registerContinuous("preRoutingTime", d.MINUTE,
			     "The amount of time between when requests are " +
			     "dispatched and Pending receives the message.",
			     messages);
	
	d.registerContinuous("subRoutingTime", d.MINUTE,
			     "The amount of time between when requests are " +
			     "received by Pending and when the first route " +
			     "is found.", messages);
	
	d.registerContinuous("searchDataRoutingTime", d.MINUTE,
			     "The amount of time used to check that the key " +
			     "is not in the datastore. Not logged if the key IS "+
			     "in the datastore - see searchDataTime", messages);
	
	d.registerContinuous("getRouteTime", d.MINUTE,
			     "The amount of time taken to actually create the " +
			     "route object from the routing table.", messages);
	
	d.registerContinuous("stillInSendOnTime", d.MINUTE,
			     "The time to get route from the route object.", 
			     messages);
	
	d.registerContinuous("stillStillInSendOnTime", d.MINUTE,
			     "The time to call getPeer() on the route.", messages);
	
	d.registerContinuous("regotTime", d.MINUTE,
			     "Another bit.", messages);
	
	d.registerContinuous("messageInitialStateTime", d.MINUTE,
			     "The time to get a message from creation to execution.",
			     messages);
	
	d.registerContinuous("searchFoundDataTime", d.MINUTE,
			     "The time Pending took to find a file in the datastore.",
			     messages);
	
	d.registerContinuous("searchNotFoundDataTime", d.MINUTE,
			     "The time Pending took to not find a file in the datastore.",
			     messages);
	
        d.registerBinomial("receivedData", d.MINUTE,
                           "The number of times data was received, and " + 
                           "whether receiving was successful.", messages);
	
        d.registerBinomial("sentData", d.MINUTE,
                           "The number of times data was sent, and whether " +
                           "sending was successful.", messages);
        
        // threading

        d.registerCounting("jobsExecuted", d.MINUTE,
                           "The number of jobs executed by the threadpool",
                           threading);
        d.registerCounting("overflowThreads", d.MINUTE,
                           "The number of overflow threads spawned " +
                           "by the threadpool", threading);
        d.registerCounting("insufficientThreads", d.MINUTE,
                           "The number of times the threadpool rejected " +
                           "a job due to thread scarcity", threading);
        d.registerContinuous("tickerDelay", d.MINUTE,
                             "The delay between the time an MO is scheduled " +
                             "to execute on the ticker and the time it " +
                             "actually starts executing, in milliseconds.  " +
                             "It's a very rough measurement, but large " +
                             "values are an indicator of either a bug or a " +
                             "very heavily overloaded node.", threading);
        d.registerContinuous("jobsPerQThread", d.MINUTE,
                             "The amount of jobs done by each QThread.",
                             threading);
	
	d.registerBinomial("segmentSuccessRatio", d.HOUR,
			   "The proportion of segments downloaded that "+
			   "succeed.", client);
	
        Core.autoPoll = new AutoPoll(d, logger);
        Core.diagnostics = d;
    }
    
    
    private static void initMessageHandler(MessageHandler mh, Params params) {
        // FNP messages
        mh.addType( FNPRawMessage.class, 
                    VoidMessage.messageName, 
                    VoidMessage.class);
        mh.addType( FNPRawMessage.class, 
                    DataRequest.messageName,    
                    DataRequest.class    );
        mh.addType( FNPRawMessage.class, 
                    DataReply.messageName,      
                    DataReply.class      );
        mh.addType( FNPRawMessage.class, 
                    freenet.message.DataNotFound.messageName,   
                    freenet.message.DataNotFound.class );
        mh.addType( FNPRawMessage.class, 
                    QueryRejected.messageName,  
                    QueryRejected.class  );
        mh.addType( FNPRawMessage.class, 
                    QueryAborted.messageName,   
                    QueryAborted.class   );
        mh.addType( FNPRawMessage.class, 
                    QueryRestarted.messageName, 
                    QueryRestarted.class );
        mh.addType( FNPRawMessage.class, 
                    StoreData.messageName,      
                    StoreData.class      );
        mh.addType( FNPRawMessage.class, 
                    InsertRequest.messageName,  
                    InsertRequest.class  );
        mh.addType( FNPRawMessage.class, 
                    InsertReply.messageName,    
                    InsertReply.class    );
        mh.addType( FNPRawMessage.class, 
                    Accepted.messageName,       
                    Accepted.class       );
        mh.addType( FNPRawMessage.class, 
                    DataInsert.messageName,     
                    DataInsert.class     );
        mh.addType( FNPRawMessage.class, 
                    NodeAnnouncement.messageName,     
                    NodeAnnouncement.class     );
        mh.addType( FNPRawMessage.class,
                    AnnouncementReply.messageName,
                    AnnouncementReply.class    );
        mh.addType( FNPRawMessage.class,
                    AnnouncementExecute.messageName,
                    AnnouncementExecute.class  );
        mh.addType( FNPRawMessage.class,
                    AnnouncementComplete.messageName,
                    AnnouncementComplete.class );
        mh.addType( FNPRawMessage.class,
                    AnnouncementFailed.messageName,
                    AnnouncementFailed.class );
   
        // FCP messages
        mh.addType( FCPRawMessage.class, ClientHello.messageName,     ClientHello.class     );
	mh.addType( FCPRawMessage.class, ClientInfo.messageName,      ClientInfo.class      );
        mh.addType( FCPRawMessage.class, GenerateSVKPair.messageName, GenerateSVKPair.class );
        mh.addType( FCPRawMessage.class, InvertPrivateKey.messageName, InvertPrivateKey.class );
        mh.addType( FCPRawMessage.class, GenerateCHK.messageName,     GenerateCHK.class     );
	mh.addType( FCPRawMessage.class, GenerateSHA1.messageName,    GenerateSHA1.class    );
        mh.addType( FCPRawMessage.class, ClientGet.messageName,       ClientGet.class       );
        mh.addType( FCPRawMessage.class, ClientPut.messageName,       ClientPut.class       );
        mh.addType( FCPRawMessage.class, GetDiagnostics.messageName,  GetDiagnostics.class  );

        // FCP Messages for Forward Error Correction (FEC)
        mh.addType( FCPRawMessage.class, FECSegmentFile.messageName,  
                    FECSegmentFile.class  );
        mh.addType( FCPRawMessage.class, FECEncodeSegment.messageName, 
                    FECEncodeSegment.class  );
        mh.addType( FCPRawMessage.class, FECDecodeSegment.messageName,  
                    FECDecodeSegment.class  );
        mh.addType( FCPRawMessage.class, FECSegmentSplitFile.messageName,  
                    FECSegmentSplitFile.class  );
        mh.addType( FCPRawMessage.class, FECMakeMetadata.messageName,  
                    FECMakeMetadata.class  );

        mh.addType( FCPRawMessage.class,                              Illegal.class         );

        // additional messages given in the configuration
        String[] messageList = params.getList("messageTypes");
        if (messageList != null) {
            for (int i=0; i+2 < messageList.length; i+=3) {
                try {
                    mh.addType(
                        Class.forName(messageList[i]),
                        messageList[i+1],
                        Class.forName(messageList[i+2])
                    );
                }
                catch (ClassNotFoundException e) {
                    Core.logger.log(Main.class, 
                                    "Cannot register message type", 
                                    e, Logger.ERROR);
                }
            }
        }
    }
    


    private static void copyStream(InputStream in, OutputStream out, long length)
                                                                throws IOException {
        byte[] buf = new byte[Core.blockSize];
        while (length > 0) {
            int n = in.read(buf, 0, (int) Math.min(length, buf.length));
            if (n == -1) throw new EOFException();
            out.write(buf, 0, n);
            length -= n;
        }
    }


    private static final void copyBuffer(Buffer src, Buffer dst) throws IOException {
        copyStream(src.getInputStream(), dst.getOutputStream(), src.length());
    }



    
    /**
     * Write a FieldSet to a file or stdout ("-")
     */
    private static void writeFieldSet(FieldSet fs, String file) throws IOException {
        if (file.equals("-")) {
            fs.writeFields(new WriteOutputStream(System.out));
        }
        else {
            FileOutputStream out = new FileOutputStream(file);
            try {
                fs.writeFields(new WriteOutputStream(out));
            }
            finally {
                out.close();
            }
        }
    }

    
    /**
     * Read NodeReferences from a file or stdin ("-")
     */
    private static NodeReference[] readSeedNodes(String file) throws IOException {
        InputStream in = (file.equals("-") ? System.in : new FileInputStream(file));
        try {
	    CommentedReadInputStream rin = 
		new CommentedReadInputStream(in, "#");
            Vector v = new Vector();
            synchronized (v) {
                try {
                    while (true) {
                        //FieldSet fs = new FieldSet(rin);
                        // FIXME:
                        // shouldn't the FieldSet constructor and the parseFields
                        // methods throw the EOFException instead of eating it?
                        
                        FieldSet fs = new FieldSet();
                        if (null == fs.parseFields(rin))
                            throw new EOFException();
                        
                        try {
                            v.addElement(new NodeReference(fs));
                        }
                        catch (BadReferenceException e) {
                            Core.logger.log(Main.class,
                                "Skipped bad NodeReference while reading seed nodes",
                                e, Logger.ERROR);
                        }
                    }
                }
                catch (EOFException e) {}
                
                NodeReference[] ret = new NodeReference[v.size()];
                v.copyInto(ret);
                // shuffle
                for (int j = ret.length - 1 ; j > 0 ; j--) {
                    int k = (int) ((j + 1) * Core.randSource.nextFloat());
                    NodeReference temp = ret[k];
                    ret[k] = ret[j];
                    ret[j] = temp;
                }
                return ret;
            }
        }
        finally {
            in.close();
        }
    }


    /**
     * Crams a set of node refs into the routing table with fake keys.
     * The fake keys end in 0x0000.
     * @param force  true means to add a ref even if the node already
     *               exists in the routing table
     */
    // public so Announcing can weak-reseed if things are completely FUBAR
    public static void seedRoutingTable(RoutingTable rt,
					NodeReference[] nodes,
					boolean force) {
	Vector n = new Vector(nodes.length);
	for(int x=0;x<nodes.length;x++) {
	    n.add(nodes[x]);
	} // it is ridiculous that we have to do this by hand, but that's java :(
	Core.logger.log(Main.class, "Seeding Routing Table: "+nodes.length+
			" nodes", Core.logger.DEBUG);
        for (int i=0; i<nodes.length; ++i) {
	    int x = Core.randSource.nextInt() % n.size();
	    if(x < 0) x = -x;
	    NodeReference ref = (NodeReference)(n.elementAt(x));
	    n.remove(x);
	    Core.logger.log(Main.class, "Node "+i, Core.logger.DEBUG);
            if (force || !rt.references(ref.getIdentity())) {
		Core.logger.log(Main.class, "Doing node "+i, 
				Core.logger.DEBUG);
                int r = Core.randSource.nextInt();
                int c = ref.hashCode();
                byte[] k = new byte[18];
                for (int j=0; j<16; ++j)
                    k[j] = (byte) (0xff & ( r>>(8*(j/4)) ^ c>>(8*(j%4)) ));
		Core.logger.log(Main.class, "Referencing node "+i+" to key "+k,
				Core.logger.DEBUG);
                rt.reference(new Key(k), ref);
            } else if (rt.references(ref.getIdentity())) {
		rt.reference(null, ref);
		// Give opportunity to update ARK
	    }
        }
	Core.logger.log(Main.class, "Seeded Routing Table", 
			Core.logger.DEBUG);
    }
    

    /**
     * Print version information
     */
    public static void version() {
        System.out.println(
            Version.nodeName + " version " + Version.nodeVersion
            +", protocol version "         + Version.protocolVersion
            +" (build "                    + Version.buildNumber
            +", last good build "          + Version.lastGoodBuild
            +")"
        );
    }

    /**
     * Print usage information
     */
    public static void usage() {
        version();
        System.out.println("Usage: java "+Main.class.getName()+" [options]");
        System.out.println("");
        System.out.println("Configurable options");
        System.out.println("--------------------");
        Node.config.printUsage(System.out);
        System.out.println("");
        System.out.println("Command-line switches");
        System.out.println("---------------------");
        switches.printUsage(System.out);
        System.out.println("");
        System.out.println("Send support requests to support@freenetproject.org.");
        System.out.println("Bug reports go to devl@freenetproject.org.");
    }

    /**
     * Print HTML manual
     */
    public static void manual() {
        System.out.println("<html><body>");
        manual(new PrintWriter(System.out));
        System.out.println("</body></html>");
    }

    public static void manual(PrintWriter out) {
        out.println("<br><br>");
        out.println("<h2>Freenet Reference Daemon Documentation</h2>");
        out.println("<h3>" + Config.htmlEnc(Version.getVersionString()) +
                    "</h3>");
        out.println("<br>");
        java.text.DateFormat df = java.text.DateFormat.getDateTimeInstance();
        out.println("<i>(This manual was automatically generated on " + 
                    Config.htmlEnc(df.format(new Date()))
                    + ". If you have updated Freenet since then, you " +
                    "may wish regenerate it.)</i>");
        out.println("<br><br>");
        out.println("FRED (Freenet REference Daemon) is the standard " +
                    "implementation of Freenet. This is the node, which " +
                    "serves as a router, data cache, and personal gateway " +
                    "all rolled into one. For FRED to run, it requires a " +
                    "configuration file to be present - this can be created " +
                    "either during the installation, by starting the node " +
                    "with the --config switch (see below), or running the "+
                    "freenet.config.Setup class manually.");
        out.println("<br><br>");
        out.println("See the <a href=\"http://www.freenetproject.org/"
                    + "tiki-index.php?page=Documentation\"> project documentation" +
                    " pages</a> for more information, or ask pointed & " +
                    " specific questions on the <a href=\"" +
                    "http://www.freenetproject.org/tiki-index.php?page=MailingLists\">" +
                    "mailing lists</a>.");
        out.println("<br><br>");
        out.println("<br>");
        out.println("<h3>Command line switches: </h3>");
        out.println("<hr>");
        switches.printManual(out);
        out.println("<h3>Configuration options: </h3>");
        out.println("These can reside either in the configuration file " +
                    "or be given as command line arguments.");
        out.println("<hr>");
        Core.config.printManual(out);

    }

    public static void loadNodeFile() throws IOException {
	try {
	    File f = new File(Node.nodeFile);
	    if(f.length() == 0) throw new FileNotFoundException();
	    InputStream in = new FileInputStream(Node.nodeFile);
	    try {
		Core.logger.log(Main.class, "Reading node file...", 
				Core.logger.MINOR);
		CommentedReadInputStream cs = new CommentedReadInputStream(in, "#");
		FieldSet fs;
		try {
		    fs = new FieldSet(cs);
		} finally {
		    cs.close();
		    in = null;
		}
		privateKey = new DSAAuthentity(fs.getSet("authentity"));
		if (cipher != null) {
		    cipherKey = Fields.hexToBytes(fs.get("cipherKey"));
		    baseIV = Fields.hexToBytes(fs.get("baseIV"));
		}
		FieldSet phys = fs.getSet("physical");
		if(phys != null)
			oldTCPAddressAndPort = phys.get("tcp");

		 
		Core.logger.log(Main.class, "Got oldAddress: "+
				oldTCPAddressAndPort==null?"(null)":oldTCPAddressAndPort,
				Core.logger.DEBUG);
		FieldSet ark = fs.getSet("ARK");
		String s = ark==null ? null : ark.get("revision");
		if(s != null) {
		    Core.logger.log(Main.class, 
				    "Keeping old ARK data, revision "+s,
				    Core.logger.DEBUG);
		    try {
			ARKversion = Fields.hexToLong(s);
		    } catch (NumberFormatException e) {
			Core.logger.log(Main.class, 
					"ARK.revision does not parse",
					e, Core.logger.ERROR);
			throw new ParseIOException("bad node file: "+e);
		    }
		    initialARKversion = ARKversion;
		    s = ark.get("encryption");
		    if(s != null && s.length() > 8)
			ARKcrypt = Fields.hexToBytes(s);
		    else ARKcrypt = null;
		    if(ARKcrypt == null) {
			Core.logger.log(Main.class, 
					"ARK Encryption key not found!",
					Core.logger.ERROR);
			long oldver = ARKversion;
			newARK();
			ARKversion = oldver++;
		    }
		    s = ark.get("revisionInserted");
		    if(s != null) {
			try {
			    initialARKversion = Fields.hexToLong(s);
			} catch (NumberFormatException e) {
			    Core.logger.log(Main.class,
					    "ARK.revisionInserted corrupt!",
					    e, Core.logger.ERROR);
			}
		    }
		    if(initialARKversion > ARKversion)
			ARKversion = initialARKversion;
		    Core.logger.log(Main.class, "initialARKversion = "+
				    initialARKversion+", ARKversion = "+
				    ARKversion, Core.logger.DEBUG);
		    s = ark.get("format");
		    if(s == null || !s.equals("1"))
			ARKversion++; // reinsert - we inserted as old format
		} else {
		    Core.logger.log(Main.class, "No ARK.revision found", 
				    Core.logger.MINOR);
		    // happens normally when upgrading from a pre-ARK node
		    newARK();
		    try {
			writeNodeFile();
		    } catch (IOException e) {
			Core.logger.log(Main.class, 
					"IOException writing node file after creating ARK!",
					Core.logger.ERROR);
		    }
		}
		Core.logger.log(Main.class, "Read node file", Core.logger.NORMAL);
	    }
	    finally {
		if(in != null) in.close();
	    }
	    if(oldTCPAddressAndPort != null) {
		Core.logger.log(Main.class, "Old address was "+oldTCPAddressAndPort,
				Core.logger.DEBUG);
            }
	} catch (FileNotFoundException e) {
	    createNodeFile();
	}
    }

    public static void createNodeFile() throws IOException {
	Core.logger.log(Main.class, "Creating node keys: "+Node.nodeFile, 
			Logger.NORMAL);
	
	// FIXME: nodes should generate their own DSA group
	privateKey = new DSAAuthentity(Global.DSAgroupC, Node.randSource);
	if (cipher != null) {
	    cipherKey = new byte[cipher.getKeySize() >> 3];
	    baseIV = new byte[cipher.getBlockSize() >> 3];
	    Node.randSource.nextBytes(cipherKey);
	    Node.randSource.nextBytes(baseIV);
	}
	newARK();
	initialARKversion = 0;
	writeNodeFile();
    }

    public static void newARK() {
	ARKversion = 0;
	ARKcrypt = new byte[32]; // FIXME!
	Node.randSource.nextBytes(ARKcrypt);
    }
    
    public static void writeNodeFile() throws IOException {
	Core.logger.log(Main.class, "Writing node file...", Core.logger.DEBUG);
	FieldSet fs = new FieldSet();
	
	fs.put("authentity", privateKey.getFieldSet());
	
	if (cipher != null) {
	    fs.put("cipherKey", Fields.bytesToHex(cipherKey));
	    fs.put("baseIV", Fields.bytesToHex(baseIV));
	}
	
	tcpAddress tcp = getTcpAddress();
	if(tcp != null)
	    fs.makeSet("physical").put("tcp", tcp.toString());

	if(ARKcrypt != null) {
	    FieldSet ark = fs.makeSet("ARK");
	    ark.put("revision", Fields.longToString(ARKversion));
	    ark.put("revisionInserted", 
		    Fields.longToString(initialARKversion));
	    ark.put("format", "1");
	    ark.put("encryption", Fields.bytesToHex(ARKcrypt));
	}
	
	File temp = new File(Node.nodeFile+"-temp");
	File outfile = new File(Node.nodeFile);
	
	OutputStream out = 
	    new FileOutputStream(temp);
	try {
	    fs.writeFields(new WriteOutputStream(out));
	} catch (IOException e) {
	    Core.logger.log(Main.class, "Cannot write node file", e, Core.logger.ERROR);
	    System.err.println("Cannot write node file: "+e);
	    e.printStackTrace(System.err);
	} finally {
	    out.close();
	}
	
	if(!temp.renameTo(outfile)) {
	    if(!(outfile.delete()  && temp.renameTo(outfile))) {
		Core.logger.log(Main.class, "Cannot rename "+temp+" to "+outfile,
				Core.logger.ERROR);
		System.err.println("CANNOT RENAME NODE FILE "+temp+" TO "+outfile);
		return;
	    }
	}
	Core.logger.log(Main.class, "Written node file", Core.logger.DEBUG);
    }
    
    /**
     * Get the current tcpAddress of the node, from the NodeReference
     */
    public static tcpAddress getTcpAddress() {
	if(myRef == null || th == null)
	    return null;
	Transport tcp = th.get("tcp");
	Address addr = null;
	try {
	    addr = myRef.getAddress(tcp);
	} catch (BadAddressException e) {
	    Core.logger.log(Main.class, "BadAddressException getting our address from reference!", e, Core.logger.ERROR);
	    Node.badAddress = true;
	    return null;
	}
	Node.badAddress = false;
	return (tcpAddress)addr;
    }
    
    /**
     * Get the current internet address of the node from the NodeReference
     */
    public static InetAddress getInetAddress() {
	tcpAddress addr = getTcpAddress();
	if(addr == null) 
	    return null;
	else {
	    try {
		return addr.getHost();
	    } catch (java.net.UnknownHostException e) {
		Core.logger.log(Main.class, "Our own address "+addr.toString()+
				" is not resolvable!", Core.logger.ERROR);
		return null;
	    }
	}
    }
    
    public static void newInetAddress(InetAddress addr) {
	tcpTransport tcp = (tcpTransport)(th.get("tcp"));
	if(Node.localIsOK) {
	    Core.logger.log(Main.class, "Accepting "+addr+" because localIsOK",
			    Logger.DEBUG);
	} else {
	    if(tcp.checkAddress(addr)) {
		Core.logger.log(Main.class, "Address checked out: "+addr,
				Logger.DEBUG);
	    } else {
		Core.logger.log(Main.class, "Not updating to "+addr+
				" - local address", Logger.DEBUG);
		return; // Don't update to a nonlocal address
	    }
	}
	tcpAddress a = null;
	Address[] addrs = null;
	try {
	    addrs = getAddresses(th, params, addr);
	} catch (BadAddressException x) {
	    Core.logger.log(Main.class,
			    "BadAddressException updating node address",
			    x, Core.logger.ERROR);
	    Node.badAddress = true;
	    return;
	};
	Node.badAddress = false;
	a = (tcpAddress)addrs[0]; 
	// FIXME if we ever add multiple address support
	newTcpAddress(a, tcp, false);
    }

    public static void newTcpAddress(tcpAddress addr, Transport tcp, boolean force) {
	// FIXME: Move to Node
	// FIXME: easier way to increment ARKversion without changing address
	if((!force) && addr.equals(getTcpAddress())) {
	    Core.logger.log(Main.class, "getTcpAddress == "+addr+
			    " - not updating tcpAddress", Logger.DEBUG);
	    return;
	}
	Core.logger.log(Main.class, "Address changed from "+
			(getTcpAddress() == null ? "(null)" : 
			 getTcpAddress().toString())+" to "+addr.toString(), 
			Core.logger.MINOR);
	ARKversion++;
	Core.logger.log(Main.class, "ARKversion now "+ARKversion+
			", initialARKversion now "+initialARKversion,
			Logger.DEBUG);
	if(myRef != null) {
	    String[] physical = myRef.physical;
	    physical = (String[])physical.clone();
	    physical = myRef.setPhysicalAddress(physical, tcp, addr);
	    myRef = myRef.newVersion((DSAAuthentity)privateKey,
				     physical, ARKversion);
	    if(node != null) {
		node.myRef = myRef;
		if(node.begun()) {
		    try {
			synchronized(ARKInserterLock) {
			    if(ARKinserter != null) {
				ARKinserter = new InsertARK();
				new Checkpoint(ARKinserter).schedule(node);
			    }
			}
		    } catch (KeyException e) {
			String err = "KeyException starting ARK insert! Report to devl@freenetproject.org: "+e;
			Core.logger.log(Main.class, err, e, Logger.ERROR);
			System.err.println(err);
			e.printStackTrace(System.err);
		    }
		}
	    }
	}
	
	try {
	    writeNodeFile();
	} catch (IOException e) {
	    Core.logger.log(Main.class, "IOException trying to write out node file with new address!", e, Core.logger.ERROR);
	}
    }

    public static FieldSet getPhysicalFieldSet() {
	FieldSet fs = new FieldSet();
	fs.put("tcp", getTcpAddress().toString());
	// REDFLAG: this needs fixing if we ever support multiple addresses
	return fs;
    }
    
    public static InetAddress getDetectedAddress() {
	IPAddressDetector d = ipDetector;
	if(d != null) return d.getAddress();
	else return null;
    }
    
    public static InetAddress getDetectedAddress(int x) {
	IPAddressDetector d = ipDetector;
	if(d != null) return d.getAddress(x);
	else return null;
    }
    
    
    static class InsertARK extends AutoBackoffNodeRequester {

	boolean died = false;

	public InsertARK() throws KeyException {
	    super(new InternalClient(node), getURI(initialARKversion+1),
		  true, myGetBucket(), Node.maxHopsToLive);
	    hopsToLive = node.maxHopsToLive;
	}

	public String getCheckpointName() {
	    return "Inserting ARK";
	}

	public long nextCheckpoint() {
	    if(died) return -1;
	    synchronized(ARKInserterLock) {
		if (ARKversion <= initialARKversion) {
		    ARKinserter = null;
		    return -1;
		} else {
		    // We need to insert the ARK for ARKversion
		    return System.currentTimeMillis() + sleepTime;
		}
	    }
	}

	public void checkpoint() {
	    if(internalRun()) {
		synchronized(ARKInserterLock) {
		    initialARKversion++;
		    if(Core.logger.shouldLog(Core.logger.DEBUG))
			Core.logger.log(Main.class, "Success inserting ARK; "+
					"initialARKVersion = "+initialARKversion+
					", ARKversion = "+ARKversion,
					Core.logger.DEBUG);
		    try {
			writeNodeFile();
		    } catch (IOException e) {
			Core.logger.log(this, 
					"I/O Error writing node file after inserting ARK!", 
					Core.logger.ERROR);
		    }
		}
		try {
		    uri = getURI(initialARKversion+1);
		} catch (KeyException e) {
		    Core.logger.log(Main.class, "KeyException determining ARK insert URI! "+
				    "Report to devl@freenetproject.org!: "+e, e, Logger.NORMAL);
		    died = true;
		}
		Core.logger.log(Main.class, "Inserted ARK for version "
				+initialARKversion, Core.logger.MINOR);
		sleepTime = initialSleepTime;
		// Success with this key
		int x = hopsToLive;
		x = (x * 5)/4;
		if(x > node.maxHopsToLive) x = node.maxHopsToLive;
		hopsToLive = x;
		
		// We never set finished; we finish when ARKversion == initialARKversion
	    } else {
		// Retry in the normal way
	    }
	}
	
	protected static FreenetURI getURI(long x) throws KeyException {
	    DSAGroup dg = ((DSAIdentity)(myRef.getIdentity())).getGroup();
	    InsertURI uri = null;
	    uri = new InsertURI("SSK", Fields.longToHex(x),
				((freenet.crypt.DSAPrivateKey)privateKey).getX().toByteArray(),
				ARKcrypt, dg);
	    uri.setGroup(dg);
	    if(Core.logger.shouldLog(Core.logger.DEBUG)) {
		freenet.client.ClientKey ssk = 
		    freenet.client.ClientSSK.createFromInsertURI(null, 
								 uri);
		FreenetURI pubURI = ssk.getURI();
		Core.logger.log(Main.class,
				"Inserting ARK URI "+uri+" -> "+pubURI,
				Core.logger.DEBUG);
	    }
	    return uri;
	    // The routingKey includes more than just the privkey fingerprint, doesn't it?
	}
	
	protected Bucket getBucket() {
	    bucket = myGetBucket();
	    return bucket;
	}

	protected static Bucket myGetBucket() {
	    Bucket bucket;
	    ByteArrayOutputStream bos = new ByteArrayOutputStream(32);
	    try {
		myRef.getFieldSet().writeFields(new WriteOutputStream(bos));
	    } catch (IOException e) {
		Core.logger.log(Main.class, 
				"IOException writing to byte array!",
				e, Core.logger.ERROR);
		return null; // not our fault
	    }
	    bucket = new ArrayBucket (bos.toByteArray());
	    return bucket;
	}
	
	protected boolean success() {
	    return true;
	} // do not set finished=true

	protected boolean failure() {
	    if(hopsToLive>1) hopsToLive--;
	    return false;
	}

	protected boolean failedCollision(CollisionEvent e) {
	    Core.logger.log(this, "Collision alert inserting ARK ("+
			    (initialARKversion+1)+")!",
			    Core.logger.NORMAL);
	    synchronized(ARKInserterLock) {
		if ((initialARKversion+1) >= ARKversion) {
		    try {
			newTcpAddress((tcpAddress)(
						   (getAddresses(th, params, ipDetector.getAddress()))[0]), th.get("tcp"), false); // REDFLAG multiple addresses
		    } catch (BadAddressException x) {
			Node.badAddress = true;
			Core.logger.log(this, "Bad Address Exception trying to force update after collision!", x, Core.logger.ERROR);
		    }
		    Node.badAddress = false;
		}
	    }
	    return true;
	}
	
    }
    
    static class GarbageCollectionCheckpointed implements Checkpointed {

	// This isn't pretty, but unfortunately java is pretty dumb, and we're not as careful as we should be

	public void checkpoint() {
	    Core.logger.log(Main.class, "Currently used memory before GC: "+(Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()), Core.logger.DEBUG);
	    System.gc();
	    System.runFinalization();
	    /*** FIXME: remove before release ***/
	    // Why, zab?
	    if(Core.logger.shouldLog(Logger.MINOR)) {
		String status = "dump of interesting objects after gc in checkpoint:"+
		    "\ntcpConnections " +freenet.transport.tcpConnection.instances+
		    "\ntcpConnections open " +freenet.transport.tcpConnection.openInstances+
		    ((freenet.transport.tcpConnection.openInstances-freenet.transport.tcpConnection.instances > 2) ? "\n******** ERROR: MORE OPEN THAN EXTANT! ********" :"")+
		    
		    "\nsize of socketMap " + freenet.transport.tcpConnection.socketConnectionMap.size()+
		    "\nFnpLinks " +freenet.session.FnpLink.instances+
		    "\nPlainLinks " +freenet.session.PlainLink.instances+
		    "\nNIOOS " + freenet.support.io.NIOOutputStream.instances+
		    "\nNIOIS " + freenet.support.io.NIOInputStream.instances+
		    "\nCH " + freenet.ConnectionHandler.instances+
		    //"\nCH terminated " + freenet.ConnectionHandler.terminatedInstances+
		    "\nCHIS " +freenet.ConnectionHandler.CHISinstances+
		    "\nCHOS " +freenet.ConnectionHandler.CHOSinstances+
		    "\nRIS " +freenet.ConnectionHandler.RISinstances+
		    "\nSOS " +freenet.ConnectionHandler.SOSinstances+
		    "\nThrottled queue (R): "+tcpConnection.getRSL().
		    throttleQueueLength()+
		    "\nThrottled queue (W): "+tcpConnection.getWSL().
		    throttleQueueLength()+
		    "\ncloseUniqueness (R): "+tcpConnection.getRSL().
		    closeUniquenessLength()+
		    "\ncloseUniqueness (W): "+tcpConnection.getWSL().
		    closeUniquenessLength()+
		    "\nWSL.uniqueness: "+tcpConnection.getWSL().uniquenessLength()+
		    "\n"+tcpConnection.getWSL().analyzeUniqueness();
		
		Core.logger.log(this, status, Logger.MINOR);
	    }
	    Core.logger.log(Main.class, "Currently used memory after GC: "+(Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()), Core.logger.DEBUG);
	}

	public String getCheckpointName() {
	    return "Garbage Collection Checkpoint";
	}

	public long nextCheckpoint() {
            if (node.aggressiveGC == 0 ) return -1;
            return System.currentTimeMillis() + node.aggressiveGC*1000;
	}
    }
    
    static class TickStat {
	long user;
	long nice;
	long system;
	long spare;
	
	boolean read(File f) {
	    String firstline;
	    try {
		FileInputStream fis = new FileInputStream(f);
		ReadInputStream ris = new ReadInputStream(fis);
		firstline = ris.readln();
		ris.close();
	    } catch (IOException e) { return false; }
	    Core.logger.log(this, "Read first line: "+firstline,
			    Logger.DEBUG);
	    if(!firstline.startsWith("cpu")) return false;
	    long[] data = new long[4];
	    for(int i=0;i<4;i++) {
		firstline = firstline.substring("cpu".length()).trim();
		firstline = firstline + ' ';
		int x = firstline.indexOf(' ');
		if(x == -1) return false;
		String firstbit = firstline.substring(0, x);
		try {
		    data[i] = Long.parseLong(firstbit);
		} catch (NumberFormatException e) {
		    return false;
		}
		firstline = firstline.substring(x);
	    }
	    user = data[0];
	    nice = data[1];
	    system = data[2];
	    spare = data[3];
	    Core.logger.log(this, "Read from file: user "+user+" nice "+nice+
			    " system "+system+" spare "+spare, Logger.DEBUG);
	    return true;
	}
	
	void calculate(TickStat old) {
	    long userdiff = user - old.user;
	    long nicediff = nice - old.nice;
	    long systemdiff = system - old.system;
	    long sparediff = spare - old.spare;
	    
	    if(userdiff + nicediff + systemdiff + sparediff <= 0) return;
	    Core.logger.log(this, "User changed by "+userdiff+", Nice: "+nicediff+
			    ", System: "+systemdiff+", Spare: "+sparediff,
			    Logger.DEBUG);
	    int usage = (int)((100 * (userdiff + nicediff + systemdiff)) / 
			      (userdiff + nicediff + systemdiff + sparediff));
	    Core.logger.log(this, "CPU usage: "+usage, Logger.DEBUG);
	}
	
	void copyFrom(TickStat old) {
	    user = old.user;
	    nice = old.nice;
	    system = old.system;
	    spare = old.spare;
	}
    }
    
    static void runMiscTests() throws Throwable {
	int fds = 0;
	freenet.transport.tcpTransport transport = 
	    new freenet.transport.TCP(99, false);
	File fname = new File("/root/TODO");
	LinkedList fdlist = new LinkedList();
	while(true) {
	    try {
		FileInputStream fis = new FileInputStream(fname);
		fdlist.add(fis);
		fds++;
		Core.logger.log(Main.class, "Opened file "+fds+" times",
				Logger.DEBUG);
	    } catch (IOException e) {
		Core.logger.log(Main.class, "Got exception "+e, Logger.DEBUG);
		break;
	    }
	}
	Core.logger.log(Main.class, "We have "+fds+" fds", Logger.DEBUG);
	if(File.separator.equals("/")) {
	    File f = new File("/proc/stat");
	    TickStat tsOld = new TickStat();
	    TickStat tsNew = new TickStat();
	    tsOld.read(f);
	    if(f.exists() && f.canRead()) {
		for(int x = 0; x < 1000; x++) {
		    Thread.sleep(1000);
		    if(!tsNew.read(f)) {
			Core.logger.log(Main.class, "Failed to parse", Logger.ERROR);
		    }
		    tsNew.calculate(tsOld);
		    tsOld.copyFrom(tsNew);
		}
	    }
	}
	
	
	// Snip cross-platform attempt
	// 	    long totalTime = 0;
	// 	    long targetTime = 0;
	// 	    for(int y = 0; y < 100; y++) {
	// 		long startTime = System.currentTimeMillis();
	// 		Thread.sleep(100);
	// 		long sleepTime = System.currentTimeMillis();
	// // 		Core.logger.log(Main.class, "Sleep took "+(sleepTime-startTime), Logger.DEBUG);
	// 		Thread.yield();
	// 		targetTime += 100;
	// 		long endTime = System.currentTimeMillis();
	// 		long len = endTime - startTime;
	// 		totalTime += len;
	// // 		Core.logger.log(Main.class, "Took "+len, Logger.DEBUG);
	// 	    }
	// 	    long usage = (100 * (totalTime - targetTime)) / totalTime;
	// 	    Core.logger.log(Main.class, "CPU usage estimate "+x+": "+
	// 			    usage, Logger.DEBUG);
	long startTime = System.currentTimeMillis();
	for(int x=0;x<1000;x++) {
	    long enteredTime = System.currentTimeMillis();
	    BigInteger rand = new BigInteger(256, Core.randSource);
	    long gotRandTime = System.currentTimeMillis();
	    Core.logger.log(Main.class, "Got random in "+(gotRandTime - enteredTime),
			    Core.logger.DEBUG);
	    DSAGroup grp = Global.DSAgroupC;
	    BigInteger out = grp.getG().modPow(rand, grp.getP());
	    long modPowTime = System.currentTimeMillis();
	    Core.logger.log(Main.class, "modPow() in "+(modPowTime - gotRandTime),
			    Core.logger.DEBUG);
	}
	
	
	byte[] b = new byte[2];

	for(int y=-128;y<127;y++) {
	    b[0] = (byte)y;
	    for(int x=-128;x<127;x++) {
		b[1] = (byte)x;
		FileNumber fn = new FileNumber(b);
		Core.logger.log(Main.class, "" + x + ": " + fn.hashCode()+", "+
				fn.longHashCode(), Core.logger.DEBUG);
	    }
	}
	
	Vector v = new Vector();
	Core.logger.log(Main.class, 
			"Vector default capacity: "+v.capacity(),
			Core.logger.DEBUG);

	for(int x=0;x<256;x++) {
	    v.addElement(new Object());
	    Core.logger.log(Main.class, "Vector: "+v.capacity()+" at "+x,
			    Core.logger.DEBUG);
	}
	    
	for(int x=0;x<256;x++) {
	    v.removeElementAt(0);
	    Core.logger.log(Main.class, "Vector: "+v.capacity()+" at "+x,
			    Core.logger.DEBUG);
	}

	System.gc();
	System.runFinalization();
	System.gc();
	System.runFinalization();
	    
	Core.logger.log(Main.class, "Vector NOW: "+v.capacity(),
			Core.logger.DEBUG);
	v.trimToSize();
	Core.logger.log(Main.class, "Vector NOW: "+v.capacity(),
			Core.logger.DEBUG);

	Core.logger.log(Main.class, "Currently used memory (A): "+(Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()), Core.logger.DEBUG);

	Object [] vv = new Object[1024];

	Core.logger.log(Main.class, "Currently used memory (B): "+(Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()), Core.logger.DEBUG);

	for(int x=0;x<vv.length;x++)
	    vv[x] = new Object();
	    
	Core.logger.log(Main.class, "Currently used memory (C): "+(Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()), Core.logger.DEBUG);

	vv = null;
        Object[] f = new Object[1024];

	Core.logger.log(Main.class, "Currently used memory (D): "+(Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()), Core.logger.DEBUG);

	for(int x=0;x<f.length;x++)
	    f[x] = new FileNumber(new byte[23]);

	Core.logger.log(Main.class, "Currently used memory (E): "+(Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()), Core.logger.DEBUG);

	System.gc();
	System.runFinalization();
	System.gc();
	System.runFinalization();

	Core.logger.log(Main.class, "Currently used memory: "+(Runtime.getRuntime().totalMemory() - Runtime.getRuntime().freeMemory()), Core.logger.DEBUG);

	Random rand = new Random();

	for(int j=0;j<10;j++) {

	    byte[] k = new byte[18];
	    rand.nextBytes(k);
	    FileNumber fn = new FileNumber(k);

	    int x = fn.hashCode();
	    long xl = fn.longHashCode();
	    int y = Fields.hashCode(k);
	    long yl = Fields.longHashCode(k);

	    Core.logger.log(Main.class, "Generated \"random\" key, results: "+
			    x+", "+xl+"; "+y+", "+yl, Core.logger.DEBUG);
	}
	InetAddress myAddr = InetAddress.getLocalHost();
	Core.logger.log(Main.class, "Our address is "+myAddr.getHostAddress(),
			Core.logger.DEBUG);
	
	String s = Security.getProperty("networkaddress.cache.ttl");
	if(s == null) {
	    Core.logger.log(Main.class, "Cache TTL STILL ZERO!",
			    Core.logger.ERROR);
	} else {
	    Core.logger.log(Main.class, "Cache TTL now: "+s,
			    Core.logger.DEBUG);
	}
	Core.logger.log(Main.class, "Reset current DNS cache TTL",
			Core.logger.DEBUG);
	
	long t = System.currentTimeMillis();
	for(int x=0;x<100000;x++) {
	    Thread.sleep(2000);
	    long c = System.currentTimeMillis();
	    InetAddress addr = InetAddress.getByName("pascal.rockford.com");
	    long ct = System.currentTimeMillis();
	    Core.logger.log(Main.class, "Address: "+addr.getHostAddress()+
			    " at "+(c-t)+" ms - lookup took "+
			    (ct-c)+" ms", Core.logger.DEBUG);
	}
    }
    
    static class NativeFSTempBucketHook implements TempBucketHook {
	public void enlargeFile(long curLength, long writeLength) 
	    throws IOException {
	    if(Core.logger.shouldLog(Core.logger.DEBUG))
		Core.logger.log(this, "enlargeFile("+curLength+","+
				writeLength+")", Core.logger.DEBUG);
	    if(curLength > writeLength) {
		Core.logger.log(this, "curLength > writeLength!", Core.logger.ERROR);
		throw new IllegalArgumentException("curLength("+curLength+
						   ") > writeLength("+
						   writeLength+")!");
	    }
	    long x = ((NativeFSDirectory)(node.dir)).
		clearWrite(curLength, writeLength);
	    if(Core.logger.shouldLog(Core.logger.DEBUG))
		Core.logger.log(this, "NativeFSDir returned "+x, Core.logger.DEBUG);
	    if(x < 0) throw new IOException("insufficient storage");
	    if(x > 0) {
		while(x > 0) {
		    if(Core.logger.shouldLog(Core.logger.DEBUG))
			Core.logger.log(this, "Getting "+x+" bytes of space",
					Core.logger.DEBUG);
		    dsDir.getSpace(x);
		    if(Core.logger.shouldLog(Core.logger.DEBUG))
			Core.logger.log(this, "Got some space",
					Core.logger.DEBUG);
		    x = ((NativeFSDirectory)(node.dir)).
			clearWrite(curLength, writeLength);
		}
		if(x == 0) return;
		else throw new IOException("insufficient storage");
	    }
	}
	
	public void shrinkFile(long shorter, long longer) {
	    if(Core.logger.shouldLog(Logger.DEBUG))
		Core.logger.log(this, "shrinkFile("+longer+","+shorter+")",
				new Exception("debug"), Logger.DEBUG);
	    if(shorter > longer)
		throw new IllegalArgumentException("longer("+longer+
						   ")>shorter("+shorter+")!");
	    long x = ((NativeFSDirectory)(node.dir)).
		clearWrite(longer, shorter);
	    if(x < 0 || x > 0) {
		Exception e = new Exception
		    ("Can't allocate space for temp file SHRINK!");
		Core.logger.log(this, "shrinkFile("+longer+","+shorter+
				") failed", e, Core.logger.ERROR);
	    }
	}
	
	public void deleteFile(long length) {
	    if(Core.logger.shouldLog(Core.logger.DEBUG))
		Core.logger.log(this, "Deleting file of size "+length,
				Core.logger.DEBUG);
	    long status = 
		((NativeFSDirectory)(node.dir)).onDeleteTempFile(length);
	    if(status == 0) {
		if(Core.logger.shouldLog(Core.logger.DEBUG))
		    Core.logger.log(this, "Deleted file of size "+length+
				    " from temp space successfully.",
				    Core.logger.DEBUG);
	    } else { // status > 0 || status < 0
		Core.logger.log(this, "Impossible to delete temp file of size "+
				length, new Exception("debug"), 
				Core.logger.ERROR);
	    }
	}
	
	public void createFile(long length) throws IOException {
	    if(Core.logger.shouldLog(Core.logger.DEBUG))
		Core.logger.log(this, "Creating file of size "+length,
				Core.logger.DEBUG);
	    enlargeFile(-1, length);
	}
    }
}
