package freenet.node;

import freenet.Core;
import freenet.diagnostics.Diagnostics;
import freenet.diagnostics.DiagnosticsCategory;
import freenet.support.*;
import freenet.support.sort.*;
import freenet.support.Comparable;
import freenet.fs.dir.FileNumber;

import java.util.Hashtable;
import java.util.Enumeration;
import java.io.PrintWriter;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU Public Licence (GPL) version 2.  See
  http://www.gnu.org/ for further details of the GPL.
*/

/**
 * Helper class to keep track of the data needed to
 * do intra-node load balancing.
 * @author giannij
 * @author oskar
 **/
public class LoadStats implements Checkpointed {

    private DataObjectStore table;
    private DoublyLinkedListImpl lru = new DoublyLinkedListImpl();
    private float totalGlobal = 0;

    private int maxTableSize;

    private Diagnostics diag;

    private volatile double meanGlobalTraffic;
    private volatile double resetProbability;
    private final double defaultResetProbability;

    private long lastCheck;

    private final Object timesLock = new Object();
    private long[] times;
    private int timesPos;
    private short ratio;
    
    /**
     * Create a new loadstats object.
     * @param maxTableSize  The maximum number of peers to store traffic from.
     * @param diag          The nodes diagnostics object. A category named
     *                      "Network Load" will be added with the following
     *                      fields:
     *                      Binomial          localQueryTraffic
     *                      Continuous        globalQueryTrafficMean
     *                      Continuous        globalQueryTrafficMedian
     *                      Continuous        globalQueryTrafficDeviation
     *                      Continuous        resetProbability
     *                      Binomial          resetRatio
     * @param parent        The category to make the parent of the 
     *                      new category, or null to leave it at the top.
     */
    public LoadStats(DataObjectStore table, int maxTableSize, Diagnostics diag,
                     DiagnosticsCategory parent, 
		     double defaultResetProbability) throws IOException {
        this.table = table;
        this.maxTableSize = maxTableSize;
        this.diag = diag;
	this.defaultResetProbability = defaultResetProbability;

        this.times = new long[100];
        this.timesPos = 0;
        this.ratio = 100;

        for (Enumeration e = table.keys(true) ; e.hasMoreElements(); ) {
            LoadEntry le = getLoadEntry((FileNumber) e.nextElement());
            lru.push(le);
        }

        DiagnosticsCategory traffic = 
            diag.addCategory("Network Load", 
                             "Measurements related to the local and global " +
                             "network load.", parent);

        diag.registerBinomial("localQueryTraffic", diag.MINUTE,
                              "The amount of queries received, and the " +
                              "number that are not rejected.", traffic);
        diag.registerContinuous("globalQueryTrafficMean", diag.HOUR,
                                "The mean traffic of the known peers, " + 
                                "measured regularly.",
                                traffic);
        diag.registerContinuous("globalQueryTrafficMedian", diag.HOUR,
                               "The median traffic of the known peers, " + 
                               "measured regularly.", traffic);
        diag.registerContinuous("globalQueryTrafficDeviation", diag.HOUR,
                                "The standard deviation in traffic of the " + 
                                "known peers, measured regularly.", traffic);
        diag.registerContinuous("resetProbability", diag.HOUR, 
                                "The probability of reseting the datasource "+
                                "of a reply to point to us if load " + 
                                "balancing is used.", traffic);
        diag.registerBinomial("resetRatio", diag.MINUTE,
                              "The actual ratio of times we actually do " +
                              "reset the DataSource to data responses.",
                              traffic);

        checkpoint();
    }

    
    /**
     * Call to increment the query count.
     * @boolean  Whether the query was initially accepted by the node.
     **/
    public final void receivedQuery(boolean accepted) {
        diag.occurrenceBinomial("localQueryTraffic", 1,  accepted ? 1 : 0);

        synchronized(timesLock) {
            if (times[timesPos] >= 0) // note =, since we start with 1
                ratio--;

            times[timesPos] = System.currentTimeMillis() * (accepted ? 1 : -1);
            timesPos = (timesPos + 1) % times.length;

            if (accepted)
                ratio++;
        }
    }


    /**
     * Call to update the table used to estimate global network load.
     **/
    public final synchronized void storeTraffic(NodeReference nr, 
                                                long requestsPerHour) {
        if (nr == null || (requestsPerHour == -1)) {
            // These cases are legal NOPs.
            return;
        }

        LoadEntry le = new LoadEntry(nr.getIdentity().fingerprint(),
                                     requestsPerHour);

        try {
            LoadEntry oldle = getLoadEntry(le.fn);

            if (oldle != null) {
                lru.remove(oldle);
            }

            table.set(le.fn, le);
            lru.push(le);
        
	    boolean old = insufficientData;
            // Don't let the table grow without bound.
            while (lru.size() > maxTableSize) {
                LoadEntry last = (LoadEntry) lru.shift();
                table.remove(last.fn);
            }
	    if (lru.size() >= maxTableSize)
		insufficientData = false;
	    if((!insufficientData) && old) {
		Core.logger.log(this, "Doing checkpoint, passed 100 queries",
				Logger.MINOR);
		checkpoint();
		Core.logger.log(this, "Done checkpoint, passed 100 queries",
				Logger.MINOR);
	    }
        } catch (IOException e) {
            // already logged
        }
    }
    
    boolean insufficientData = true;
    
    /**
     * @return The number of outbound requests per hour made from
     *         this node.
     */
    public final double localQueryTraffic() {
        long time;
        synchronized(timesLock) {
            time = times[timesPos];
        }
        if (time == 0)
            return 0;
        if (time < 0)
            time = time * -1;
        
        return (3600 * 1000 * 100.0) / 
            ((double) (System.currentTimeMillis() - time));
    }

    /**
     * @return An estimate of the global per node network 
     *         load in requests per hour.
     **/
    public final double globalQueryTraffic() {
        return meanGlobalTraffic;
    }

    /**
     * Rolls to see if the DataSource of a message should be reset.
     * This respects the setting of Node.doLoadBalance.
     */
    public final boolean shouldReset() {
        float p = Node.randSource.nextFloat();
        boolean b = ((Node.doLoadBalance && p < resetProbability) ||
                     (!Node.doLoadBalance && p < defaultResetProbability));
        if (b)
            Core.logger.log(this, "Telling a response to reset DataSource. " +
                            "Current probability " + resetProbability,
                            Core.logger.MINOR);
	
        diag.occurrenceBinomial("resetRatio", 1, b ? 1 : 0);
        return b;
    }

    /**
     * Calculates the probability of resetting the DataSource
     * that should be used when sending StoreData messages.
     * <p>
     * This is how the node does intra-node load balancing.
     * <p>
     * The current formula is as follows: Take
     * <p>
     * p_1 = min(r * M / m, 1)
     * <p>
     * where M is estimate of the global traffic, m is the local traffic,
     * and r is the mean reset ratio on the network. "r" is currently set
     * .05 since that is old unbalanced probability used by everyone, it 
     * should be more or less self fullfilling if everybody uses the same 
     * formula (*). We then take:
     * <p>
     * p_2 = a^5 * p_1
     * <p>
     * where a is the ratio of requests currently being accepted rather
     * then rejected because of load issues. This is so overloaded nodes
     * will advertise less.
     * <p>
     * Finally, we cut off the value from above and below to keep the 
     * network from degenerating in the worst case scenarios:
     * <p>
     * resetProbability = min(0.5, max(0.02, p_2));
     * <p>
     * (*) The justification for this formula (which isn't great) is that
     * if we imagine the state of the network taking discreet steps (with
     * all references replaced at every step), then the amount of traffic we 
     * get after one step should be the current traffic, times the probability
     * of reseting, times one over the chance of other nodes reseting (which 
     * gives the expected number of nodes that our reset will reach). So
     * if we want the traffic we get in one step to be the global mean, then
     * that gives the formula for p_1 above.
     **/
    public final synchronized double resetProbability() {
        return resetProbability;
    }

    /**
     * @return  "Calculate and report global load averages"
     */
    public String getCheckpointName() {
        return "Calculate and report global load averages";
    }

    /**
     * @return  10 min
     */
    public long nextCheckpoint() {
        return lastCheck + (10 * 60 * 1000);
    }

    /**
     * Execute a maintenance checkpoint.
     */
    public void checkpoint() {
	if(Core.logger.shouldLog(Core.logger.DEBUG))
	    Core.logger.log(this, "Executing checkpoint in LoadStats",
			    Core.logger.DEBUG);
        lastCheck = System.currentTimeMillis();
        LoadEntry[] les;
        synchronized(this) {
            les = new LoadEntry[lru.size()];
            int i = 0;
            for (Enumeration e = lru.elements(); e.hasMoreElements();i++) {
                les[i] = (LoadEntry) e.nextElement();
            }
        }

        QuickSorter.quickSort(new ArraySorter(les));

        // ignore some from the top and bottom.
        int ignore = Math.min(5, les.length / 10);

        double total = 0;
        double totalSquared = 0;
        for (int i = ignore ; i < les.length - ignore ; i++) {
            total += les[i].qph;
            totalSquared += les[i].qph * les[i].qph;
        }

        long n = les.length - 2 * ignore;
        double mean = n == 0 ? 0 : total / n;
        long median = (n == 0) ? 0 :
            (les.length % 2 == 1 ? les[les.length / 2].qph :
             ((les[les.length / 2 - 1].qph + les[les.length / 2].qph) / 2));
        double deviation = n <= 1  ? 0 : // bias corrected
            Math.sqrt(totalSquared / (n - 1.0) - 
                      (total * total) / (n * (n - 1.0)));
        
        double local = localQueryTraffic();
        double relProb = local == 0 ? 1 : Math.min(1, defaultResetProbability
						   * (mean / local));
        double acceptRatio;
	int x = 0;
        synchronized (timesLock) {
	    x = ratio;
        }
	acceptRatio = ((double)x)/100;
        // acceptRation ^ 2 * relProb with some bounds to avoid
	// degeneration.
        double prob = Math.min(Math.max((Double.isNaN(acceptRatio) ? 1 : 
                                         Math.pow(acceptRatio, 5)) 
                                        * relProb,
                                        0.02),
                               0.5);
        

        
        diag.occurrenceContinuous("globalQueryTrafficMean", mean);
        diag.occurrenceContinuous("globalQueryTrafficMedian", median);
        diag.occurrenceContinuous("globalQueryTrafficDeviation", deviation);
        diag.occurrenceContinuous("resetProbability", prob);
        
        meanGlobalTraffic = mean;
        resetProbability = prob;

        try {
            table.flush();
        } catch (IOException e) {
            Core.logger.log(this, "Error flushing load stats!", e, 
                            Logger.ERROR);
        }
	if(Core.logger.shouldLog(Core.logger.DEBUG))
	    Core.logger.log(this, "Finished executing checkpoint on loadStats",
			    Core.logger.DEBUG);
    }

    public final synchronized void dump(PrintWriter pw) {
        pw.println("# entries: " + lru.size());
        pw.println("# globalRequestsPerHour: " + globalQueryTraffic());
        pw.println("# localRequestsPerHour: " + localQueryTraffic());
        pw.println("# format: <requests per hour> <node fingerprint>");                   
        try {
            for (Enumeration e = table.keys(true) ; e.hasMoreElements() ;) {
                FileNumber key = (FileNumber) e.nextElement();
                long rate = getLoadEntry(key).qph;
                pw.println(rate + "\t" + "\"" +  key.toString() + "\"");
            }
        } catch (IOException e) {
            pw.println("Error reading data: ");
            pw.println();
            e.printStackTrace(pw);
        }
    }

    public final synchronized void dumpHtml(PrintWriter pw) {
        pw.println("<ul>");
        pw.println("<li> entries: " + lru.size() + "</li>");
        pw.println("<li> Global mean traffic (queries per hour):" 
                   + globalQueryTraffic() + "</li>");
        pw.println("<li> Local mean traffic (queries per hour): " 
                   + localQueryTraffic() + "</li>");
        pw.println("<li> Current advertise probability: " + resetProbability +
                   "</li>");
	double acceptRatio;
	int x = 0;
	synchronized(timesLock) {
	    x = ratio;
	}
	acceptRatio = ((double)x)/100;
	pw.println("<li> Current proportion of requests being accepted: "+
		   acceptRatio + "</li>");
        pw.println("</ul>");
        pw.println("<table border=1><tr><th>Queries per hour</th>" + 
                   "<th>Node fingerprint</th></tr>");
        try {
            for (Enumeration e = table.keys(true) ; e.hasMoreElements() ;) {
                FileNumber key = (FileNumber) e.nextElement();
                long rate = getLoadEntry(key).qph;
                pw.println("<tr><td>" + rate + "</td><td>" +  key.toString() 
                           + "<td></tr>");
            }
            pw.println("</table>");
        } catch (IOException e) {
            pw.println("</table><br><br>");
            pw.println("Error reading data!<br>");
            pw.println("<pre>");
            e.printStackTrace(pw);
            pw.println("</pre>");
        }
    }

    private LoadEntry getLoadEntry(FileNumber key) throws IOException {
        try {
            return (LoadEntry) table.get(key);
        } catch (DataObjectUnloadedException dp) {
            try {
                return new LoadEntry(dp.getDataInputStream());
            } catch (IOException e) {
		Core.logger.log(this, "Error restoring load stats", e, 
                                Logger.ERROR);
                throw e;
            }
        }
    }

    private class LoadEntry extends DoublyLinkedListImpl.Item 
        implements Comparable, DataObject {

        private FileNumber fn;
        private long qph;

        private LoadEntry(byte[] b, long qph) {
            this.fn = new FileNumber(b);
            this.qph = qph;
        }

        private LoadEntry(DataInputStream in) throws IOException {
            byte[] bs = new byte[in.readInt()];
            in.readFully(bs);
            fn = new FileNumber(bs);
            qph = in.readLong();
        }

        public int getDataLength() {
            return 8 + 4 + fn.getByteArray().length;
        }

        public void writeTo(DataOutputStream out) throws IOException {
            byte[] bs = fn.getByteArray();
            out.writeInt(bs.length);
            out.write(bs);
            out.writeLong(qph);
        }

        public int compareTo(Object o) {
            long qph2 = ((LoadEntry) o).qph;
            return qph < qph2 ? -1 : qph == qph2 ? 0 : 1;
        }
    }

}








