package freenet.node;

import freenet.Core;
import freenet.Key;
import freenet.support.Checkpointed;
import freenet.support.Heap;
import freenet.support.Heap.Element;
import freenet.support.sort.ArraySorter;
import freenet.support.sort.QuickSorter;


import java.util.Hashtable;
import java.util.Date;

import java.util.Enumeration;
import java.util.Random;

import java.io.PrintWriter;

/**
 * Keeps track of keys that have been failed recently, and automatically 
 * fails requests for that key if the hops to live is lesser or equal.
 */

public class FailureTable implements Checkpointed {

    // Todo: make this a test.

//      public static void main(String[] args) {
//          FailureTable ft = new FailureTable(100, 1800000);
//          Random r = new Random();
//          Key[] keys = new Key[16];
//          for (int i = 0 ; i < 16 ; i++) {
//              long time = System.currentTimeMillis() - r.nextInt(3600000);
//              byte[] keyval = new byte[10];
//              for (int j = 0 ; j < 10 ; j++) {
//                  keyval[j] = (byte) i;
//              }
//              keys[i] = new Key(keyval);
//              ft.failedToFind(keys[i], 10, time);
//          }
        
//          for (Enumeration e = ft.queue.elements() ; e.hasMoreElements() ;) {
//              System.err.println(e.nextElement());
//          }

//          System.err.println(ft.shouldFail(keys[0], 1));
//          ft.checkpoint();

//          System.err.println("---");

//          while (ft.queue.size() > 0) {
//              System.err.println(ft.queue.pop());
//          } 
//      }

    private int maxSize;
    private long maxMillis;
    private long lastCp = 0;

    private Hashtable failedKeys;
    private Heap queue;

    /**
     * Creates a new.
     * @param size     The number of entries to hold.
     * @param millis   The max amount of time to keep an entry.
     */
    public FailureTable(int size, long millis) {
        maxSize = size;
        maxMillis = millis;

        failedKeys = new Hashtable();
        queue = new Heap();
    }

    /**
     * Add a failure.
     * @param k            The key that could not retrieved.
     * @param hopsToLive   The hopsToLive when it could not be found.
     * @param time         The time at which it could not be retreived.
     */
    public synchronized void failedToFind(Key k, int hopsToLive,
                                          long time) {
        FailureEntry fe = (FailureEntry) failedKeys.get(k);
        if (fe == null) {
            fe = new FailureEntry(k, hopsToLive, time);
            failedKeys.put(k, fe);
        } else {
            fe.remove();
            if (hopsToLive > fe.hopsToLive) {
                fe.time = time;
                fe.hopsToLive = hopsToLive;
            }
        }

        queue.put(fe);
        while (queue.size() > maxSize) {
            fe = (FailureEntry) queue.pop();
            failedKeys.remove(fe.key);
            Core.diagnostics.occurrenceContinuous("failureTableBlocks",
                                                  (double) fe.blocks);
        }
    }

    /**
     * Checks whether a query should fail.
     * @return  The time at which the query that failed to find the key
     *          occured, or < 0 if no such query is known.
     */
    public synchronized long shouldFail(Key k, int hopsToLive) {
        FailureEntry fe = (FailureEntry) failedKeys.get(k);
        long time = System.currentTimeMillis();

        if (fe != null && fe.hopsToLive >= hopsToLive &&
            fe.time > (time - maxMillis)) {

            fe.blocks++;
            Core.diagnostics.occurrenceContinuous("timeBetweenFailedRequests",
                                                 (double)(time - fe.lastHit));
            fe.lastHit = time;
            return fe.time;
        } else
            return -1;
    }

    /**
     * @return true, if the Key is contained in the failuretable
     */
    public synchronized boolean contains(Key k) {
        return failedKeys.containsKey(k);
    }

    /**
     * Removes the entry for this key.
     */
    public synchronized void remove(Key k) {
        FailureEntry fe = (FailureEntry) failedKeys.remove(k);
        if (fe != null) {
            fe.remove();
            Core.diagnostics.occurrenceContinuous("failureTableBlocks",
                                                  (double) fe.blocks);
        }
    }

    /**
     * Purges the queue.
     */
    public synchronized void checkpoint() {
        lastCp = System.currentTimeMillis();
        long thresh = lastCp - maxMillis;
        while (queue.size() > 0) {
            FailureEntry fe = (FailureEntry) queue.top();
            if (fe.time > thresh) {
                break;
            }
            queue.pop();
            failedKeys.remove(fe.key);
            Core.diagnostics.occurrenceContinuous("failureTableBlocks",
                                                  (double) fe.blocks);
        }
    }

    public String getCheckpointName() {
        return "Purge table of recently failed keys.";
    }

    public long nextCheckpoint() {
        return Math.max(System.currentTimeMillis(),
                        lastCp + maxMillis / 10);
    }

    public synchronized void writeHtml(PrintWriter pw) {
        pw.println("<b>max size:</b> " + maxSize + "<br>");
        pw.println("<b>current size:</b> " + failedKeys.size() + "<br>");
        pw.println("<b>seconds it lasts:</b> " + maxMillis / 1000 + "<br>");
        pw.println("<table border=1>");
        pw.println("<tr><th>Key</th><th>Blocked HTL</th>" 
                   + "<th>Age</th><th># of Blocks</th>" 
                   + "<th>lastHit</th></tr>");

        Element[] elementArray = queue.elementArray();
        QuickSorter.quickSort(new ArraySorter(elementArray));

        long time = System.currentTimeMillis();
        for (int i = 0 ; i < elementArray.length ; i++) {
            FailureEntry fe = (FailureEntry) elementArray[i];
            boolean active = (time - fe.time) < maxMillis;
            pw.println("<tr><td><font color=\"" + (active ? "red" : "green") 
                       + "\">" + fe.key + "</font></td><td>" + fe.hopsToLive 
                       + "</td><td>" + (time - fe.time) / 1000 
                       + "</td><td>" + fe.blocks 
                       + "</td><td>" + new Date(fe.lastHit) + "</td></tr>");
        }
        pw.println("</table>");
    }


    private class FailureEntry extends Element {
        private int hopsToLive;
        private Key key;
        private long time;

        private int blocks;
        private long lastHit;

        public FailureEntry(Key key, int hopsToLive, long time) {
            this.hopsToLive = hopsToLive;
            this.key = key;
            this.time = time;
            this.lastHit = time;
            this.blocks = 0;
        }

        public int compareTo(Object o) {
            //FailureEntry fe = (FailureEntry) o;
            //return (int) (fe.time - time);
            // older is "bigger"
            long ot = ((FailureEntry) o).time;
            return time == ot ? 0 : (time > ot ? -1 : 1);
        }

        public String toString() {
            return key.toString() + " " + hopsToLive + " " + new Date(time);
        }

    }


}
