package freenet.message;

import java.io.*;
import java.util.*;
import freenet.*;
import freenet.node.*;
import freenet.node.ds.*;
import freenet.node.states.data.*;
import freenet.support.*;
import freenet.support.io.*;

/**
 * This is the DataSend message
 *
 * @see Node
 * @see Address
 * @author Brandon Wiley (blanu@uts.cc.utexas.edu)
 * @author Ian Clarke (I.Clarke@strs.co.uk)
 * @author oskar (fingered everything)
 */

public abstract class DataSend extends NodeMessage 
                               implements TrailingFieldMessage {

    /** The trailing field */
    private DiscontinueInputStream in;
    
    /** The length of the trailing field */
    private long length;
    
    private boolean logDEBUG;
    
    public DataSend(long idnum, FieldSet otherfields,
                    DiscontinueInputStream data, long length) {
        super(idnum, otherfields);
        this.in = data;
        this.length = length;
	logDEBUG = Core.logger.shouldLog(Logger.DEBUG);
	if(logDEBUG)
	    Core.logger.log(this, "DataSend constructed", new Exception("debug"),
			    Logger.DEBUG);
    }

    public DataSend(ConnectionHandler source, RawMessage raw)
        throws InvalidMessageException, BadAddressException {

        super(source, raw);

        if (raw.trailingFieldLength != 0) {
            length = raw.trailingFieldLength;
            in = raw.trailingFieldStream;
        } else 
            throw new InvalidMessageException("Data sending message requires "
                                              + "the trailing field length to "
                                              + "be specified");
	logDEBUG = Core.logger.shouldLog(Logger.DEBUG);
    }

    public RawMessage toRawMessage(Presentation t) {
        RawMessage raw=super.toRawMessage(t);
        
        // this was screwing the sending of DataReply because
        // the trailing field length wasn't getting filled in
        //if (in != null) {
            //raw.trailingFieldStream = in;
            raw.trailingFieldLength = length;
            raw.trailingFieldName="Data";
        //}

        return raw;
    }

    /**
     * This will drop the trailing field (and with it the connection it was
     * to be received on) if it has not already been read.
     */
    public void drop(Node n) {
	if(logDEBUG)
	    Core.logger.log(this, "Dropping "+this+" (in="+in+"), length "+length,
			    Logger.DEBUG);
        try {
            if (in != null)
                in.close();
        } catch (IOException e) {
	    if(logDEBUG)
		Core.logger.log(this, "Closing dropped DataSend threw exception :" 
				+ e,Logger.DEBUGGING);
        }
    }

    public InputStream getDataStream() {
        return in;
    }

    public Storables getStorables() {
        return otherFields == null ? null : Storables.readFrom(otherFields);
    }

    public long length() {
        return length;
    }

    public void length(long length) {
        this.length = length;
    }

    /**
     * Sets up the caching of the data in the this message to the nodes
     * datastore.
     *
     * @param  n            The node to cache into.
     * @param  searchKey    The key to cache as.
     * @return a DataState that will read the data into the store
     * @exception IOException  If creating the stream to CACHE the data fails.
     * @exception DataNotValidIOException  If the data does not validate
     *                                     for the provided key.
     */
    public ReceiveData cacheData(Node n, Key searchKey)
                                    throws IOException, DataNotValidIOException,
                                                        KeyCollisionException {
        Storables storables = getStorables();
        if (storables == null) {
            throw new DataNotValidIOException(Presentation.CB_BAD_KEY);
        }
        VerifyingInputStream vis = searchKey.verifyStream(in, storables, length);
	if(logDEBUG)
	    Core.logger.log(this, "Trying to cache data: "+searchKey+":"+length+
			    ":"+in, Core.logger.DEBUG);
        KeyOutputStream out = n.ds.putData(searchKey, length, storables);
        return new ReceiveData(n.randSource.nextLong(), this.id, vis, out, length);
    }

    /** Used to swallow a DataInsert.
      */
    public void eatData(Node n) {
        boolean eating = false;
        try {
            Storables storables = getStorables();
            if (storables == null) throw new IOException();
            VerifyingInputStream vin = new VerifyingInputStream(in, length);
            (new EatData(n.randSource.nextLong(), vin, length)).schedule(n);
            // FIXME -- should use ControlInputStream but i haven't made
            //          sure it will work yet
        }
        catch (IOException e) {
            try { in.close(); }
            catch (IOException e2) {}
        }
    }
}










