package freenet.interfaces;

import freenet.*;
import freenet.support.Logger;
import freenet.support.io.NIOInputStream;
import freenet.session.*;
import freenet.node.Main;
import freenet.transport.tcpConnection;
import java.io.*;
import java.nio.*;
import java.nio.channels.*;
import java.net.*;

/**
 * Runs connections using standard Freenet
 * session and presentation negotiation.
 */
public final class FreenetConnectionRunner implements ConnectionRunner {
    
    protected final Core core;
    protected SessionHandler sh;
    protected PresentationHandler ph;
    protected OpenConnectionManager ocm;
    protected int maxInvalid;


    /**
     * @param core        the Core containing the SessionHandler,
     *                    PresentationHandler, and the node's key pair
     * @param ocm         passed to the ConnectionHandler constructor
     * @param maxInvalid  passed to the ConnectionHandler constructor
     */
    public FreenetConnectionRunner(Core core,
                                   SessionHandler sh, PresentationHandler ph,
                                   OpenConnectionManager ocm, int maxInvalid) {
        this.core = core;
        this.sh = sh;
        this.ph = ph;
        this.ocm = ocm;
        this.maxInvalid = maxInvalid;
    }
    
    public void handle(Connection conn) {
	ConnectionHandler ch = null;
		boolean logDEBUG = core.logger.shouldLog(Logger.DEBUG); 
        try {
            Link l;
            Presentation p;
            
            conn.setSoTimeout(Core.authTimeout);
            InputStream raw = conn.getIn();
            int i = raw.read() << 8 | raw.read();
            if (i < 0)
                throw new EOFException();
            LinkManager lm = sh.get(i);
            if (lm == null) {
                throw new NegotiationFailedException(conn.getPeerAddress(),
                                                     "Unsupported link 0x"
                                                     + Integer.toHexString(i));
            }
            
            l = lm.acceptIncoming(core.privateKey, 
                                  core.identity, 
                                  conn);
            l.getOutputStream().flush();
	    // Or it won't get written at all
	    // FIXME: this should go with the first packet somehow
            InputStream crypt = l.getInputStream();
            i = crypt.read() << 8 | crypt.read();
            if (i < 0)
                throw new EOFException();
            p = ph.get(i);
            if (p == null) {
                throw new NegotiationFailedException(conn.getPeerAddress(),
                                                     l.getPeerIdentity(),
                                                     "Unsupported presentation 0x"
                                                     + Integer.toHexString(i));
            }
            conn.enableThrottle();
            ch = new ConnectionHandler(ocm, p, l, core.ticker(),
				       maxInvalid, core.maxPadding, false);
	    boolean success = false;
            try {
		if(logDEBUG) Core.logger.log(this,"starting transfer from NIOIS to CH",Logger.DEBUG);
		Socket sock = null;
		try {
		    sock = ((tcpConnection)conn).getSocket();
		} catch (IOException e) {
		    return;
		}
		NIOInputStream niois = (NIOInputStream) ((tcpConnection)conn).getUnderlyingIn();
		niois.setNextReader(ch);
		if(logDEBUG) Core.logger.log(this, "NIOIS was "+niois+" for "+conn+
				" (to be "+ch+")", Logger.DEBUG);
		tcpConnection.getRSL().unregister(niois);
		while(niois.isRegistered() && (!niois.alreadyClosedLink())) {
		    synchronized(niois.unregLock) {
			if(niois.isRegistered()) break;
			try{
			    niois.unregLock.wait(200);
			}catch(InterruptedException e) {
			    Core.logger.log(this,"couldn't complete unregistration of NIOIS",Logger.ERROR);
			    return;
			}
		    }
		}
		if(niois.alreadyClosedLink()) {
		    Core.logger.log(this, "Already closed link, not registering: "+
				    this, Logger.MINOR);
		    conn.close();
		    ch.terminate();
		    return;
		}
		// is now unregistered, it will not be checked until we are
		// registered
		if(logDEBUG) Core.logger.log(this, "NIOInputStream "+niois+" unregistered for "+this+":"+
				ch, Logger.DEBUG);
		/*try {
		  sc.configureBlocking(false);
		  } catch (IOException e) {
		  Core.logger.log(this, "Cannot configure nonblocking mode on SocketChannel!", Logger.ERROR);
		  }*/
		if(sock != null) {
		    // Some done by niois.unregister
// 		    ch.configRSL(tcpConnection.getRSL());
		    ch.configWSL(tcpConnection.getWSL());
		    ch.registerOCM();
// 		    tcpConnection.getRSL().register(sock, ch);
// 		    tcpConnection.getRSL().scheduleMaintenance(ch);
		} // else already closed
		//ch.run(); // we already have our own thread
		success = true;
	    } finally {
		if(!success) ch.terminate();
	    }
	} catch (IOException e) {
	    core.logger.log(this, "Inbound connection failed: "+e+" on "+conn,
			    e, Logger.MINOR);
	    conn.close();
	    if(ch != null) ch.terminate();
	} catch (CommunicationException e) {
	    core.logger.log(this, "Inbound connection failed: "+e+" on "+conn,
			    e, Logger.MINOR);
	    conn.close();
	    if(ch != null) ch.terminate();
	}
	if(logDEBUG) Core.logger.log(this, "Leaving handle("+conn+")", Logger.DEBUG);
    }
    
    public void starting() {};
    
    public boolean needsThread() { return true; }
}






