package freenet.fs.dir;

import freenet.fs.acct.Fragment;
import freenet.fs.acct.sys.AccountingTreeMarshal;
import freenet.support.*;
import freenet.support.Comparable;
import java.io.*;

/**
 * A ticket is basically a 64-bit unique ID for an allocation.  The ticket
 * keeps track of the allocated fragments along with an access timestamp and
 * file-number.  Only the timestamp of a ticket can change.  If fragments or
 * file-numbers must be reassigned, the old ticket must be freed and a new
 * one obtained.  Each ticket ID is unique throughout the lifetime of the
 * accounting system.
 * @author tavin
 */
public final class Ticket implements Comparable {

    static final class Marshal implements AccountingTreeMarshal {
        public final Comparable readEntry(DataInput din, int len)
                                            throws IOException {
            long ticketID = din.readLong();
            long timestamp = din.readLong();
            Fragment[] ranges = new Fragment[din.readUnsignedShort()];
            Fragment.parseRangeList(ranges, din);
            int dirID = din.readUnsignedShort();
            byte[] key = new byte[len - 20 - 16*ranges.length];
            din.readFully(key);
            return new Ticket(ticketID, timestamp, ranges, new FileNumber(dirID, key));
        }
        public final int getEntryLength(Comparable entry) {
            Ticket t = (Ticket) entry;
            return 20 + 16*t.ranges.length + t.fn.key.length;
        }
        public final void writeEntry(Comparable entry, DataOutput out)
                                                    throws IOException {
            Ticket t = (Ticket) entry;
            out.writeLong(t.ticketID);
            out.writeLong(t.timestamp);
            out.writeShort(t.ranges.length);
            Fragment.writeRangeList(t.ranges, out);
            out.writeShort(t.fn.dirID);
            out.write(t.fn.key);
        }
    }
    

    final long ticketID;
    
    public long timestamp;
    
    public final Fragment[] ranges;  // made public for the console servlets
    final FileNumber fn;

    
    Cacheable cel = null;  // LRU element for live tickets

    int users = 0;  // count active ticket-locks
    

    Ticket(long ticketID, long timestamp, Fragment[] ranges, FileNumber fn) {
        this.ticketID = ticketID;
        this.timestamp = timestamp;
        this.ranges = ranges;
        this.fn = fn;
    }

    Ticket(long ticketID) {
        this(ticketID, -1, null, null);
    }

    public final String toString() {
        return
            "#" + ticketID +
            " : " + timestamp +
            " : " + fn +
            " @ " + Fragment.rangeList(ranges);
    }

    public final int compareTo(Object o) {
        return compareTo((Ticket) o);
    }

    public final int compareTo(Ticket t) {
        return ticketID == t.ticketID ? 0
                                      : (ticketID > t.ticketID ? 1 : -1);
    }

/*
    public final boolean equals(Object o) {
        return o instanceof Ticket && equals((Ticket) o);
    }

    public final boolean equals(Ticket t) {
        return ticketID == t.ticketID;
    }

    public final int hashCode() {
        // FIXME:  smarter hashcode
        return ((int) (ticketID & 0xffffffffL)) ^ ((int) (ticketID >> 32));
    }
*/
}



