package freenet.fs.dir;

import freenet.fs.acct.*;
import freenet.fs.acct.Fragment.ComparatorBySize;
import freenet.fs.acct.Fragment.ComparatorByPos;
import freenet.fs.acct.sys.*;
import freenet.support.*;
import freenet.support.Comparable;
import freenet.support.BinaryTree.Node;
import java.util.Vector;
import java.io.*;

/**
 * Maintains a set of position-ordered fragments with the ticket IDs they
 * are allocated to.  Used in managing allocations and consistency checks.
 *
 * The entries in this map are allocated fragments.  Free fragments are
 * virtual and implied by the gaps between allocated fragments.  It is
 * like a photo negative of the FragmentSizeMap (with different ordering).
 *
 * @see FragmentSizeMap
 * @author tavin
 */
final class FragmentPositionMap extends FragmentMap {

    /**
     * @param proc   the shared accounting process for the FSDirectory
     * @param cache  the shared accounting block cache
     */
    FragmentPositionMap(AccountingProcess proc, Cache cache) {
        super(proc, new FragmentPos.Marshal(), cache);
    }

    
    /**
     * Finds allocated fragments overlapping the query fragment.
     * @return  any fragments overlapping the query fragment
     *          (may be an empty array, but not null)
     */
    FragmentRecord[] probeAlloc(Fragment f) {
        Node n = treeMatch(new FragmentPos(f), -1);
        if (n == null) {
            return new FragmentRecord[0];
        }
        synchronized (probeVec) {
            try {
                do {
                    FragmentRecord frec = (FragmentRecord) n.getObject();
                    if (f.getUpperBound() < frec.getLowerBound())
                        break;
                    if (f.getLowerBound() <= frec.getUpperBound())
                        probeVec.addElement(frec);
                    n = treeSuccessor(n);
                }
                while (n != null);
                
                FragmentRecord[] ret = new FragmentRecord[probeVec.size()];
                probeVec.copyInto(ret);
                return ret;
            }
            finally {
                probeVec.removeAllElements();
            }
        }
    }

    private final Vector probeVec = new Vector();
    
    

    /**
     * @return  the virtual free fragment enclosing the query fragment,
     *          or null if the query fragment overlaps with any
     *          allocated fragments
     */
    Fragment probeFree(Fragment f) {
        
        long xlo = 0;
        long xhi = Long.MAX_VALUE - 1;
        
        Node n = treeMatch(new FragmentPos(f), -1);
        while (n != null) {
            Fragment frec = (Fragment) n.getObject();
            if (f.getUpperBound() < frec.getLowerBound()) {
                xhi = frec.getLowerBound() - 1;
                break;
            }
            if (f.getLowerBound() <= frec.getUpperBound()) {
                return null;
            }
            xlo = frec.getUpperBound() + 1;
            n = treeSuccessor(n);
        }
        
        return new Fragment(xlo, xhi);
    }
    
    /**
     * @return  the virtual free fragment that existed
     *          before the allocation, or null if the
     *          allocation was unsuccessful
     */
    Fragment allocate(Fragment f, long ticketID) {
        
        long xlo = 0;
        long xhi = Long.MAX_VALUE - 1;
        
        Node n = treeMatch(new FragmentPos(f), -1);
        while (n != null) {
            Fragment frec = (Fragment) n.getObject();
            if (f.getUpperBound() < frec.getLowerBound()) {
                xhi = frec.getLowerBound() - 1;
                break;
            }
            if (f.getLowerBound() <= frec.getUpperBound()) {
                return null;
            }
            xlo = frec.getUpperBound() + 1;
            n = treeSuccessor(n);
        }
        
        n = new AccountingTreeNode(new FragmentPos(f, ticketID));
        treeInsert(n, false);
        return new Fragment(xlo, xhi);
    }
        

    /**
     * @return  the virtual free fragment that is restored
     *          when the input fragment is freed, or null
     *          if no fragment could be freed
     */
    Fragment free(Fragment f) {
        Node n = treeSearch(new FragmentPos(f));
        if (n == null) {
            return null;
        }
        else {
            long xlo = 0;
            long xhi = Long.MAX_VALUE - 1;
            
            Node nlo = treePredecessor(n);
            Node nhi = treeSuccessor(n);
            
            if (nlo != null)
                xlo = ((Fragment) nlo.getObject()).getUpperBound() + 1;
            
            if (nhi != null)
                xhi = ((Fragment) nhi.getObject()).getLowerBound() - 1;
            
            treeRemove(n);
            
            return new Fragment(xlo, xhi);
        }
    }


    
    //
    // Fragment -> 64-bit ticket ID mapping
    //

    private static final class FragmentPos extends FragmentRecord {
    
        private static final class Marshal implements AccountingTreeMarshal {
            public final Comparable readEntry(DataInput din, int len)
                                                throws IOException {
                return new FragmentPos(din.readLong(),
                                       din.readLong(),
                                       din.readLong());
            }
            public final int getEntryLength(Comparable entry) {
                return 24;
            }
            public final void writeEntry(Comparable entry, DataOutput out)
                                                        throws IOException {
                FragmentPos f = (FragmentPos) entry;
                out.writeLong(f.getLowerBound());
                out.writeLong(f.getUpperBound());
                out.writeLong(f.ticketID);
            }
        }

        
        FragmentPos(Fragment f) {
            super(f);
        }
        
        FragmentPos(Fragment f, long ticketID) {
            super(f, ticketID);
        }
        
        FragmentPos(long lo, long hi, long ticketID) {
            super(lo, hi, ticketID);
        }
        
        public final int compareTo(Object o) {
            return compareTo((FragmentPos) o);
        }

        public final int compareTo(FragmentPos f) {
            return ComparatorByPos.compare(this, f);
        }
    }
}


