package freenet.fs.dir;

import freenet.fs.*;
import freenet.fs.acct.Fragment;
import freenet.crypt.Digest;
import freenet.support.Fields;
import java.io.*;

/**
 * Two slots are set aside at the beginning of the storage.
 * @author tavin
 */
public class FSDirectoryRoot {

    public static final long VERSION = 3;
    
    
    long rootNumber;
    Fragment[] ranges;
    
    
    /**
     * Reads the directory root blocks.
     */
    public FSDirectoryRoot(LockGrantor lg, Digest ctx, int slotSize) throws IOException {
        this.rootNumber = -2;
        do {   
            long pos = ((2+this.rootNumber) % 2) * slotSize;
            InputStream in = ReadLock.getInputStream(lg, pos, pos + slotSize - 1);            
            try {
                byte[] checksum = new byte[ctx.digestSize() >> 3];
                byte[] data = new byte[slotSize - checksum.length];
                
                PushbackInputStream pin = new PushbackInputStream(in, data.length);
                DataInputStream din = new DataInputStream(pin);

                din.readFully(checksum);
                din.readFully(data);
 
                ctx.update(data);
                if (Fields.byteArrayEqual(ctx.digest(), checksum)) {
                    pin.unread(data);
                    long rootNumber = din.readLong();
                    long version = din.readLong();
                    if (version == 1 || version == 2) {
                        // treat as if checksum failed (reset the store)
                    }
                    else if (version != VERSION) {
                        throw new DirectoryException("cannot parse root version "+version);
                    }
                    else if (this.rootNumber <= rootNumber) {
                        this.rootNumber = rootNumber;
                        this.ranges = new Fragment[din.readUnsignedShort()];
                        Fragment.parseRangeList(this.ranges, din);
                    }
                }
            }
            finally {
                in.close();
            }
        }
        while (this.rootNumber++ % 2 == 0);
    }

    /**
     * Writes the directory root blocks.
     */
    FSDirectoryRoot(LockGrantor lg, Digest ctx, int slotSize,
                    long rootNumber, Fragment[] ranges) throws IOException {

        long pos = (rootNumber % 2) * slotSize;
        OutputStream out = WriteLock.getOutputStream(lg, pos, pos + slotSize - 1);
        try {
            ByteArrayOutputStream bout = new ByteArrayOutputStream();
            DataOutputStream dout = new DataOutputStream(bout);
            dout.writeLong(rootNumber);
            dout.writeLong(VERSION);

            // FIXME - okay, technically this could exceed the end of the block
            dout.writeShort(ranges.length);
            Fragment.writeRangeList(ranges, dout);
            
            byte[] data = new byte[slotSize - (ctx.digestSize() >> 3)];
            System.arraycopy(bout.toByteArray(), 0, data, 0, bout.size());
            ctx.update(data);
            
            out.write(ctx.digest());
            out.write(data);
        }
        finally {
            out.close();
        }

        this.rootNumber = 1 + rootNumber;
        this.ranges = ranges;
    }

    /**
     * @return  the next value that should be used
     *          as the root number
     */
    public final long getNextRootNumber() {
        return rootNumber;
    }

    /**
     * @return  the list of Fragments recorded in the slot
     */
    public final Fragment[] getRanges() {
        return ranges;
    }
}


