package freenet.fs.acct.sys;

import freenet.fs.acct.*;
import freenet.support.*;
import freenet.support.BinaryTree.Node;
import freenet.support.Comparable;
import java.util.Vector;
import java.io.*;


/**
 * A red-black tree that can serialize and deserialize itself using
 * a simple binary language.  There is a 1-1 relationship between these
 * and the accounting blocks in the AccountingTree.
 * @see freenet.fs.acct.sys.AccountingTree
 * @author tavin
 */
public class SerialTree extends RedBlackTree {

    /**
     * @return  a descriptive name for the command code
     *          designated by the given number
     */
    public static final String getNameFor(int command) {
        switch (command) {
            case STOP:      return "STOP";
            case STORE:     return "STORE";
            case DELETE:    return "DELETE";
            default:        return "(unknown)";
        }
    }

    /**
     * Prints out a verbose, human-readable description of the commands
     * and associated records supplied from the input stream.
     */
    public static final void dumpRecords(DataInput din, PrintWriter out)
                                                    throws IOException {
        while (true) {
            int cmd;
            try {
                cmd = din.readUnsignedShort();
                if (cmd == STOP)
                    break;
            }
            catch (EOFException e) {
                break;
            }
        
            out.print("Command: 0x");
            out.print(Integer.toHexString(cmd));
            out.print(" (" + getNameFor(cmd) + ")");

            out.print("\tElement: "+din.readUnsignedShort());

            if (cmd == STORE) {
                int len = din.readUnsignedShort();
                out.println("\tLength: "+len);
                byte[] data = new byte[len];
                din.readFully(data);
                out.println("Record: "+Fields.bytesToHex(data));
            }
            else out.println();
        }
    }
    

    /** the command codes */    
    public static final int STOP   = 0,
                            STORE  = 1,
                            DELETE = 2;
    

                            

    private final AccountingTreeBlock superTreeNode;
    private final AccountingTreeMarshal marshal;

    private BlockTransaction btx;

    private BlockList freeCells = new BlockList();
    private int nextID = 0;

    private int serialWidth = 0;

    
    boolean retired = false;
    
    
    /**
     * Create a new, empty SerialTree (for a new accounting block).
     * @param superTreeNode  the parent block-node in the accounting tree
     * @param marshal  the application-level marshalling routine
     * @param btx  the block transaction object representing the
     *             creation of this block
     */
    SerialTree(AccountingTreeBlock superTreeNode,
               AccountingTreeMarshal marshal,
               BlockTransaction btx) {
        
        this.superTreeNode = superTreeNode;
        this.marshal = marshal;
        this.btx = btx;
    }

    /**
     * Load an existing SerialTree from the intermediate Vector used
     * from parseIntoVector().
     * @param superTreeNode  the parent block-node in the accounting tree
     * @param marshal  the application-level marshalling routine
     * @param nodes  the Vector filled with tree nodes by parseIntoVector()
     * @param btx  the in-progress transaction associated with this block
     *             (may be null if there isn't one)
     * @throws AccountingException
     *         if there is a collision inserting the nodes into the tree
     * @see parseIntoVector()
     */
    SerialTree(AccountingTreeBlock superTreeNode,
               AccountingTreeMarshal marshal,
               Vector nodes, BlockTransaction btx) {
        
        this(superTreeNode, marshal, btx);

        // todo: optimize tree filling algo

        for (int i=0; i<nodes.size(); ++i) {
            if (nodes.elementAt(i) == null) {
                freeCells.push(i);
            }
            else {
                AccountingTreeNode atn = (AccountingTreeNode) nodes.elementAt(i);
                if (null != super.treeInsert(atn, false)) {
                    //Node old = super.treeInsert(atn, false);
                    //System.err.println("old = " + old.getObject());
                    //System.err.println("new = " + atn.getObject());
                    throw new AccountingException("duplicate entry at #" + i +
                                                  " in " + this.superTreeNode);
                }
                atn.owner = this;
                serialWidth += 6 + marshal.getEntryLength(atn.getObject());
            }
        }
        nextID = nodes.size();

        superTreeNode.setKey(nextID > 0 ? treeMin().getObject() : null);
    }

    /**
     * Reads commands supplied from the input stream and converts into
     * nodes for the red-black tree.  They are stored in an intermediate
     * Vector that is afterwards passed to the constructor.
     * @param marshal   the application-level marshalling routine
     * @param din       the serialized data source
     * @param nodes     intermediate storage for the deserialized tree nodes
     * @param superTreeNode  the parent block in the accounting tree
     *                       (only used for error reporting; may be null)
     * @throws AccountingException
     *         if an unknown command code is encountered
     */
    static void parseIntoVector(AccountingTreeMarshal marshal, DataInput din,
                                Vector nodes, AccountingTreeBlock superTreeNode)
                                                            throws IOException {
        int cmd;
        while (true) {
            try {
                cmd = din.readUnsignedShort();
                if (cmd == STOP)
                    return;
            }
            catch (EOFException e) {
                return;
            }
            
            if (cmd == STORE) {
                int num = din.readUnsignedShort();
                if (nodes.size() < num + 1) {
                    nodes.setSize(num + 1);
                }
                int len = din.readUnsignedShort();
                Comparable c = marshal.readEntry(din, len);
                nodes.setElementAt(new AccountingTreeNode(c, num), num);
            }
            else if (cmd == DELETE) {
                int num = din.readUnsignedShort();
                nodes.setElementAt(null, num);
            }
            else throw new AccountingException("no such command code: 0x"
                                               + Integer.toHexString(cmd)
                                               + " in " + superTreeNode);
        }
    }


    /**
     * Called when this SerialTree is to be dropped from memory.
     * Breaks circular ref loops.
     */
    final void retire() {
        retired = true;
        AccountingTreeNode treeRoot = (AccountingTreeNode) treeRoot();
        if (treeRoot != null)
            treeRoot.retire();
    }


    /**
     * @return  the byte length of the serialized form of this SerialTree
     */
    final int getSerialWidth() {
        return serialWidth;
    }


    // todo: optimize tree dumping algo
    
    /**
     * Writes the serialized form of this SerialTree.
     * @param out  destination of the serialized data
     */
    void freeze(DataOutput out) throws IOException {
        freeCells = new BlockList();
        nextID = 0;
        Walk walk = treeWalk(true);
        AccountingTreeNode atn;
        while (null != (atn = (AccountingTreeNode) walk.getNext())) {
            out.writeShort(STORE);
            out.writeShort(atn.num = nextID++);
            out.writeShort(marshal.getEntryLength(atn.getObject()));
            marshal.writeEntry(atn.getObject(), out);
        }
        btx = null;
    }


    /**
     * @return  the parent block from the accounting tree
     */
    final AccountingTreeBlock superTreeNode() {
        return superTreeNode;
    }

    /**
     * @return  the in-progress transaction (may be null)
     */
    final BlockTransaction transaction() {
        return btx;
    }

    
    private final void annotate(byte[] annot) {
        if (btx == null)
            btx = superTreeNode.transact();
        btx.annotate(annot);
    }



    private final byte[] logStore(int num, Comparable entry) {
        try {
            int len = marshal.getEntryLength(entry);
            ByteArrayOutputStream bout = new ByteArrayOutputStream(6 + len);
            DataOutputStream dout = new DataOutputStream(bout);
            dout.writeShort(STORE);
            dout.writeShort(num);
            dout.writeShort(len);
            marshal.writeEntry(entry, dout);
            dout.close();
            return bout.toByteArray();
        }
        catch (IOException e) {
            // impossible
            throw new RuntimeException(""+e);
        }
    }

    private final byte[] logDelete(int num) {
        try {
            ByteArrayOutputStream bout = new ByteArrayOutputStream(4);
            DataOutputStream dout = new DataOutputStream(bout);
            dout.writeShort(DELETE);
            dout.writeShort(num);
            dout.close();
            return bout.toByteArray();
        }
        catch (IOException e) {
            // impossible
            throw new RuntimeException(""+e);
        }
    }
    
    
    private void store(AccountingTreeNode atn) {
        atn.num = (freeCells.count() > 0 ? freeCells.pop() : nextID++);
        byte[] record = logStore(atn.num, atn.getObject());
        annotate(record);
        serialWidth += record.length;
//        freenet.Core.logger.log(this,
//                                "STORE # " + atn.num
//                                + " len=" + record.length
//                                + " data=" + Fields.bytesToHex(record)
//                                + " in " + superTreeNode,
//                                Logger.DEBUG);
    }

    private void delete(AccountingTreeNode atn) {
//        freenet.Core.logger.log(this,
//                                "DELETE # " + atn.num
//                                + " in " + superTreeNode,
//                                Logger.DEBUG);
        byte[] record = logDelete(atn.num);
        annotate(record);
        serialWidth -= 6 + marshal.getEntryLength(atn.getObject());
        freeCells.push(atn.num);
        atn.num = -1;
    }

    
    

    public final Node treeInsert(RBNode n, boolean replace) {
        AccountingTreeNode atn = (AccountingTreeNode) n;
        AccountingTreeNode old = (AccountingTreeNode) super.treeInsert(atn, replace);
        if (old == null || replace) {
            if (old != null) {
                delete(old);
                old.owner = null;
            }
            store(atn);
            atn.owner = this;
            if (superTreeNode.getObject() == null
                || atn.getObject().compareTo(superTreeNode.getObject()) < 0)
                { superTreeNode.setKey(atn.getObject()); }
        }
        return old;
    }
    
    public final boolean treeRemove(RBNode n) {
        if (super.treeRemove(n)) {
            AccountingTreeNode atn = (AccountingTreeNode) n;
            delete(atn);
            atn.owner = null;
            return true;
        }
        return false;
    }
}


