package freenet.fs.acct;

import java.io.*;

/**
 * Used to achieve an atomic change to the accounting table when
 * more than one block is being written or erased at once.
 * @author tavin
 */
class SynchronizationBlock extends ControlBlock {
    
    private final int[] newBlocks;
    private final int[] oldBlocks;

    SynchronizationBlock(int bnum, int antecedent, int[] newBlocks, int[] oldBlocks) {
        super(bnum, SYNCHRONIZATION_BLOCK, antecedent);
        this.newBlocks = newBlocks;
        this.oldBlocks = oldBlocks;
    }

    SynchronizationBlock(int bnum, int antecedent, DataInput din) throws IOException {
        super(bnum, SYNCHRONIZATION_BLOCK, antecedent);
        int n;
        n = din.readUnsignedShort();
        newBlocks = new int[n];
        for (int i=0; i<n; ++i) {
            newBlocks[i] = din.readInt();
        }
        n = din.readUnsignedShort();
        oldBlocks = new int[n];
        for (int i=0; i<n; ++i) {
            oldBlocks[i] = din.readInt();
        }
    }

    public String toString() {
        
        StringBuffer sb = new StringBuffer(super.toString());
        
        sb.append(" New: ");
        if (newBlocks.length > 0) {
            sb.append(newBlocks[0]);
            for (int i=1; i<newBlocks.length; ++i)
                sb.append(',').append(newBlocks[i]);
        }
        else sb.append("none");
        
        sb.append(" Old: ");
        if (oldBlocks.length > 0) {
            sb.append(oldBlocks[0]);
            for (int i=1; i<oldBlocks.length; ++i)
                sb.append(',').append(oldBlocks[i]);
        }
        else sb.append("none");

        return sb.toString();
    }
    

    final int[] getNewBlocks() {
        return newBlocks;
    }

    final int[] getOldBlocks() {
        return oldBlocks;
    }

    /**
     * @return  the number of bytes needed to serialize
     *          the synchronization record
     */
    final int getBodySize() {
        return 4 + 4 * (newBlocks.length + oldBlocks.length);
    }

    void writeBody(DataOutput out) throws IOException {
        out.writeShort(newBlocks.length);
        for (int i=0; i<newBlocks.length; ++i) {
            out.writeInt(newBlocks[i]);
        }
        out.writeShort(oldBlocks.length);
        for (int i=0; i<oldBlocks.length; ++i) {
            out.writeInt(oldBlocks[i]);
        }
    }
}



