package freenet.fs.acct;

import java.io.*;

/**
 * @author tavin
 */
public class SharedAccountingProcess implements AccountingProcess {

    public static final int TAG_WIDTH = 2;
    
    
    private final int tag;
    private final byte[] tagBytes;
    private final SingleAccountingProcess proc;
    
    public SharedAccountingProcess(int tag, SingleAccountingProcess proc) {
        this.tag = tag;
        this.proc = proc;
        tagBytes = new byte[] { (byte) ((tag >> 8) & 0xff),
                                (byte) (tag & 0xff) };
    }

    public final String toString() {
        return "0x"+Integer.toHexString(tag);
    }

    public final int getDataWidth() {
        return proc.getDataWidth() - TAG_WIDTH;
    }

    public final int getBlockCount() {
        return proc.getBlockCount();
    }

    public final int getUseCount() {
        return proc.getUseCount();
    }

    public final int getFreeCount() {
        return proc.getFreeCount();
    }

    public final DataInput resume(BlockTransaction btx, BlockEntry be) {
        DataInput din = proc.resume(btx, new TagWrapper(be));
        try {
            din.skipBytes(2);
        }
        catch (IOException e) {}  // yeah right
        return din;
    }

    public final BlockTransaction create(BlockEntry be) {
        BlockTransaction btx = proc.create(new TagWrapper(be));
        btx.annotate(tagBytes);
        return btx;
    }
    
    public final BlockTransaction modify(int bnum, BlockEntry be) {
        BlockTransaction btx = proc.modify(bnum, new TagWrapper(be));
        btx.annotate(tagBytes);
        return btx;
    }

    public final void abandon(BlockTransaction btx) {
        proc.abandon(btx);
    }
    
    public final void retire(int bnum) {
        proc.retire(bnum);
    }
    
    public final DataInput load(int bnum) throws IOException {
        DataInput din = proc.load(bnum);
        if (din != null)
            din.skipBytes(2);
        return din;
    }


    private final class TagWrapper implements BlockEntry {
        
        private final BlockEntry be;
        
        TagWrapper(BlockEntry be) {
            this.be = be;
        }
        
        public final void freeze(int dest, DataOutput out) throws IOException {
            out.writeShort(tag);
            be.freeze(dest, out);
        }
    }
}


