package freenet.fs.acct;

import freenet.fs.FileSystem;
import freenet.fs.*;
import freenet.crypt.*;
import freenet.config.*;
import java.io.*;

/**
 * Tools for using a file as a dedicated database
 * for an accounting table.
 * @author tavin
 */
public class AccountingFile {

    public static final int BLOCK_WIDTH = 10000;
    public static final String DIGEST = "SHA1";
    
    protected static final Config switches = new Config();
    static {
        switches.addOption ("width",  'w', 1, BLOCK_WIDTH, 10);
        switches.addOption ("digest", 'd', 1, DIGEST,      20);
        
        switches.argDesc   ("width", "<int>");
        switches.shortDesc ("width", "accounting block width");

        switches.argDesc   ("digest", "<name>");
        switches.shortDesc ("digest", "digest algorithm for checksums");
    }

    public static void main(String[] args) throws Exception {
        new AccountingFile().mainImpl(args);
    }

    public void mainImpl(String[] args) throws Exception {
        
        Params params = new Params(switches.getOptions());
        params.readArgs(args);
    
        try {
            if (params.getNumArgs() < 1)
                throw new IllegalArgumentException();

            File file = new File(params.getArg(0));
            int blockWidth = params.getInt("width");
            Digest ctx = Util.getDigestByName(params.getString("digest"));
    
            FileSystem fs = new FileSystem(new UnlimitedRAFStorage(file));
            Fragment f = new Fragment(fs.size());
            AccountingTable acct = new AccountingTable(fs, new Fragment[] {f},
                                                       ctx, blockWidth);
        
            handle(params, acct);
        }
        catch (IllegalArgumentException e) {
            usage();
            System.exit(1);
        }
    }
    
    protected void handle(Params params, AccountingTable acct) throws Exception {
            
        String cmd = params.getArg(1);
        if (cmd == null)
            cmd = "";

        if (cmd.equals("extract")) {
            int bnum = -1;
            try {
                bnum = Integer.parseInt(params.getArg(2));
            }
            catch (RuntimeException e) {
                // catch NPE or NFE
                throw new IllegalArgumentException();
            }
            byte[] data = acct.getBlock(bnum);
            if (data != null) {
                System.out.write(data);
                System.out.flush();
            }
        }
        else if (cmd.equals("scan")) {
            AccountingInitializer init = new AccountingInitializer(acct);
            PrintWriter out = new PrintWriter(System.out);
            AccountingInitializer.dump(init, out);
            //init.initialize(new AccountingInitializerDump(out));
            out.flush();
        }
        else throw new IllegalArgumentException();
    }

    protected void usage() throws IOException {
        System.out.println("Usage: java " + AccountingFile.class.getName() +
                           " <file> [<command>]");
        switches.printUsage(System.out);
    }
}


