package freenet.fs;

import java.io.*;

/**
 * A file-system inside a single file, or a group of files
 * logically concatenated into one.  Each file has to be
 * the same size.
 */
public class RAFStorage implements Storage {

    private final File[] files;
    private final long sz;

    private final long trunc;
    
    
    /**
     * A RAFStorage using only one file.
     * @see RAFStorage(File[], long)
     */
    public RAFStorage(File file, long size) {
        this(new File[] {file}, size);
    }
    
    /**
     * A RAFStorage using a group of files.
     * @param files  the collection of Files to use
     * @param sz     size of each individual file
     */
    public RAFStorage(File[] files, long sz) {
        this.files = files;
        this.sz = sz;
	// use a tmp as trunc is final and should only be set once, don't know why 
	// jikes/J2DK accept it anyway but gcj needs it to be stricter
	long tmp = size(); 
        for (int i=0; i<files.length; ++i) {
            if (files[i].length() < sz) {
                tmp = i*sz + files[i].length();
                break;
            }
            if (files[i].length() > sz) {
                tmp = (i+1)*sz;
                break;
            }
        }
        trunc = tmp;
    }
    
    public final long size() {
        return sz * files.length;
    }

    public final long truncation() {
        return trunc;
    }

    public final InputStream getInputStream(long start, long end) throws IOException {
        return new FSInputStream(start);
    }

    public final OutputStream getOutputStream(long start, long end) throws IOException {
        return new FSOutputStream(start);
    }

        
    //=== FSInputStream ========================================================

    private class FSInputStream extends InputStream {

        private RandomAccessFile raf;
        private long bx;  // bytes left to go in current file
        private int fx;   // current index into files array
        
        private FSInputStream(long pos) throws IOException {
            fx = (int) (pos / sz);
            bx = sz - pos % sz;
            raf = new RandomAccessFile(files[fx], "r");
            raf.seek(pos % sz);
        }

        public int read() throws IOException {
            if (bx == 0 && !next())
                return -1;
            int rv = raf.read();
            if (rv != -1 && 0 == --bx)
                raf.close();
            return rv;
        }

        public int read(byte[] buf, int off, int len) throws IOException {
            if (bx == 0 && !next())
                return -1;
            int rv = raf.read(buf, off, (int) Math.min(len, bx));
            if (rv != -1 && 0 == (bx -= rv))
                raf.close();
            return rv;
        }
        
        public void close() throws IOException {
            if (bx > 0)
                raf.close();
            bx = 0;
            fx = files.length;
        }

        protected void finalize() throws IOException {
            close();
        }

        private final boolean next() throws IOException {
            if (++fx >= files.length)
                return false;
            raf = new RandomAccessFile(files[fx], "r");
            bx = sz;
            return true;
        }
    }

    
    //=== FSOutputStream =======================================================
    
    private class FSOutputStream extends OutputStream {

        private RandomAccessFile raf;
        private long bx;  // bytes left to go in current file
        private int fx;   // current index into files array
        
        private FSOutputStream(long pos) throws IOException {
            fx = (int) (pos / sz);
            bx = sz - pos % sz;
            raf = new RandomAccessFile(files[fx], "rw");
            raf.seek(pos % sz);
        }

        public void write(int b) throws IOException {
            if (bx == 0 && !next())
                throw new EOFException();
            raf.write(b);
            if (--bx == 0) {
                raf.getFD().sync();
                raf.close();
            }
        }

        public void write(byte[] buf, int off, int len) throws IOException {
            while (len > 0) {
                if (bx == 0 && !next())
                    throw new EOFException();
                int x = (int) Math.min(len, bx);
                raf.write(buf, off, x);
                off += x;
                len -= x;
                if ((bx -= x) == 0) {
                    raf.getFD().sync();
                    raf.close();
                }
            }
        }

        public void flush() throws IOException {
            if (bx > 0)
                raf.getFD().sync();
        }

        public void close() throws IOException {
            if (bx > 0) {
                raf.getFD().sync();
                raf.close();
            }
            bx = 0;
            fx = files.length;
        }

        protected void finalize() throws IOException {
            close();
        }

        private final boolean next() throws IOException {
            if (++fx >= files.length)
                return false;
            raf = new RandomAccessFile(files[fx], "rw");
            bx = sz;
            return true;
        }
    }
}


