package freenet.fs;

import java.io.*; 

/**
 * A LockedOutputStream chains together a group of write locks
 * for writing as a single output stream.
 * @author tavin
 */
public class LockedOutputStream extends FilterOutputStream {

    protected WriteLock[] locks;
    protected int lx = 0;

    protected boolean closed = false;


    /**
     * Create a locked output stream that writes up to the end
     * of an array of write locks.
     * @param locks  the array of write locks
     */
    public LockedOutputStream(WriteLock[] locks) throws IOException {
        super(locks[0].getStream());
        this.locks = locks;
    }
    

    public void write(int b) throws IOException {
        fixate();
        out.write(b);
    }

    public void write(byte[] buf, int off, int len) throws IOException {
        while (len > 0) {
            fixate();
            int n = (int) Math.min(len, locks[lx].getBytesLeft());
            out.write(buf, off, n);
            off += n;
            len -= n;
        }
    }

    public void close() throws IOException {
        if (!closed) {
            closed = true;
            try {
                out.close();
            }
            finally {
                while (lx < locks.length)
                    locks[lx++].unlock();
            }
        }
    }

    private void fixate() throws IOException {

        if (closed)
            throw new IOException("stream closed");
        else if (lx >= locks.length)
            throw new EOFException();
        else if (locks[lx].getBytesLeft() > 0)
            return;

        out.close();
        locks[lx].unlock();
        if (++lx >= locks.length)
            throw new EOFException();
        out = locks[lx].getStream();
    }

    protected void finalize() throws Throwable {
        close();
    }
}


