package freenet.fs;

import freenet.support.Fields;

/**
 * A LockSignal implementation used by locks to block, and wake up
 * only when the byte pointers of all locks blocked on have reached
 * a certain minimum value, or unlocked.
 * @author tavin
 */
public class LockSlide implements LockSignal {

    private final static class SlideTuple {
        
        SlideTuple next = null;
        
        final LockTicket ticket;
        long pos;

        SlideTuple(LockTicket ticket) {
            this.ticket = ticket;
        }

        public final String toString() {
            return ticket + " @ " + Fields.longToHex(pos);
        }
    }


    /** linked list of info about the locks we're sliding on */
    private final SlideTuple stHead = new SlideTuple(null);
    
    /** true if any lock we were sliding on unlocked prematurely */
    private boolean slipped = false;

    /** the pos we are waiting for in waitForSlide() */
    private long wakePos = -1;
    
    /** last known minimum position among all the locks */
    private long minPos = -1;
    
    
    /**
     * @return  a list of the tickets to slide against and their positions,
     *          or "(free)" if there aren't any
     */
    public String toString() {
        SlideTuple st = stHead.next;
        if (st == null) {
            return "(free)";
        }
        StringBuffer sb = new StringBuffer(st.toString());
        while (st.next != null) {
            st = st.next;
            sb.append(", ").append(st);
        }
        return sb.toString();
    }
    

    /**
     * Called by a LockGrantor to queue up the tickets to slide on.
     */
    void add(LockTicket ticket) {
        SlideTuple st = new SlideTuple(ticket);
        st.next = stHead.next;
        stHead.next = st;
    }

    /**
     * Called by a LockGrantor, after the tickets
     * to slide on have been queued up.
     */
    synchronized void register() {
        SlideTuple st = stHead;
        while (st.next != null) {
            long pos = st.next.ticket.register(this);
            if (pos == -1) {
                st.next = st.next.next;
            }
            else {
                if (minPos == -1 || minPos > pos)
                    minPos = pos;
                st = st.next;
                st.pos = pos;
            }
        }
    }

    /**
     * @return  true, if any lock being slid on was unlocked prematurely
     */
    public final boolean slipped() {
        return slipped;
    }

    /**
     * @return  the current "slide pointer", i.e. the minimum value
     *          of the byte pointers of the locks being slid on,
     *          or -1 if they've all unlocked
     */
    public final synchronized long getSlidePos() {
        return minPos;
    }
    
    /**
     * Blocks until all locks being slid on have advanced at least
     * to the desired minimum value.
     * @param pos  minimum byte-pointer all locks must slide to
     * @return  the getSlidePos() value at the instant of return
     */
    public synchronized long waitForSlide(long pos) {
        while (minPos != -1 && minPos < pos) {
            try {
                wakePos = pos;
                this.wait(200);
            }
            catch (InterruptedException e) {}
            finally {
                wakePos = -1;
            }
        }
        return minPos;
    }

    /**
     * Blocks until all locks being slid on have unlocked.
     */
    public synchronized void waitForUnlock() {
        while (minPos != -1) {
            try { this.wait(200); }
            catch (InterruptedException e) {}
        }
    }
    
    /**
     * LockSignal impl. -- called when Locks advance
     */
    public synchronized void signalMove(LockTicket src, long pos) {
        minPos = -1;
        SlideTuple st = stHead;
        while (st.next != null) {
            st = st.next;
            if (st.ticket == src)
                st.pos = pos;
            if (minPos == -1 || minPos > st.pos)
                minPos = st.pos;
        }
        if (wakePos != -1 && minPos >= wakePos)
            this.notify();
    }

    /**
     * LockSignal impl. -- called when Locks unlock
     */
    public synchronized void signalUnlock(LockTicket src, boolean failed) {
        minPos = -1;
        SlideTuple st = stHead;
        while (st.next != null) {
            if (st.next.ticket == src) {
                slipped = slipped || failed;
                st.next = st.next.next;
            }
            else {
                st = st.next;
                if (minPos == -1 || minPos > st.pos)
                    minPos = st.pos;
            }
        }
        if (wakePos != -1 || minPos == -1)
            this.notify();
    }
}


