package freenet.fs;

import freenet.support.*;
import java.util.Enumeration;
import java.util.Random;
import java.io.*;

/**
 * A LockGrantor implementation for an unencrypted file-system.
 * Provides a method to initialize the storage.
 * @author tavin
 */
public class FileSystem implements LockGrantor {

    /** linked list of all live locks */
    private final DoublyLinkedList locks = new DoublyLinkedListImpl();

    /** list element and callback hook */
    private final class LockTuple extends DoublyLinkedListImpl.Item
                                  implements LockSignal {

        final LockTicket ticket;
        
        LockTuple(LockTicket ticket) {
            this.ticket = ticket;
        }
    
        public final void signalMove(LockTicket src, long pos) {}

        public final void signalUnlock(LockTicket src, boolean failed) {
            synchronized (semaphore()) {
                locks.remove(this);
            }
        }
    }


    /** raw storage layer */    
    private final Storage storage;
    

    /**
     * @param storage  the Storage to manage
     */
    public FileSystem(Storage storage) {
        this.storage = storage;
    }
    
    
    /**
     * Writes a bunch of zeroes to initialize the file-system.
     * Might take a while..
     * @param rand  not used
     * @param notifyMe  if non-null, will be notify()'d once the FS
     *                  is ready to be used
     */
    public void initialize(Random rand, Object notifyMe) throws IOException {
        long len = size() - truncation();
        if (len <= 0)
            throw new IllegalStateException("FS does not require initialization");
        
        OutputStream out = WriteLock.getOutputStream(this, truncation(),
                                                     size() - 1);
        if (notifyMe != null) {
            synchronized (notifyMe) {
                notifyMe.notify();
            }
        }
            
        try {
            byte[] nada = new byte[0x10000];
            while (len > 0) {
                out.write(nada, 0, (int) Math.min(len, nada.length));
                len -= nada.length;
            }
        }
        finally {
            out.close();
        }
    }


    /**
     * @return  truncation-value of the underlying Storage
     */
    public final long truncation() {
        return storage.truncation();
    }
    
    /**
     * @return  byte size of the underlying Storage
     */
    public final long size() {
        return storage.size();
    }

    /**
     * @return  synchronization object for multiple lock operations
     */
    public final Object semaphore() {
        return this;
    }

    /**
     * LockGrantor implementation..
     */
    public LockSlide grant(LockTicket ticket, int protMask) {
        
        LockTuple ntu = new LockTuple(ticket);
        LockSlide slide = new LockSlide();
        
        synchronized (semaphore()) {
            Enumeration lte = locks.elements();
            while (lte.hasMoreElements()) {
                LockTicket t = ((LockTuple) lte.nextElement()).ticket;
                if (ticket.lo <= t.hi && ticket.hi >= t.lo
                    && 0 != (protMask & t.typeMask))
                { slide.add(t); }
            }
            locks.push(ntu);
        }
        
        ticket.register(ntu);
        ticket.storage = this.storage;
        
        slide.register();
        return slide;
    }
}


