package freenet.diagnostics;
import freenet.FieldSet;
import freenet.support.Fields;
import java.util.Enumeration;
import java.util.Date;
import java.text.DateFormat;
import java.io.DataOutputStream;
import java.io.DataInputStream;
import java.io.IOException;

/**
 * Implementation of a the continuous random var.
 *
 * @author oskar
 */


/*
 * There is a well known mathematical equality that says that 
 * var[X] = E[X^2] - E[X]^2
 * which we apply here to calculate the standard deviation, since 
 * the mean and square-mean can easily be aggregated correctly 
 * over time.
 */
class Continuous extends RandomVar {

    public static long VERSION = 2;

    static {
        try {
            RandomVar.registerType("Continuous", RandomVar.class);
        } catch (NoSuchMethodException e) {
        }
    }
    
    public Continuous(StandardDiagnostics sd, String name, int period,
                      String comment) {
        super(sd, name, period, comment);
    }

    public String getType() {
        return "Continuous";
    }
    
    public void add(long time, double value, long lasttime) {
        super.add(new ContinuousVarEvent(time, value, value*value, value,
                                         value, 1),
                  lasttime);
    }
    
    public VarEvent aggregate(EventList ll, long time, long after) {
        double tsum = 0, tsqsum = 0;
        double tmin = Double.NaN, tmax = Double.NaN; 
        long tn = 0;
        for (Enumeration e = ll.elements() ; e.hasMoreElements();) {
            ContinuousVarEvent cve = (ContinuousVarEvent) e.nextElement(); 
            if (cve.time() > after) {
                if (cve.time() > time)
                    break;
                tsum += cve.sum;
                tsqsum += cve.sqsum;
                tn += cve.n;
                if (!(cve.min >= tmin)) { // the double negation is for NaN
                    tmin = cve.min;
                }
                if (!(cve.max <= tmax)) {
                    tmax = cve.max;
                }
            }
        }
        return new ContinuousVarEvent(time, tsum, tsqsum, tmin, tmax, tn);
    }

    public VarEvent readEvent(DataInputStream in) throws IOException {
        long ver = in.readLong();
        // my initial implementation wrote time first) 
        long time = ver;
        if (ver > StandardDiagnostics.y2k) {
            //System.err.println(ver);
            ver = 1;
        }
        time = (ver < 2 ? time : in.readLong());
        double sum = in.readDouble();
        double sqsum = in.readDouble();
        double min = (ver < 2 ? 0 : in.readDouble());
        double max = (ver < 2 ? 0 : in.readDouble());
        long n = in.readLong();
        if (ver < 2) {
            min = sum / n; // set them both to the mean...
            max = sum / n;
            //System.err.println(n + "   " + sum + "  " + min + "   " + max);
        }
        return new ContinuousVarEvent(time, sum, sqsum, min, max, n);
    }

    public String[] headers() {
        return new String[] { "Mean" ,
                              "Std. Dev.",
                              "Minimum",
                              "Maximum",
                              "Observations"
        };
    }
}

