/* -*- Mode: java; c-basic-indent: 4; tab-width: 4 -*- */
package freenet.client.metadata;
import freenet.FieldSet;
import freenet.client.Request;
import freenet.client.RequestProcess;
import freenet.client.GetRequestProcess;
import freenet.client.PutRequestProcess;
import freenet.client.FreenetURI;
import freenet.support.Bucket;
import freenet.support.BucketFactory;
import freenet.support.Fields;
import java.net.MalformedURLException;

public class StreamPart extends MetadataPart {
    
	public static final String name = "StreamPart";
    
	public final int revision;
	public final boolean live;
	public final String fecAlgorithm;
	public final int blockCount;
	public final int checkBlockCount;
	public final long startChunk;
	public final long endChunk;
	public final int chunkSeconds;
	
	public final boolean header;
	public final FreenetURI uri;
    
	public StreamPart(int revision, boolean live, String fecAlgorithm, int blockCount,
					  int checkBlockCount, long startChunk, long endChunk, 
					  int chunkSeconds, boolean header, FreenetURI uri) {
		this.revision = revision;
		this.live = live;
		this.fecAlgorithm = fecAlgorithm;
		this.blockCount = blockCount;
		this.checkBlockCount = checkBlockCount;
		this.startChunk = startChunk;
		this.endChunk = endChunk;
		this.chunkSeconds = chunkSeconds;
		this.header = header;
		this.uri = uri;
	}
	
	public StreamPart(FieldSet rawFields,
				  MetadataSettings ms) throws InvalidPartException {
		
		String ver = rawFields.get("Revision");
		if(ver == null) 
			throw new InvalidPartException(name()+": Requires Revision");
		
		try {
			revision = Fields.hexToInt(ver);
		} catch (NumberFormatException e) {
			throw new InvalidPartException(name()+": "+ver+
										   " is an invalid value for FEC.BlockCount");
		}
		
		String type = rawFields.get("Type");
		if(type == null)
			throw new InvalidPartException(name()+": Requires Type");
		
		if(type.equalsIgnoreCase("static"))
			live = false;
		else if(type.equalsIgnoreCase("live"))
			live = true;
		else throw new InvalidPartException(name()+": Invalid Type "+type);
		
		String s = rawFields.get("StartChunk");
		
		if(s == null) {
			startChunk = -1;
		} else {
			try {
				startChunk = Fields.hexToLong(s);
			} catch (NumberFormatException e) {
				throw new InvalidPartException(name()+": "+s+" is an invalid value for"+
											   " StartChunk");
			}
		}

		s = rawFields.get("EndChunk");
		
		if(s == null) {
			endChunk = -1;
		} else {
			try {
				endChunk = Fields.hexToLong(s);
			} catch (NumberFormatException e) {
				throw new InvalidPartException(name()+": "+s+" is an invalid value for"+
											   " EndChunk");
			}
		}

		s = rawFields.get("ChunkSeconds");
		
		if(s == null) {
			chunkSeconds = -1;
		} else {
			try {
				chunkSeconds = Fields.hexToInt(s);
			} catch (NumberFormatException e) {
				throw new InvalidPartException(name()+": "+s+" is an invalid value for"+
											   " ChunkSeconds");
			}
		}
		
		s = rawFields.get("Header");
		
		if(s == null || s.equalsIgnoreCase("false")) {
			header = false;
		} else if(s.equalsIgnoreCase("true")) {
			header = true;
		} else {
			throw new InvalidPartException(name()+": "+s+" is an invalid value for "+
										   "Header");
		}
		
		s = rawFields.get("URI");
		
		if(s == null) {
			uri = null;
		} else {
			try {
				uri = new FreenetURI(s);
			} catch (MalformedURLException e) {
				throw new InvalidPartException(name()+": "+s+" is an invalid value for"+
											   " URI");
			}
		}
		
		FieldSet fs = rawFields.getSet("FEC");
		if(fs == null)
			throw new InvalidPartException(name()+": Requires FEC Subsection");
		
		fecAlgorithm = fs.get("AlgoName");
		if(fecAlgorithm == null)
			throw new InvalidPartException(name()+": Requires FEC.AlgoName");
		
		s = fs.get("BlockCount");
		
		if(s == null)
			throw new InvalidPartException(name()+": Requires FEC.BlockCount");
		
		try {
			blockCount = Fields.hexToInt(s);
		} catch (NumberFormatException e) {
			throw new InvalidPartException(name()+": "+s+
										   " is an invalid value for FEC.BlockCount");
		}
		
		s = fs.get("CheckBlockCount");
		
		if(s == null)
			throw new InvalidPartException(name()+": Requires FEC.CheckBlockCount");
		
		try {
			checkBlockCount = Fields.hexToInt(s);
		} catch (NumberFormatException e) {
			throw new InvalidPartException(name()+": "+s+" is an invalid value for"+
										   " FEC.CheckBlockCount");
		}
		
	}
	
    public String name() {
		return name;
	}
	
	public boolean isControlPart() {
		return false; // we don't want it to work automatically through *RequestProcess... or do we? FIXME
	}
	
	public void addTo(FieldSet fs) {
		FieldSet me = new FieldSet();
		me.add("Revision", Fields.longToHex((long)revision));
		me.add("Type", (live ? "live" : "static"));
		if(startChunk != -1)
			me.add("StartChunk", Fields.longToHex(startChunk));
		if(endChunk != -1)
			me.add("EndChunk", Fields.longToHex(endChunk));
		if(chunkSeconds != -1)
			me.add("ChunkSeconds", Fields.longToHex(chunkSeconds));
		me.add("Header", header ? "true" : "false");
		if(uri != null)
			me.add("URI", uri.toString());
		FieldSet fec = new FieldSet();
		fec.add("AlgoName", fecAlgorithm);
		fec.add("BlockCount", Fields.longToHex(blockCount));
		fec.add("CheckBlockCount", Fields.longToHex(checkBlockCount));
		me.put("FEC", fec);
		fs.put(name(), me);
	}
	
	public String toString() {
		return "StreamPart (revision "+revision+", "+(live?"live":"stored")+") using "+
			fecAlgorithm+"("+blockCount+"/"+(blockCount+checkBlockCount)+")";
	}
}
