package freenet.client.http;

import javax.servlet.*;
import javax.servlet.http.*;

import java.io.*;
import java.util.*;
import java.text.*;

import freenet.client.*;

import freenet.support.*;
import freenet.support.servlet.*;
import freenet.client.http.filter.*;

import freenet.client.metadata.MimeTypeUtils;

import freenet.message.client.FEC.SegmentHeader;

// REDFLAG: Remove Node dependancies.
//
// Mattew:
// We need to get rid of the perturbHtl calls.
// I'm not convinced that they provide as much protection
// as people think.  Anyway, that functionality belongs
// in the ClientFactory implementations (e.g. FCPClient, InternalClient)
// not sprinkled througout client code.
//
// We should aim to make our serlvets run in non-freenet containers,
// like Tomcat.
import freenet.node.Node;

/**
 * Servlet to handle to handle downloading of FEC SplitFiles
 * from freenet.
 * <p>
 *
 * @author  <a href="mailto:giannijohansson@attbi.com">Gianni Johansson</a>
 */
public class SplitFileRequestServlet extends ServletWithContext {

    String DEFAULT_MIME_TYPE = null;

    // REDFLAG: Get all default state into one place?

    // Default values which are used if no values are specified in
    // the config file or the request URL.
    int defaultHtl = 15;
    int defaultBlockHtl = 10;
    int defaultRetries = 3;
    int defaultRetryHtlIncrement = 5;
    // These values seem a little bit low. /Bombe
    int defaultHealHtl = 5;
    int defaultHealPercentage = 5;
    int defaultThreads = 5;
    boolean defaultDoParanoidChecks = true;
    int defaultRefreshIntervalSecs = 30;

    boolean defaultForceSave = false;
    boolean defaultWriteToDisk = false;
    boolean disableWriteToDisk = false;
    boolean defaultSkipDS = false;
    boolean defaultUseUI = true;
    //int defaultUseUIMinSize = 1<<21; // 2MB
    boolean defaultRunFilter = true;
    boolean defaultRandomSegs = true;
    boolean defaultFilterParanoidStringCheck = false;
    String defaultDownloadDir = "";
    
    String filterPassThroughMimeTypes = "";
    
    int defaultLifetimeMs = 60 * 60000;
    boolean pollForDroppedConnections = true;

    private NumberFormat nf = NumberFormat.getInstance();

    ////////////////////////////////////////////////////////////
    // Servlet initialization
    ////////////////////////////////////////////////////////////

    public void init() {
        DEFAULT_MIME_TYPE = MimeTypeUtils.fullMimeType("application/octet-stream", null, null);

        ServletContext context = getServletContext();
        setupLogger(context);
        setupBucketFactory(context);
        setupClientFactory(context);
        
        // Some params will need to be duped in fproxy? REDFLAG: revisit

	defaultHtl = ParamParse.readInt(this, logger, "requestHtl", defaultHtl, 0, 100);
	defaultHtl = Node.perturbHTL(defaultHtl);
	defaultBlockHtl = ParamParse.readInt(this, logger, "sfBlockRequestHtl", defaultBlockHtl, 0, 100);
	defaultBlockHtl = Node.perturbHTL(defaultBlockHtl);
	defaultRetries = ParamParse.readInt(this, logger, "sfRequestRetries", defaultRetries, 0, 50);
	defaultRetryHtlIncrement =
	    ParamParse.readInt(this, logger, "sfRetryHtlIncrement", defaultRetryHtlIncrement, 0, 100);
	defaultHealHtl = ParamParse.readInt(this, logger, "sfHealHtl", defaultHealHtl, 0, 100);
	defaultHealHtl = Node.perturbHTL(defaultHealHtl);
	defaultHealPercentage = ParamParse.readInt(this, logger, "sfHealPercentage", defaultHealPercentage, 0, 100);
	defaultThreads = ParamParse.readInt(this, logger, "sfRequestThreads", defaultThreads, 0, 100);
	defaultDoParanoidChecks = ParamParse.readBoolean(this, logger, 
							 "sfDoParanoidChecks", defaultDoParanoidChecks); 
	defaultRefreshIntervalSecs = ParamParse.readInt(this, logger, "sfRefreshIntevalSecs",
							defaultRefreshIntervalSecs, -1, 3600);
	defaultForceSave = ParamParse.readBoolean(this, logger, "sfForceSave", defaultForceSave); 
	defaultWriteToDisk = ParamParse.readBoolean(this, logger, 
						    "sfDefaultWriteToDisk", 
						    defaultWriteToDisk);
	
	disableWriteToDisk = ParamParse.readBoolean(this, logger,
						    "sfDisableWriteToDisk",
						    disableWriteToDisk);
	
	if(disableWriteToDisk) defaultWriteToDisk = false;
	
	defaultSkipDS = ParamParse.readBoolean(this, logger, "sfSkipDS", defaultSkipDS); 
	defaultUseUI = ParamParse.readBoolean(this, logger, "sfUseUI", defaultUseUI); 
	//defaultUseUIMinSize = ParamParse.readInt(this, logger, "sfUseUIMinSize", defaultUseUIMinSize, 0, Integer.MAX_VALUE);
	defaultRunFilter = ParamParse.readBoolean(this, logger, "sfRunFilter", defaultUseUI); 
	defaultRandomSegs = ParamParse.readBoolean(this, logger, "sfRandomizeSegments",
						   defaultRandomSegs);
	defaultFilterParanoidStringCheck = ParamParse.readBoolean(this, logger, "sfFilterParanoidStringCheck", defaultFilterParanoidStringCheck);
	
	defaultDownloadDir = getInitParameter("sfDefaultSaveDir");
	
	String s = getInitParameter("sfFilterPassThroughMimeTypes");
	if(!(s == null || s.equals("")))
	    filterPassThroughMimeTypes = s;
	else
	    filterPassThroughMimeTypes = Node.filterPassThroughMimeTypes;
        logger.log(this, "New SplitFileRequestServlet created", Logger.MINOR);
	if(logger.shouldLog(Logger.DEBUG)) {
	    logger.log(this, "   requestHtl = " + defaultHtl, Logger.DEBUGGING);
	    logger.log(this, "   sfBlockHtl = " + defaultBlockHtl, Logger.DEBUGGING);
	    logger.log(this, "   sfRequestRetries = " + defaultRetries, 
		       Logger.DEBUGGING);
	    logger.log(this, "   sfRetryHtlIncrement = " + defaultRetryHtlIncrement, 
		       Logger.DEBUGGING);
	    logger.log(this, "   sfRequestThreads = " + defaultThreads, 
		       Logger.DEBUGGING);
	    logger.log(this, "   sfRefreshIntervalSecs = " + defaultRefreshIntervalSecs,
		       Logger.DEBUGGING);
	    logger.log(this, "   sfForceSave = " + defaultForceSave, Logger.DEBUGGING);
	    logger.log(this, "   sfSkipDS = " + defaultSkipDS, Logger.DEBUGGING);
	    logger.log(this, "   sfUseUI = " + defaultUseUI, Logger.DEBUGGING);
	    //logger.log(this, "   sfMinSizeUseUI = " + defaultUseUIMinSize,
	    //Logger.DEBUGGING);
	    logger.log(this, "   sfRunFilter = " + defaultRunFilter, 
		       Logger.DEBUGGING);
	}
    }
    
    
    ////////////////////////////////////////////////////////////
    // Presentation
    ////////////////////////////////////////////////////////////
    
    // Presentation states.
    // TODO: remove these and references ones in SFRContext
    public final static int STATE_INIT  = 1;
    public final static int STATE_REQUESTING_METADATA  = 2;
    public final static int STATE_STARTING  = 3;
    public final static int STATE_WORKING = 4;
    public final static int STATE_FILTERING_DATA = 5;
    public final static int STATE_FILTER_FAILED = 6;
    public final static int STATE_SENDING_DATA = 7;
    public final static int STATE_DONE = 8;
    public final static int STATE_FAILED = 9;
    public final static int STATE_CANCELED = 10;


    // suburls
    // /starting        reads metadata
    // /download/<uri>  downloads url once
    // /cancel          cancel's pending request
    // /override_filter sends data after filter trips
    // /status_progress renders download progress


    ////////////////////////////////////////////////////////////
    // This is kind of verbose, but I want to make it
    // as easy as possible for other people to modify
    // the UI in a maintainable way.
    //
    protected void handleIllegalDownloadState(HttpServletRequest req, HttpServletResponse resp,
                                              SFRContext context)
        throws ServletException, IOException {
        sendHtml(resp, HttpServletResponse.SC_BAD_REQUEST,
                 "<html> " +
                 "<head> " +
                 "<title>You can only download once.</title> " +
                 "</head> " +
                 "<body> " +
                 "<h1>You can only download once.</h1> " +
                 "</body> " +
                 "</html> ");
    }

    protected void handleNotASplitFile(HttpServletRequest req, HttpServletResponse resp,
                                       String uri,
                                       SFRContext context)
        throws ServletException, IOException {

        
        sendHtml(resp, HttpServletResponse.SC_NOT_IMPLEMENTED,
                 "<html> " +
                 "<head> " +
                 "<title>Not a SplitFile</title> " +
                 "</head> " +
                 "<body> " +
                 "<h1>Not a SplitFile</h1> " +
                 " The requested URI wasn't a splitFile. Use fproxy instead!" +
                 "</body> " +
                 "</html> ");

    }


    protected void handleNotFECEncoded(HttpServletRequest req, HttpServletResponse resp,
                                       String uri,
                                       SFRContext context)
        throws ServletException, IOException {

        
        sendHtml(resp, HttpServletResponse.SC_NOT_IMPLEMENTED,
                 "<html> " +
                 "<head> " +
                 "<title>Not a FEC Encoded</title> " +
                 "</head> " +
                 "<body> " +
                 "<h1>Not a FEC Encoded</h1> " +
                 " The requested URI is a SplitFile, but it isn't FEC encoded." +
                 " <p> " +
                 " This file can't be downloaded. Retrying won't help. " +
                 " Ask the content author to re-insert it with FEC." +
                 "</body> " +
                 "</html> ");

    }


    protected void handleFilterException(HttpServletRequest req, HttpServletResponse resp,
                                         FilterException fe,
                                         SFRContext context)
        throws ServletException, IOException {

        sendHtml(resp, HttpServletResponse.SC_OK,
                 "<html> " +
                 "<head> " +
                 "<title>Filter Exception</title> " +
                 "</head> " +
                 "<body> " +
                 "<h1>Filter Exception</h1> " +
                 "</body> " +
                 "</html> ");

        // REDFLAG:
        // Unimplemented.

        // should print filter exception, with link to context.overrideFilterURL
    }

    protected void handleRequestFailed(HttpServletRequest req, HttpServletResponse resp,
                                       SFRContext context)
        throws ServletException, IOException {

        sendHtml(resp, HttpServletResponse.SC_NOT_FOUND,
                 "<html> " +
                 "<head> " +
                 "<title>Freenet Request Failed</title> " +
                 "</head> " +
                 "<body> " +
                 "<h1>Freenet Request Failed</h1> " +
                 "</body> " +
                 "</html> ");
    }

    protected void handleMetadataRequestFailed(HttpServletRequest req, HttpServletResponse resp,
                                               SFRContext context)
        throws ServletException, IOException {

        sendHtml(resp, HttpServletResponse.SC_NOT_FOUND,
                 "<html> " +
                 "<head> " +
                 "<title>Freenet SplitFile Metadata Request Failed</title> " +
                 "</head> " +
                 "<body> " +
                 "<h1>Freenet SplitFile Metadata Request Failed</h1> " +
                 "Couldn't start downloading the SplitFile because the SplitFile metadata " +
                 "couldn't be retrieved from Freenet. " +
                 "</body> " +
                 "</html> ");
    }


    ////////////////////////////////////////////////////////////
    // Helper functions render appropriate replies for
    // each type of suburl request.
    //

    protected void onNewContext(HttpServletRequest req, HttpServletResponse resp)
        throws ServletException, IOException {
        
        // Parse out url
        String key = null;
        String path = null;
	boolean logDEBUG = logger.shouldLog(Logger.DEBUG);
        try {
	    if(logDEBUG) logger.log(this, "Got request: "+key, Logger.DEBUG);
	    // getRequestURI() does not include params, so can be decoded
            key = URLDecoder.decode(req.getRequestURI());
	    if(logDEBUG) logger.log(this, "Decoded: "+key, Logger.DEBUG);
            path = URLDecoder.decode(req.getServletPath());
	    
            int pos = key.indexOf(req.getServletPath());
            
            if (pos == -1 || pos + req.getServletPath().length() == key.length() - 1) {
                handleBadURI(req, resp);
                return;
            }

            key = key.substring(pos + req.getServletPath().length());

        }
        catch (URLEncodedFormatException uefe) {
            handleBadURI(req, resp);
            return;
        }

        
        // chop leading /
        if (key != null && key.startsWith("/")) {
            key = key.substring(1);
        }
        
        // Registers itself.
        SFRContext context = new SFRContext(defaultLifetimeMs, key, path, this,
					    bf);
	
        String warning = context.updateParameters(req);
	if(warning == null) {
	    
	    if (context.getMetadata(req, resp)) {
		// Send redirect to kick off 
		// request for metadata.
                //          String s = req.getHeader("accept");
		// 	    freenet.Node.logger.log(this, "Accept header: "+s, 
		// 				    freenet.Node.logger.DEBUG);
		// Disabled: Mozilla at least changes the Accept headers on redirection
		// 	    if((s.indexOf("image") != -1) && (s.indexOf("text") == -1))
		// 		context.useUI = false; // Streaming as an inline image
		// 	    if(context.sf.getSize() < context.useUIMinSize) 
		// 		context.useUI = false;
		if (context.useUI) {
		    resp.sendRedirect(URLEncoder.encode(context.parameterURL()));
		    if(logDEBUG) logger.log(this, "Sending redirect: " + 
					    context.parameterURL(), Logger.DEBUGGING);
		}
		else {
		    resp.sendRedirect(URLEncoder.encode(context.downloadURL()));
		    if(logDEBUG) logger.log(this, "Sending redirect: " + 
					    context.downloadURL(), Logger.DEBUGGING);
		}
	    }
	    // else
	    // getMetaData sends back an error msg.
	} else {
	    onParameterForm(req, resp, context, warning);
	}
    }

    protected void onDownload(HttpServletRequest req, HttpServletResponse resp,
                              SFRContext context)
        throws ServletException, IOException {
	boolean logDEBUG = logger.shouldLog(Logger.DEBUG);
	if(logDEBUG) logger.log(this, "onDownload(,,"+context+") on "+this,
				Logger.DEBUG);
        try {
            // This blocks until the request is finished or canceled.
            context.doRequest(req, resp);
        }
        catch (IllegalStateException es) {
            // Thrown if the request has been called more than once.
	    if(logDEBUG) logger.log(this, "onDownload(,,"+context+") on "+
				    this+" got "+es, es, Logger.DEBUG);
            handleIllegalDownloadState(req,resp,context);
        }
    }
    
    protected void onCancel(HttpServletRequest req, HttpServletResponse resp,
                            SFRContext context)
        throws ServletException, IOException {
	boolean logDEBUG = logger.shouldLog(Logger.DEBUG);
	if(logDEBUG) logger.log(this, "Called onCancel()", Logger.DEBUG);
        context.cancel();
	if(logDEBUG) logger.log(this, "Out of context.cancel()", Logger.DEBUG);
	
        StringWriter psw = new StringWriter();
        PrintWriter ppw = new PrintWriter(psw);
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);

        context.titleBoxTmp.set("TITLE", "Splitfile Download Cancelled");
        pw.println("<p>You have chosen to cancel the splitfile download. ");
        pw.println("Use your browser's history function to return to the page you ");
        pw.println("visited when you started this download.</p>");
        context.titleBoxTmp.set("CONTENT", sw.toString());
        context.titleBoxTmp.toHtml(ppw);
	
	if(logDEBUG) logger.log(this, "onCancel writing: "+psw.toString(), 
				Logger.DEBUG);
	
        resp.setStatus(HttpServletResponse.SC_OK);
        resp.setContentType("text/html");

        context.pageTmp.set("TITLE", "Splitfile Download Cancelled");
        context.pageTmp.set("BODY", psw.toString());
	PrintWriter w = resp.getWriter();
        context.pageTmp.toHtml(w);
	w.close();
	
	if(logDEBUG) logger.log(this, "Finished writing in onCancel()", Logger.DEBUG);
        resp.flushBuffer();
	if(logDEBUG) logger.log(this, "Out of onCancel()", Logger.DEBUG);
    }

    // hmmm... not really required. but pedantic.
    protected void onOverrideFilter(HttpServletRequest req, HttpServletResponse resp,
                            SFRContext context)
        throws ServletException, IOException {
        context.doOverrideFilter(req, resp);
    }

    private final void renderTopStatusFrame(PrintWriter pw,
            SFRContext context) throws IOException {

        SegmentHeader header = null;
        if (context.status != null) {
            header = context.status.segment();
        }
        HtmlTemplate titleBoxTmp = context.titleBoxTmp;

        StringWriter tsw = new StringWriter();
        PrintWriter tpw = new PrintWriter(tsw);

        // split uri into key and filename (if a filename is given)
        String key = context.displayKey();
        String filename = context.filename();
	
        titleBoxTmp.set("TITLE", "Splitfile Download");
        tpw.println("<table border=\"0\"><tr>");
        tpw.println("<td valign=\"top\"><img src=\"/servlet/images/aqua/download.png\" width=\"42\" height=\"41\" alt=\"\"></td>");
        tpw.println("<td width=\"10\">&nbsp;</td>");
        tpw.println("<td valign=\"top\">");
        String shortKey = key;
        if(shortKey.startsWith("freenet:"))
            shortKey = key.substring("freenet:".length(),key.length());
        String saveKey = context.uri;
        if(saveKey.startsWith("freenet:"))
            saveKey = saveKey.substring("freenet:".length(), 
                                        saveKey.length());
        tpw.println("<p><b>Key</b>: freenet:<a href=\"/"+
                   HTMLEncoder.encode(URLEncoder.encode(saveKey))+
                   "\">" + shortKey + "</a>");
        if (filename != null) {
            tpw.println("<br><b>Filename</b>: " + filename + ", " +
                "<b>Length:</b> " + format(context.sf.getSize()));
        }
        if (header != null) {
            long totalSize = context.status.dataSize();
            if (totalSize == 0) {
                totalSize = context.sf.getSize();
            }
            double progress = (double) context.status.retrievedBytes() / (double) totalSize;

            tpw.println("<br><b>Status</b>: ");
            tpw.print("<img src=\"/servlet/images/aqua/" + ((progress != 0) ? "green" : "blue") + "_start.png\" alt=\"\">");
            if (progress != 0) {
                tpw.print("<img src=\"/servlet/images/aqua/green.png\" width=\"" + (int) (progress * 300.0) + "\" height=\"16\" title=\"" + (int) (progress * 100.0)+ "%\" alt=\"" + (int) (progress * 100.0)+ "%\">");
            }
            if (progress != 1) {
                tpw.print("<img src=\"/servlet/images/aqua/blue.png\" width=\"" + (int) ((1 - progress) * 300.0) + "\" height=\"16\" title=\"" + (int) (progress * 100.0) + "%\" alt=\"" + (int) (progress * 100.0) + "%\">");
            }
            tpw.print("<img src=\"/servlet/images/aqua/" + ((progress != 1) ? "blue" : "green") + "_end.png\" alt=\"\">");
            //long deltat = (System.currentTimeMillis() - context.status.touched()) / 1000;
            //if (deltat < 60) {
            //    tpw.print(" <b>Idle</b>: " + deltat + " seconds");
            //} else {
            //    tpw.print(" <b>Idle</b>: " + (int) (deltat / 60) + " minutes");
            //}
            tpw.print("<br><b>Request Started:</b> " + context.dateFormat.format(context.startTime.getTime()));
            tpw.println(", <b>Time Elapsed:</b> " + timeDistance(context.startTime, Calendar.getInstance()));
            if ((progress >= 0.1) || ((context.status.blocksProcessed() > 4) && (context.status.retrievedBytes() > 0))) {
                long start = context.startTime.getTime().getTime();
                long elapsed = Calendar.getInstance().getTime().getTime() - start;
                long end = start + (long) ((double) elapsed / progress);
                Calendar eta = Calendar.getInstance();
                eta.setTime(new Date(end));
                tpw.println("<br><b>Estimated Finish Time:</b> " + context.dateFormat.format(eta.getTime()));
                long throughput = context.status.retrievedBytes() * 1000 / elapsed;
                tpw.println("<br><b>Speed:</b> " + format(throughput) + "/s"); //K.I.S.S.!!!
            }
        }
        tpw.println("</p></td></tr></table>");
        titleBoxTmp.set("CONTENT", tsw.toString());
        titleBoxTmp.toHtml(pw);
        pw.println("<br>");
    }

    // Break table into a separate function.
    private final void renderRunningDownloadStatus(PrintWriter pw, 
            SFRContext context)
        throws  IOException {

        SegmentHeader header = context.status.segment();

        StringWriter tsw = new StringWriter();
        PrintWriter tpw = new PrintWriter(tsw);
        
        HtmlTemplate titleBoxTmp = context.titleBoxTmp;

        if (header != null) {
            tsw = new StringWriter();
            tpw = new PrintWriter(tsw);

            if (header.getCheckBlockCount() != 0) {
                titleBoxTmp.set("TITLE", "Required Blocks: " + header.getBlocksRequired() + ", Received Blocks: " + context.status.blocksProcessed());
                for (int i = 0; i < header.getBlocksRequired(); i++) {
                    if (i < context.status.blocksProcessed()) {
                        tpw.println("<img src=\"/servlet/images/aqua/success.png\" width=\"23\" height=\"23\"> ");
                    } else {
                        tpw.println("<img src=\"/servlet/images/aqua/waiting.png\" width=\"23\" height=\"23\"> ");
                    }
                }
            } else {
                titleBoxTmp.set("TITLE", "Attempting Non-Redundant Splitfile Download");
                tpw.println("<p>Warning! You are attempting a non-redundant splitfile download! ");
                tpw.println("This means that the download will fail if just one block can not ");
                tpw.println("be retrieved from the network! If you can, please notify the ");
                tpw.println("person who inserted this file and ask her to re-insert it using ");
                tpw.println("the Fproxy File Insertion utility from the Freenet Gateway Page. ");
                tpw.println("That will automatically make sure that redundant check blocks ");
                tpw.println("are inserted, thus making a complete download much easier.</p>");
            }
            titleBoxTmp.set("CONTENT", tsw.toString());
            titleBoxTmp.toHtml(pw);
            pw.println("<br>");
            
            tsw = new StringWriter();
            tpw = new PrintWriter(tsw);

            int completed = 0;
            for (int p = 0, i = 0; i < header.getBlockCount() + header.getCheckBlockCount(); i++) {
                String img = "";
                String alt = "[Block " + i + "] ";
                switch (context.status.retrievedBlockStatus(i)) {
                case SplitFileStatus.QUEUED:
                    img = "waiting";
                    alt += "Queued";
                    break;
		case SplitFileStatus.UNDEFINED:
		    img = "waiting";
		    alt += "Waiting";
		    break;
                case SplitFileStatus.RUNNING:
		    if(context.status.retrievedBlockSize(i) > 0) {
			img = "transfer";
			alt += "Transferring... (" +
			    (context.status.retrievedBlockRetries(i) + 1) + ". Try)";
		    } else {
			img = "progress";
			alt += "In Progress... (" + 
			    (context.status.retrievedBlockRetries(i) + 1) + ". Try)";
		    }
                    break;
                case SplitFileStatus.FAILED_RNF:
                    img = "failed";
                    alt += "Failed (Route Not Found)";
                    break;
                case SplitFileStatus.FAILED_DNF:
                    img = "failed";
                    alt += "Failed (Data Not Found)";
                    break;
                case SplitFileStatus.FAILED:
                    img = "failed";
                    alt += "Failed (Unknown Reason)";
                    break;
                case SplitFileStatus.REQUEUED_RNF:
                    img = "retry";
                    alt += "Route Not Found, Will Retry";
                    break;
                case SplitFileStatus.REQUEUED_DNF:
                    img = "retry";
                    alt += "Data Not Found, Will Retry";
                    break;
                case SplitFileStatus.REQUEUED:
                    img = "retry";
                    alt += "Unknown Error, Will Retry";
                    break;
                case SplitFileStatus.SUCCESS:
                    img = "success";
                    alt += "Success (" + (p = context.status.retrievedBlockRetries(i)) + " Retr" + ((p == 1) ? "y" : "ies") + ")";
                    completed++;
		    break;
		default:
		    img = "waiting";
		    alt += "Running - status "+
			context.status.retrievedBlockStatus(i);
                }
		
                if (img.equals("retry") && (context.status.retrievedBlockRetries(i)) > 1) {
                    img = "retry2";
                    alt = alt + " (" + context.status.retrievedBlockRetries(i) + ". Retry)";
                }
                tpw.println("<img src=\"/servlet/images/aqua/" + img + ".png\" width=\"23\" height=\"23\" title=\"" + alt + "\" alt=\"" + alt + "\"> ");
            }
            if (header.getCheckBlockCount() != 0) {
                titleBoxTmp.set("TITLE", "Segment " + (header.getSegmentNum() + 1) 
                                + ", " + (context.status.segmentNr() + 1) + " of "
                                + header.getSegments() + ", Download Queue: "
                                + (header.getBlockCount() + header.getCheckBlockCount())
                                + " Blocks");
            } else {
                titleBoxTmp.set("TITLE", "Download Queue: " + (header.getBlockCount()) +
                    " Blocks, Received Blocks: " + completed);
            }
            titleBoxTmp.set("CONTENT", tsw.toString());
            titleBoxTmp.toHtml(pw);
        } else {
            tpw.println("<p>Waiting for download to start&hellip;</p>");
            titleBoxTmp.set("TITLE", "Waiting for Download");
            titleBoxTmp.set("CONTENT", tsw.toString());
            titleBoxTmp.toHtml(pw);
        }
    }

    protected boolean setupFilter = false;

    protected void onSplitBottom(HttpServletRequest req, HttpServletResponse resp,
                                    SFRContext context)
        throws ServletException, IOException {
	
	boolean logDEBUG = logger.shouldLog(Logger.DEBUG);
        // if the download has already started, skip the bottom
        // frame and redirect to the status_progress page.
        if (context.state != STATE_STARTING) {
            resp.sendRedirect(URLEncoder.encode(context.progressURL()));
	    if(logDEBUG) logger.log(this, "Sending redirect: " + context.progressURL(),
				    Logger.DEBUGGING);
            return;
        }

	String warning = context.updateParameters(req);
	
	if(warning == null) {

	    if(!(context.writeDir == null || disableWriteToDisk || 
		 context.writeToDisk == false)) {
		
		if(logDEBUG) logger.log(this, "Starting background request for "+
					this, Logger.DEBUG);
		
		context.doBackgroundRequest();
		
		resp.sendRedirect(context.progressURL());
		return;
	    }
	    
	    if(!setupFilter) {
		setupFilter = true;
		context.setupFilter(); // need to run it if only to null out filter... see comments
	    }
	    
	    Writer w = resp.getWriter();
	    String s = "<html><head><title>";
	    if(context.state == context.STATE_FILTER_FAILED ||
	       context.state == context.STATE_CANCELED ||
	       context.state == context.STATE_FAILED)
		s += "Failed download - ";
	    else if(context.state == context.STATE_DONE)
		s += "Finished download - ";
	    else if (context.state == context.STATE_SENDING_DATA)
		s += "Sending data - ";
	    else if (context.state == context.STATE_FILTERING_DATA)
		s += "Filtering data - ";
	    else s += ((int)(context.progress()*100.0))+" % - ";
	    w.write(s + HTMLEncoder.encode(context.filename())+
		    "</title></head>");
	    String dlFrame = "  <frame name=\"download\" src=\"" + 
		HTMLEncoder.encode(URLEncoder.encode(context.downloadURL()))
		+ "\">";
	    String detailFrame = "  <frame name=\"details\" src=\"" + 
		HTMLEncoder.encode(URLEncoder.encode(context.progressURL())) +
		"\">";
	    if(logDEBUG) logger.log(this, "MIME type: "+context.mimeType+
				    ", mimeTypeParsed: "+context.mimeTypeParsed+
				    ", default: "+DEFAULT_MIME_TYPE, Logger.DEBUG);
	    
	    if(context.mimeType.equals(DEFAULT_MIME_TYPE)) {
		if(logDEBUG) logger.log(this, "Writing *,0 frameset for "+
					context.uri, logger.DEBUG);
		w.write("<frameset rows=\"*,0\">");
	    } else {
		if(logDEBUG) logger.log(this, "Writing 1,1 frameset for "+
					context.uri, logger.DEBUG);
		w.write("<frameset rows=\"1,1\">");
	    }
	    w.write(detailFrame);
	    w.write(dlFrame);
	    w.write("</frameset></html>");
	    resp.setStatus(HttpServletResponse.SC_OK);
	    resp.setContentType("text/html");
	    w.flush();
	} else {
	    onParameterForm(req, resp, context, warning);
	}
    }

    protected void onParameterForm(HttpServletRequest req, HttpServletResponse resp,
				   SFRContext context) 
	throws ServletException, IOException {
	onParameterForm(req, resp, context, null);
    }
    
    protected void onParameterForm(HttpServletRequest req, HttpServletResponse resp,
				   SFRContext context, String warning)
        throws ServletException, IOException {

        // if the download is already running, create the next frameset.

        HtmlTemplate titleBoxTmp = context.titleBoxTmp;
        HtmlTemplate pageTmp = context.pageTmp;

        if ((context.state != STATE_INIT) && (context.state != STATE_REQUESTING_METADATA) && (context.state != STATE_STARTING)) {
            resp.sendRedirect(URLEncoder.encode(context.splitBottomURL()));
            return;
        }

        String htlAsString = Integer.toString(context.blockHtl);
        String retryHtlIncrementAsString = Integer.toString(context.retryHtlIncrement);
        String healHtlAsString = Integer.toString(context.healHtl);
        String healPercentageAsString = Integer.toString(context.healPercentage);
        String retriesAsString = Integer.toString(context.retries);
        String threadsAsString = Integer.toString(context.threads);
        String forceSaveAsString = "";
        String skipDSAsString = "";
        String randomSegsAsString = "";
        String runFilterAsString = "";
        String paranoidStringCheckAsString = "";
	String writeToDiskAsString = "";
	
	context.preSetupFilter();

	// For HTML, checkboxes are on if checked attribute is set
	// The value parameter is what is sent if they are checked
        if (context.forceSave) {
            forceSaveAsString = " checked";
        }

        if (context.skipDS) {
            skipDSAsString = " checked";
        }

        if (context.randomSegs) {
            randomSegsAsString = " checked";
        }

        if (context.runFilter) {
            runFilterAsString = " checked";
        }

        if (context.filterParanoidStringCheck) {
            paranoidStringCheckAsString = " checked";
        }
	
	if (context.writeToDisk)
	    writeToDiskAsString = " checked";
	
        StringWriter psw = new StringWriter();
        PrintWriter ppw = new PrintWriter(psw);
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);
        
        titleBoxTmp.set("TITLE", "Splitfile Download Request");
        pw.println("Downloading large SplitFiles can be a resource intensive operation. ");
        pw.println("Hit the Back button on your browser if you want to abort. ");
        pw.println("You can also abort the download once it starts by hitting ");
        pw.println("the Back button on your browser or canceling the file save ");
        pw.println("dialog if you're saving to a file.");
	if(warning != null)
	    pw.println("<hr><font color=\"red\">"+warning+"</font><hr>");
        titleBoxTmp.set("CONTENT", sw.toString());
        titleBoxTmp.toHtml(ppw);
        ppw.println("<br>");

        sw = new StringWriter();
        pw = new PrintWriter(sw);

        titleBoxTmp.set("TITLE", "Splitfile Download Parameters");
	context.writeHtml(pw, true);
        pw.println("<form method=\"GET\" action=\"" + HTMLEncoder.encode(URLEncoder.encode(context.splitBottomURL())) + "\">");
        // Hack to make checkbox work correctly.
        pw.println("<input type=\"hidden\" name=\"usedForm\" value=\"true\" >");
        pw.println("<table border=\"0\">");

        pw.println("<tr>");
        pw.println("    <td align=\"right\"><input type=\"text\" name=\"blockHtl\" value=\"" + htlAsString  + "\" size=\"5\"></td>");
        pw.println("    <td>Initial Hops to Live</td>");
        pw.println("</tr>");
        
        pw.println("<tr>");
        pw.println("    <td align=\"right\"><input type=\"text\" name=\"retries\" value=\"" + retriesAsString + "\" size=\"5\"></td>");
        pw.println("    <td>Number of Retries for a Block that Failed</td>");
        pw.println("</tr>");
        
        pw.println("<tr>");
        pw.println("    <td align=\"right\"><input type=\"text\" name=\"retryHtlIncrement\" value=\"" + retryHtlIncrementAsString + "\" size=\"5\"></td>");
        pw.println("    <td>Increment HTL on retry by this amount</td>");
        pw.println("</tr>");

        pw.println("<tr>");
        pw.println("    <td align=\"right\"><input type=\"checkBox\" name=\"forceSaveCB\" value=\"true\"" + forceSaveAsString + "></td>");
        pw.println("    <td>Force the Browser to Save the File</td>");
        pw.println("</tr>");
        
        pw.println("<tr>");
        pw.println("    <td align=\"right\"><input type=\"checkBox\" name=\"skipDSCB\" value=\"true\"" + skipDSAsString + "></td>");
        pw.println("    <td>Don't Look for Blocks in Local Data Store</td>");
        pw.println("</tr>");

        pw.println("<tr>");
        pw.println("    <td align=\"right\"><input type=\"checkBox\" name=\"randomSegs\" value=\"true\"" + randomSegsAsString + "></td>");
        pw.println("    <td>Download Segments in Random Order</td>");
        pw.println("</tr>");

        pw.println("<tr>");
        pw.println("    <td align=\"right\"><input type=\"text\" name=\"threads\" value=\"" + threadsAsString + "\" size=\"5\"></td>");
        pw.println("    <td>Number of Simultaneous Downloads</td>");
        pw.println("</tr>");

        pw.println("<tr>");
        pw.println("    <td align=\"right\"><input type=\"checkBox\" name=\"runFilterCB\" value=\"true\"" + runFilterAsString + "></td>");
        pw.println("    <td>Run Anonymity Filter on Download Completion (recommended)</td>");
        pw.println("</tr>");

        pw.println("<tr>");
        pw.println("    <td align=\"right\"><input type=\"checkBox\" name=\"filterParanoidStringCheck\" value=\"true\"" + paranoidStringCheckAsString + "></td>");
        pw.println("    <td>Make the Anonymity Filter Really Paranoid</td");
        pw.println("</tr>");

        pw.println("<tr>");
        pw.println("    <td align=\"right\"><input type=\"text\" name=\"healPercentage\" value=\"" + healPercentageAsString + "\" size=\"5\"></td>");
        pw.println("    <td>% of Missing Data Blocks to Insert (Healing)</td>");
        pw.println("</tr>");

        pw.println("<tr>");
        pw.println("    <td align=\"right\"><input type=\"text\" name=\"healHtl\" value=\"" + healHtlAsString + "\" size=\"5\"></td>");
        pw.println("    <td>Hops-to-Live for the Healing Blocks</td>");
        pw.println("</tr>");
	
	if(!disableWriteToDisk) {
	    
	    pw.println("<tr>");
	    pw.println("    <td align=\"right\"><input type=\"checkBox\" "+
		       "name=\"writeToDisk\" value=\"true\""+writeToDiskAsString+
		       " size=\"5\"></td>");
	    pw.println("    <td>Write directly to disk rather than sending to browser</td>");
	    pw.println("</tr>");
	    
	    pw.println("<tr>");
	    
	    pw.println("<tr>");
	    pw.println("    <td align=\"right\"><input type=\"text\" name=\"saveToDir\" "+
		       "value=\""+(defaultDownloadDir==null?"":defaultDownloadDir)+
		       "\" size=\"15\"></td>");
	    pw.println("    <td>Folder to write file to</td>");
	    pw.println("</tr>");
	    
	}
	
        pw.println("</table>");
	
        pw.println("<p><input type=\"submit\" value=\"Start Download\">");
        pw.println("</form>");
        titleBoxTmp.set("CONTENT", sw.toString());
        titleBoxTmp.toHtml(ppw);

        pageTmp.set("BODY", psw.toString());
	pageTmp.set("TITLE", "Splitfile Download Request");
        pageTmp.toHtml(resp.getWriter());
        
        resp.setStatus(HttpServletResponse.SC_OK);
        resp.setContentType("text/html");
        resp.getWriter().flush();
    }

    protected void onStatusProgress(HttpServletRequest req, HttpServletResponse resp,
                                    SFRContext context)
        throws ServletException, IOException {
        
        boolean updating = false;

        StringWriter psw = new StringWriter();
        PrintWriter ppw = new PrintWriter(psw);

        StringWriter sw;
        PrintWriter pw;
        
        HtmlTemplate titleBoxTmp = context.titleBoxTmp;
        HtmlTemplate pageTmp = context.pageTmp;

        synchronized (context) {
            updating = context.isUpdating();
            
            switch (context.state) {
            case STATE_INIT:
            case STATE_REQUESTING_METADATA:
                
                titleBoxTmp.set("TITLE", "Current Download Status");
                titleBoxTmp.set("CONTENT", "<p>Downloading Splitfile Metadata...</p>");
                titleBoxTmp.toHtml(ppw);
		pageTmp.set("TITLE", ((int)(context.progress()*100.0))+"% - "+
			    HTMLEncoder.encode(context.filename()));
                break;
            case STATE_STARTING:
                renderTopStatusFrame(ppw, context);
                titleBoxTmp.set("TITLE", "Starting Download");
                titleBoxTmp.set("CONTENT", "<p>The Splitfile Download is about to start...</p>");
                titleBoxTmp.toHtml(ppw);
		pageTmp.set("TITLE", ((int)(context.progress()*100.0))+"% - "+
			    HTMLEncoder.encode(context.filename()));
                break;
            case STATE_WORKING:
                synchronized (context.status) {
                    if (context.status.statusCode() == SplitFileStatus.DECODING) {
                        renderTopStatusFrame(ppw, context);
                        SegmentHeader header = context.status.segment();
                        titleBoxTmp.set("TITLE", "FEC Decoding Segment " +
                            (header.getSegmentNum() + 1) + " of " + 
                            header.getSegments());
                        titleBoxTmp.set("CONTENT", "<p>FEC decoding " +
                            "missing data blocks... this may take " +
                            "a while; please be patient.</p>");
                        titleBoxTmp.toHtml(ppw);
			pageTmp.set("TITLE", ((int)(context.progress()*100.0))+"% - "+
				    HTMLEncoder.encode(context.filename()));
                    }
                    else if (context.status.statusCode() == SplitFileStatus.INSERTING_BLOCKS) {
                        //titleBoxTmp.set("TITLE", "Current Download Status");
                        //titleBoxTmp.set("CONTENT", "<p>Re-inserting " + context.status.reinsertions() +
                        //                " reconstructed blocks " +
                        //                " to heal the network.  </p>");
                        //titleBoxTmp.toHtml(ppw);
                        // TODO: show the status display for the inserted blocks???
                        // well, we shouldn't get here anymore. background healing.
                        // print a warning if we do.
                        logger.log(this, "Warning! Reached State INSERTING_BLOCKS!", Logger.ERROR);
                        //renderTopStatusFrame(ppw, context);
                        //renderRunningUploadStatus(ppw, context);
                    } else if (context.status.statusCode() == 
			       SplitFileStatus.VERIFYING_CHECKSUM) {
                        renderTopStatusFrame(ppw, context);
                        titleBoxTmp.set("TITLE", "Current Download Status");
                        titleBoxTmp.set("CONTENT", "<p> Verifying checksum: " +
                                        context.status.checksum() + " </p>");
                        titleBoxTmp.toHtml(ppw);
			pageTmp.set("TITLE", "Verifying Checksum - "+
				    HTMLEncoder.encode(context.filename()));
                    } else {
                        renderTopStatusFrame(ppw, context);
                        renderRunningDownloadStatus(ppw, context);
                        pageTmp.set("TITLE", ((int)(context.progress()*100.0))+"% - "+
				    HTMLEncoder.encode(context.filename()));
                    }
                }
                
                break;
            case STATE_FILTERING_DATA:
                renderTopStatusFrame(ppw, context);
                titleBoxTmp.set("TITLE", "Current Download Status");
                titleBoxTmp.set("CONTENT", "<p>Running the anonymity filter. REDFLAG: UNIMPLEMENTED!!!!");
                titleBoxTmp.toHtml(ppw);
		pageTmp.set("TITLE", "Filtering data - "+
			    HTMLEncoder.encode(context.filename()));
                break;
            case STATE_FILTER_FAILED:
                // Hmmmm... what's the right thing to display for this case.
                titleBoxTmp.set("TITLE", "Current Download Status");
                titleBoxTmp.set("CONTENT", "Anonymity filter failed. REDFLAG: UNIMPLEMENTED!!!!");
		pageTmp.set("TITLE", "Anonymity Filter Failed! - "+
			    HTMLEncoder.encode(context.filename()));
                titleBoxTmp.toHtml(ppw);
                break;
            case STATE_SENDING_DATA:
                // Hmmmm... what's the right thing to display for this case.
                titleBoxTmp.set("TITLE", "Current Download Status");
                titleBoxTmp.set("CONTENT", "<p>Download finished successfully. " +
                    "Sending data to the browser...</p>" +
                    "<p><b>Time Elapsed:</b> " +
                    timeDistance(context.startTime, Calendar.getInstance()) +
				"</p>"); // we don't have endTime yet
		pageTmp.set("TITLE", "Sending data - "+
			    HTMLEncoder.encode(context.filename()));
                titleBoxTmp.toHtml(ppw);
                break;
            case STATE_DONE:
                titleBoxTmp.set("TITLE", "Current Download Status");
                titleBoxTmp.set("CONTENT", "<p>Download finished successfully.</p>" +
                    "<p><b>Time for Completion:</b> " +
                    timeDistance(context.startTime, context.endTime) + "</p>");
                titleBoxTmp.toHtml(ppw);
		pageTmp.set("TITLE", "Completed - "+
			    HTMLEncoder.encode(context.filename()));
                break;
            case STATE_FAILED:
                sw = new StringWriter();
                pw = new PrintWriter(sw);
                
                titleBoxTmp.set("TITLE", "Current Download Status");
                if (context.errMsg != null) {
                    pw.println("<p>" + context.errMsg + "</p>");
                } else {
                    pw.println("<p>Download failed. Couldn't get status of last block.</p>");
                }
                pw.println("<p>Click <a href=\""+ HTMLEncoder.encode(URLEncoder.encode(context.retryURL())) +"\" "+
                           "target=\"_top\">here</a> to retry.</p>");
                titleBoxTmp.set("CONTENT", sw.toString());
                titleBoxTmp.toHtml(ppw);
		pageTmp.set("TITLE", "Failed - "+HTMLEncoder.encode(context.filename()));
		
                break;
            case STATE_CANCELED:
                renderTopStatusFrame(ppw, context);
                titleBoxTmp.set("TITLE", "Current Download Status");
		String content = "<p>Download aborted.  The client dropped"+
		    " the connection, the Freenet request timed out, or it"+
		    " was manually cancelled.</p>";
                titleBoxTmp.set("CONTENT", content);
                titleBoxTmp.toHtml(ppw);
		pageTmp.set("TITLE", "Cancelled - "+HTMLEncoder.encode(context.filename()));
                break;
            default:
            }

            if (updating) {
                ppw.println("<p>");
                ppw.println("[&nbsp;<a href=\"" + HTMLEncoder.encode(URLEncoder.encode(context.progressURL())) +
                           "\">Update Status Info</a>&nbsp;] ");
                ppw.println("[&nbsp;<a href=\"" + HTMLEncoder.encode(URLEncoder.encode(context.cancelURL())) +
                           "\">Cancel Download</a>&nbsp;]</p>");

            }
        }

        pageTmp.set("BODY", psw.toString());
        pageTmp.toHtml(resp.getWriter());
        if ((context.refreshIntervalSecs > 0) && updating) {
            // Optional client pull updating.
            resp.setHeader("Refresh", Integer.toString(context.refreshIntervalSecs));
        }
        resp.setStatus(HttpServletResponse.SC_OK);
        resp.setContentType("text/html");
        resp.getWriter().flush();
    }

    // checked to here

    protected void handleUnknownCommand(HttpServletRequest req, HttpServletResponse resp,
                                    SFRContext context)
        throws ServletException, IOException {
        
        sendHtml(resp, HttpServletResponse.SC_BAD_REQUEST,
                 "<html> " +
                 "<head> " +
                 "<title>Unknown Command</title> " +
                 "</head> " +
                 "<body> " +
                 "<h1>Unknown Command</h1> " +
                 "</body> " +
                 "</html> ");

    }

    // LATER: Copy scipients code from from VirtualClient to
    //        automagically map requests to functions using
    //        Class.getMethod()? Would belong in base.
    // PRO: 1337.
    // CON: might mess up compiled java efforts
    //      obscures control flow.
    //      not nesc.? When has that ever kept a hack out of the codebase ;-)

    protected void doGet(HttpServletRequest req, HttpServletResponse resp)
        throws ServletException, IOException {
        
        String uri = null;
        
        try {
            uri = URLDecoder.decode(req.getRequestURI());
            //System.err.println("sfrs uri: " + uri);
            //System.err.println("servlet path: " + req.getServletPath());
        }
        catch (URLEncodedFormatException uefe) {
            handleBadURI(req, resp);
            return;
        }

        SFRContext context = (SFRContext)getContextFromURL(uri);

        if (context == null) {
            if (hasContextID(uri)) {
                // Context ID was there but it's
                // bogus or expired.

                // Implement in base.
                handleBadContext(req, resp);
                return;
            }
            else {
                // No context ID.
                // Start a new context.
                onNewContext(req, resp);
                return;
            }
        }

        String cmd = getFirstPathElement(req.getRequestURI());
        if (cmd.equals("download")) {
            onDownload(req, resp, context);
        }
        else if (cmd.equals("cancel")) {
            onCancel(req, resp, context);
        }
        else if (cmd.equals("override_filter")) {
            onOverrideFilter(req, resp, context);
        }
        else if (cmd.equals("parameter_form")) {
            onParameterForm(req, resp, context);
        }
        else if (cmd.equals("split_bottom")) {
            onSplitBottom(req, resp, context);
        }
        else if (cmd.equals("status_progress")) {
            onStatusProgress(req, resp, context);
        }
        else {
            handleUnknownCommand(req, resp, context);
        }
    }

    private String timeDistance(Calendar start, Calendar end) {
	if(end == null) throw new IllegalStateException("end NULL!");
	if(start == null) throw new IllegalStateException("start NULL!");
        String result = "";
        long tsecs = (end.getTime().getTime() - start.getTime().getTime()) / 1000;
        int sec = (int) (tsecs % 60);
        int min = (int) ((tsecs / 60) % 60);
        int hour = (int) ((tsecs / 3600) % 24);
        int days = (int) (tsecs / (3600 * 24));
        /* I hope we don't have to calculate weeks or months... :) */
        if (days > 0) {
            result += days + " day" + ((days != 1) ? "s" : "");
        }
        if (hour > 0) {
            result += ((days > 0) ? ", " : "") + hour + " hour" + ((hour != 1) ? "s" : "");
        }
        if (min > 0) {
            result += (((days + hour) > 0) ? ", " : "") + min + " minute" + ((min != 1) ? "s" : "");
        }
        if (sec > 0) {
            result += (((days + hour + min) > 0) ? ", " : "") + sec + " second" + ((sec != 1) ? "s" : "");
        }
        return result;
    }

    private String format(long bytes) {
        if (bytes == 0) return "None";
	if (bytes%1152921504606846976L == 0) return 
						 nf.format(bytes/1152921504606846976L)
						 + "EiB";
	if (bytes%1125899906842624L == 0) return nf.format(bytes/1125899906842624L) +
					      " PiB";
	if (bytes%1099511627776L == 0) return nf.format(bytes/1099511627776L) + " TiB";
        if (bytes%1073741824 == 0) return nf.format(bytes/1073741824) + " GiB";
        if (bytes%1048576 == 0) return nf.format(bytes/1048576) + " MiB";
        if (bytes%1024 == 0) return nf.format(bytes/1024) + " KiB";
        return nf.format(bytes) + " Bytes";
    }


    ////////////////////////////////////////////////////////////
    // BaseContext subclass containing code to request
    // the SplitFile from Freenet and information about
    // the request's progress.
    ////////////////////////////////////////////////////////////


    // DESIGN DECISION:
    // Keep presentation in SplitFileRequestServlet members.
    // Only Servlet stuff in the SFRContext should be
    // related to getting parameters and sending
    // the data.
    //
    /*class SFRContext extends BaseContext {
        
        SFRContext(long lifeTimeMs, String uri, String path) {
            super(lifeTimeMs);
            this.uri = uri;
            this.path = path;
            setDefaultContextValues(this);
            startTime = Calendar.getInstance();
            try {
                pageTmp = HtmlTemplate.createTemplate("EmptyPage.html");
                titleBoxTmp = HtmlTemplate.createTemplate("aqua/titleBox.tpl");
            } catch (IOException ioe1) {
                logger.log(this, "Template Initialization Failed!" + ioe1.getMessage(), Logger.ERROR);
            }
        }

        int state = STATE_INIT;

        // Stash some info about why the request failed.
        String errMsg = "";

        boolean canceling = false;
        Thread downloadThread = null;

        // Does actual freenet request
        AutoRequester requester;
        // Receives download status events that we use
        // to update the presentation.
        SplitFileStatus status;

        String uri; // completely unencoded. Must be URLEncoded, then HTMLEncoded, if it is to be included in HTML code. Same applies to get*URL()
        // Keep root servlet path for making redirects? Push into base?
        String path;
        String mimeType;
        String contentDesc;
        SplitFile sf;

        // MUST free this.
        Bucket data;

        // SplitFile request parameters
        int htl;
        int blockHtl;
        int retries;
        int retryHtlIncrement;
        int threads;
        boolean doParanoidChecks;
        int healHtl;
        int healPercentage;
	File writeDir = null;
	
        boolean forceSave = false;
        boolean skipDS = false;
        boolean useUI = true;
	//int useUIMinSize = 1<<21;

        // Anonymity filtering stuff
        boolean runFilter = false;
	boolean filterParanoidStringCheck = false;

        int refreshIntervalSecs;

	String decoderErrMsg = null;

        HtmlTemplate titleBoxTmp;
        HtmlTemplate pageTmp;
        Calendar startTime;
        Calendar endTime;
        SimpleDateFormat dateFormat = new SimpleDateFormat("EEE, d MMM yyyy HH:mm:ss z");

        synchronized void setState(int value) {
            state = value;
        }*/
        
        /* Updates parameters from values set in urls query
	 * params.
	 * @return null if ok, an error message if not
	 */
        /*synchronized String updateParameters(HttpServletRequest req) {
            // hmmm... hard coded limit constants ok for now.
            blockHtl = ParamParse.readInt(req, logger, "blockHtl", blockHtl, 0, 100);
            retryHtlIncrement = ParamParse.readInt(req, logger,
                                                   "retryHtlIncrement", retryHtlIncrement, 0, 100);
            healHtl = ParamParse.readInt(req, logger, "healHtl", healHtl, 0, 100);
            healPercentage = ParamParse.readInt(req, logger, "healPercentage", healPercentage, 0, 100);
            retries = ParamParse.readInt(req, logger, "retries", retries, 0, 10);
            threads = ParamParse.readInt(req, logger, "threads", threads, 0, 100);
            useUI = ParamParse.readBoolean(req, logger, "useUI", useUI);
	    //useUIMinSize = ParamParse.readInt(req, logger, "useUIMinSize", useUIMinSize, 0, Integer.MAX_VALUE);
	    
	    if(ParamParse.readBoolean(req, logger, "writeToDisk", false)) {
		String filename = req.getParameter("saveToDir");
		if(filename == null || filename.equals("")) {
		    return "If writing directly to disk, you must specify a folder";
		} else {
		    writeDir = new File(filename);
		    if(!writeDir.exists()) return "You must specify the name of an existing folder";
		    if(!writeDir.isDirectory() || !writeDir.canWrite())
			return "You must specify the name of an existing folder, writable by the node";
		}
	    }
            // NOTE: checkbox's return nothing at all in the unchecked
            //       state.
            if (ParamParse.readBoolean(req, logger, "usedForm", false)) {
                forceSave = ParamParse.readBoolean(req, logger, "forceSaveCB", false);
                skipDS = ParamParse.readBoolean(req, logger, "skipDSCB", false);
                runFilter = ParamParse.readBoolean(req, logger, "runFilterCB", false);
		filterParanoidStringCheck = ParamParse.readBoolean(req, logger, "filterParanoidStringCheck", false);
            } else {
                forceSave = ParamParse.readBoolean(req, logger, "forceSave", forceSave);
                skipDS = ParamParse.readBoolean(req, logger, "skipDS", skipDS);
                runFilter = ParamParse.readBoolean(req, logger, "filter", runFilter);
		filterParanoidStringCheck = ParamParse.readBoolean(req, logger, "paranoidFilter", false);
            }
	    return null;
        }
        

        // REDFLAG: BUG: Canceling isn't working reliably yet.
        //               Need to look at how AutoRequester handles
        //               having it's thread interrupted.
        //               Also need to check that FEC decode
        //               plugin implementation handles interrupting
        //               correctly.

        // DO NOT call this on the downloading thread.
        // Cancels and releases all resources as soon as possible.
        synchronized void cancel() {
            if (canceling) {
                return; // Already canceling.
            }
            canceling = true;
            switch (state) {
            case STATE_INIT:
                setState(STATE_CANCELED);
                break;
            case STATE_REQUESTING_METADATA:
                // There is no race condition because aborting
                // a request that hasn't started yet causes
                // the request to abort when it is started.
                requester.abort();
                downloadThread.interrupt();
                break;
            case STATE_STARTING:
                setState(STATE_CANCELED);
                break;
            case STATE_WORKING:
                requester.abort();
                downloadThread.interrupt();
                break;
            case STATE_FILTER_FAILED:
                setState(STATE_CANCELED);
                cleanup();
                break;
            case STATE_FILTERING_DATA:
            case STATE_SENDING_DATA:
                // Drop through on purpose.
                downloadThread.interrupt();
                break;
                
            case STATE_DONE:
            case STATE_FAILED:
            case STATE_CANCELED:
                // All NOPs. Fall through on purpose.
                break;
            default:
                throw new RuntimeException("Unknown state: " + state);
            }
        }

        // Releases resources for request, but leaves UI around
        // MUST NOT THROW. 
        void cleanup() {
            if (data != null) {
                try {bf.freeBucket(data);} catch (Exception e) {}
                data = null;
            }
        }
        
        // Releases context -- i.e. no more UI
        public boolean reap() {
            synchronized (SFRContext.this) {
                if ((state != STATE_INIT ) &&
                    (state != STATE_STARTING ) &&
                    (state != STATE_FILTER_FAILED ) &&
                    (state != STATE_DONE ) &&
                    (state != STATE_FAILED) &&
                    (state != STATE_CANCELED)) {
                    
                    // REDFLAG: test this code path.
                    // Cleanly cancel the context times out.
                    cancel();
                    return false;
                }
            }

            try {
                cleanup();
            }
            finally {
                // IMPORTANT: Deletes the context from the lookup table.
                return super.reap();
            }
        }

        boolean getMetadata(HttpServletRequest req, HttpServletResponse resp)
            throws ServletException, IOException, FilterException {
            
            synchronized(SFRContext.this) {
                if (state != STATE_INIT) {
                    throw new 
                        IllegalStateException("You can't restart a SplitFile metadata request.");
                }
                
                // Save in case we need to interrupt() later.
                downloadThread = Thread.currentThread();

                requester = new AutoRequester(cf);
                // Stops downloading when it hits the SplitFile metatata.
                requester.setHandleSplitFiles(false);
                
                setState(STATE_REQUESTING_METADATA);
            }
            
            boolean success = false;

            try {
                if (!requester.doGet(uri, new ArrayBucket(), htl)) {
                    logger.log(this, "METADATA REQUEST FAILED.", Logger.ERROR);
                    if (!canceling) {
                        // LATER: nicer error reporting
                        // AutoRequester can return errors that are like line noise.
                        //
                        // LATER: add retrying? This should never happen because
                        //        fproxy had to retrieve the metadata before
                        //        it redirected to this servlet.

                        errMsg = errMsg = "Download failed. Couldn't read SplitFile Metadata";
                        setState(STATE_FAILED);
                        handleMetadataRequestFailed(req, resp, this);
                    }
                    return false; // Finally handles cancels.
                }
                // Extract mime type.
                mimeType = MimeTypeUtils.fullMimeType(null, requester.getMetadata(), uri);

                synchronized(SFRContext.this) {
                    // Recheck predicate after acquiring lock.
                    if (state == STATE_REQUESTING_METADATA && !canceling) {
                        sf = requester.getMetadata().getSplitFile();
                        if (sf == null) {
                            // No SplitFile metadata.
                            errMsg = "URI isn't a SplitFile!";
                            setState(STATE_FAILED);
                            handleNotASplitFile(req, resp, uri, this);
                            return false;
                        }

                        // We handle non-redundant SplitFiles now.
                        if (sf.getFECAlgorithm() == null) {
                            // check for obsolete decoders
                            if (sf.getObsoleteDecoder() != null) {
                                decoderErrMsg = "This Splitfile requires an obsolete decoder (" +
                                    sf.getObsoleteDecoder() + ") which is not supported. Ask the content " +
                                    "author to re-insert it in a supported format.";
                            } 
                            else {
                                // No decoder at all.
                                decoderErrMsg = "No FEC decoder specified in SplitFile metadata!";
				logger.log(this, "No FEC decoder specified for "+uri,
					   new Exception("debug"), logger.DEBUG);
                            }
                        }

                        String fecInfo = "none";
                        if ((sf.getFECAlgorithm() != null) && (sf.getCheckBlockCount() > 0)) {
                            int redundancy = (100 * sf.getCheckBlockCount()) / sf.getBlockCount();
                            fecInfo = sf.getFECAlgorithm() + " (" + 
                                Integer.toString(redundancy) + "% redundancy)";
                        }
                        
                        // e.g. "120MB video/avi, FEC:OnionDecoder_0 (50% redundancy)";
                        contentDesc = ", &nbsp; &nbsp; " + formatByteCount(sf.getSize()) + " " +
                            mimeType + ", FEC decoder: " + fecInfo;

                        // Groovy.
                        setState(STATE_STARTING);
                    }
                }
            }
            catch (Exception e) {
                logger.log(this, "UNEXPECTED EXCEPTION: ", e, Logger.ERROR);
            }
            finally {
                synchronized (SFRContext.this) {
                    downloadThread = null;
                    requester = null;
                    success = state == STATE_STARTING;

                    if ( (state != STATE_STARTING) && 
                         (state != STATE_CANCELED)) {
                        if (canceling) {
                            setState(STATE_CANCELED);
                        }
                        else {
                            setState(STATE_FAILED);
                        }
                    }
                }
            }
            return success;
        }

        // DO NOT call this on the downloading thread.
        synchronized void handleDroppedConnection() {
            cancel();
            errMsg = "The client dropped the connection.";
        }

	void preSetupFilter() {
	    filter = ContentFilterFactory.newInstance(filterPassThroughMimeTypes);
	    String[] s = MimeTypeUtils.splitMimeType(mimeType);
	    mimeTypeParsed = s[0];
	    charset = s[1];
	    try {
		filter.wantFilter(mimeTypeParsed, charset);
	    } catch (FilterException e) {
		forceSave = true; // Force to disk if unrecognized MIME type
		// At this stage this can only be caused by an unrecognized MIME type. Simply send it as application/octet-stream and we're okay. Most uses will go through fproxy anyhow, and will therefore have runFilter=false after the confirmation dialog
		mimeType = DEFAULT_MIME_TYPE;
	    }
	}

	void setupFilter() {
	    // FIXME: could the MIME type have changed between preSetupFilter and setupFilter? Presuming so...
 	    String[] s = MimeTypeUtils.splitMimeType(mimeType);
 	    mimeTypeParsed = s[0];
 	    charset = s[1];
	    try {
		if(!filter.wantFilter(mimeTypeParsed, charset))
		    filter = null; // Don't filter if don't need to filter
		else {
		    // Filter it
		}
	    } catch (FilterException e) {
		filter = null; // At this stage this can only be caused by an unrecognized MIME type. Simply send it as application/octet-stream and we're okay. Most uses will go through fproxy anyhow, and will therefore have runFilter=false after the confirmation dialog
		mimeType = DEFAULT_MIME_TYPE;
	    }
	}
	
	void doBackgroundRequest() {
	    logger.log(this, "doBackgroundRequest", Logger.ERROR);
	    new RequestThread().start();
	}
	
	class RequestThread extends Thread {
	    public void run() {
		logger.log(this, "Starting RequestThread", Logger.ERROR);
		try {
		    doRequest(null, null);
		} catch (Throwable t) {
		    logger.log(this, "Got Throwable trying to do request in background, aborting", t,
			       Logger.ERROR);
		} finally {
		    logger.log(this, "Finished RequestThread", Logger.ERROR);
		}
	    }
	}
	
        // Synchronous
        void doRequest(HttpServletRequest req, HttpServletResponse resp)
            throws ServletException, IOException, FilterException {
	    
	    
            synchronized(SFRContext.this) {
                if (state != STATE_STARTING) {
                    throw new 
                        IllegalStateException("You can't restart a SplitFile request.");
                }
                
                // Save in case we need to interrupt() later.
                downloadThread = Thread.currentThread();

                setState(STATE_WORKING);

                // updateParameters() is now called from onSplitBottom()
                // because that is were the parameter form submits the
                // parameters to. We can't get them anymore once we are
                // here. /Bombe
                //updateParameters(req);
            
                requester = new AutoRequester(cf);
                status = new SplitFileStatus() {
                        // Anonymous Adapter causes the context
                        // to get touch()ed every time a SplitFileEvent
                        // is received.  This keeps the Reaper from
                        // releasing it while the request is in progress.
                        public void receive(ClientEvent ce) {
                            if (!(ce instanceof SplitFileEvent)) {
                                return;
                            }
                            super.receive(ce);
                            touch();
                        }
                    };


                requester.setHandleSplitFiles(true);
                requester.setBlockHtl(blockHtl);
                requester.setSplitFileRetries(retries);
                requester.setSplitFileRetryHtlIncrement(retryHtlIncrement);
                requester.setSplitFileThreads(threads);
                requester.setNonLocal(skipDS);

                // TODO: make parameters, add to GUI options.

                // Ask the AutoRequester to re-insert 5% of the
                // unretrievable SplitFileBlocks at an HTL of 5.
                //
                // "Healing" the network like this should make 
                // SplitFiles much more easily retrievable.
                // If most people do it, that is...
                requester.setHealPercentage(healPercentage);
                requester.setHealingHtl(healHtl);

                requester.enableParanoidChecks(doParanoidChecks);

                requester.setBackgroundInserter(BackgroundInserter.getInstance());

                // Uncomment this to get really verbose status info dumped
                // to std.err.
                //
                // requester.addEventListener(new 
                //    freenet.client.cli.CLISplitFileStatus( new PrintWriter(System.err)));

                requester.addEventListener(status);
                data = bf.makeBucket(-1);
            }

            //System.err.println("------------------------------------------------------------");
            //System.err.println("htl                        : " + htl);
            //System.err.println("blockHtl                   : " + blockHtl);
            //System.err.println("retries                    : " + retries);
            //System.err.println("retryHtlIncrement          : " + 
            //                   retryHtlIncrement);
            //System.err.println("healHtl                    : " + healHtl);
            //System.err.println("healPercentage             : " + healPercentage);
            //System.err.println("threads                    : " + threads);
            //System.err.println("doParanoidChecks           : " + 
            //                   doParanoidChecks);
            //System.err.println("forceSave                  : " + forceSave);
            //System.err.println("skipDS                     : " + skipDS);
            //System.err.println("useUI                      : " + useUI);
	    //System.err.println("useUIMinSize               : " + useUIMinSize);
            //System.err.println("runFilter                  : " + runFilter);
            //System.err.println("filterParanoidStringCheck  : " + 
            //                   filterParanoidStringCheck);
            //System.err.println("------------------------------------------------------------");
	    
            OutputStream out = null;
            if (forceSave || mimeType.equals(DEFAULT_MIME_TYPE)) {
                // IMPORTANT: 
                // We can't send back an html error messages on this response.
                // once the headers are set.  See below. If the user
                // is using the UI, the msg should be rendered in the
                // status pane.  
                //
                // If they aren't running the UI the connection just 
                // drops without all the data.
                // REDFLAG: Underwhelming but I can't see how 
                //          to do better. :-(
                // REDFLAG: This will cause grief with clients like wget 
                //          that retry if the error is not recoverable.
                // REVIST: Do I always need to do this? i.e. will the browser
                //         really wait tens of minutes without getting anything
                //         back on the socket?
                
                // Send the headers immediately so that the browser pops
                // up the save dialog box.
                mimeType = DEFAULT_MIME_TYPE;
		if(resp != null) out = sendDataHeaders(resp);
		runFilter = false; // don't filter unless need to - we can't do anything with application/octet-stream
		filter = null;
            }

            ConnectionPoller poller = null;
            // RELEASE LOCK.
            try {
                if (req != null && pollForDroppedConnections) {
                    poller = new ConnectionPoller(req, 10000,
                                                  // Anonymous adapter to cancel the request
                                                  // if the connection is dropped.
                                                  new Runnable() { 
						      public void run() { handleDroppedConnection(); }
						  });
                }
                
                
                if (!requester.doGet(uri, data, htl)) {
                    if (!canceling) {
                        // LATER: nicer error reporting
                        // AutoRequester can return errors that are like line noise.
                        errMsg = requester.getError();
                        if (errMsg != null) {
                            if (errMsg.equals("")) {
                                errMsg = "Download failed. Couldn't get status of last segement.";
                            }
                        }
                        setState(STATE_FAILED);
                        if (!forceSave) {
                            // Can only send html if we haven't already
                            // sent the headers to setup for sending the data.
			    if(req != null && resp != null)
				handleRequestFailed(req, resp, this);
                        }
                    }
                    return; // Finally handles cancels.
                }
		// IF we run the filter, we run it during download, streaming
//                 if (runFilter) {
//                     // We already have the mime type
//                     setState(STATE_FILTERING_DATA);
//                     try {
//                         // Must be interruptable.
//                         filterData();
//                     }
//                     catch (FilterException fe) {
//                         synchronized (SFRContext.this) {
//                             downloadThread = null;
//                             setState(STATE_FILTER_FAILED);
//                         }
//                         if (!forceSave) {
//                             handleFilterException(req, resp, fe, this);
//                         }
//                         return;
//                     }
//                 }

                if (poller != null) {
                    // Don't let the poller run while sending
                    // data because there would be a race
                    // condition if the client dropped
                    // the connection after receiving all the
                    // data but before setState(STATE_DONE) was
                    // called. 
                    //
                    // The socket writes will throw if
                    // connection has been dropped anyway.
                    poller.stop();
                    poller = null;

                }
                setState(STATE_SENDING_DATA);
                // SENDING
                // Must be interruptable.
		if(resp != null) {
		    if (out == null) {
			sendData(resp);
		    }
		    else {
			sendDataWithoutHeaders(out);
			out = null; // call above closes out.
		    }
		} else {
		    writeData();
		}
                setState(STATE_DONE);
            }
            catch (InterruptedException ie) {
                // NOP
                // cancel() interrupt()'s this thread.
            }
            finally {
                if (poller != null) {
                    poller.stop();
                }

                if (out != null) {
                    try { out.close(); } catch(Exception e) {}
                }
                synchronized (SFRContext.this) {
                    downloadThread = null;
                    if (state != STATE_FILTER_FAILED) {
                        cleanup();
                    }
                    if ( (state != STATE_DONE) && 
                         (state != STATE_FAILED) &&
                         (state != STATE_CANCELED) &&
                         (state != STATE_FILTER_FAILED)) {
                        if (canceling) {
                            // REDFLAG: why didn't I set errMsg here?
                            setState(STATE_CANCELED);
                        }
                        else {
                            if (state == STATE_SENDING_DATA) {
                                errMsg = "The file was downloaded from Freenet but couldn't be " +
                                    "sent to the browser.  Maybe the user dropped the " +
                                    "connection?";
                            }
                            setState(STATE_FAILED);
                        }
                    }
                }
		endTime = Calendar.getInstance();
            }
        }

        // Synchronous
        // Called to send the data after the filter has failed.
        void doOverrideFilter(HttpServletRequest req, HttpServletResponse resp)
            throws ServletException, IOException, FilterException {

            try {
                synchronized(SFRContext.this) {
                    if (state != STATE_FILTER_FAILED) {
                        throw new 
                            IllegalStateException("Filter didn't fail?");
                    }
                    
                    // Save in case we need to interrupt() later.
                    downloadThread = Thread.currentThread();
                    
                    setState(STATE_SENDING_DATA);
                }

                // Check to see if the mime type was overriden in
                // the url's query parameters.
                String oldMimeType = mimeType;
                updateParameters(req);
                if (mimeType != oldMimeType) {
                    mimeType = MimeTypeUtils.fullMimeType(mimeType, null, null);
                }
                
                // SENDING
                // Must be interruptable.
		if(resp != null) sendData(resp);
		else writeData();
                setState(STATE_DONE);
                
            }
            catch (InterruptedException ie) {
                // NOP
                // cancel() interrupt()'s this thread.
            }
            finally {
                synchronized (SFRContext.this) {
                    downloadThread = null;
                    cleanup();
                    if ( state != STATE_DONE) {
                        if (canceling) {
                            setState(STATE_CANCELED);
                        }
                        else {
                            if (state == STATE_SENDING_DATA) {
                                errMsg = "The file was downloaded from Freenet but couldn't be " +
                                    "sent to the browser.  Maybe the user dropped the " +
                                    "connection?";
                            }
                            setState(STATE_FAILED);
                        }
                    }
                }
            }
        }
	
	ContentFilter filter = null;
	String mimeTypeParsed = null;
	String charset = null;
	
//         void filterData() throws FilterException, InterruptedException {
//             synchronized (SFRContext.this) {
//                 if (state != STATE_FILTERING_DATA) {
//                     throw new IllegalStateException();
//                 }
//             }
//         }
	
        OutputStream sendDataHeaders(HttpServletResponse resp) 
            throws ServletException, IOException {
	    
            // Set response headers, 200
            resp.setStatus(HttpServletResponse.SC_OK);
            resp.setContentType(mimeType);
            resp.setContentLength((int) sf.getSize());
            OutputStream out = resp.getOutputStream();
            try {
                out.flush();
            }
            catch (IOException ioe) {
                try { out.close(); } catch (Exception e) {}
                throw ioe;
            }
	    
            return out;
        }
        
        void sendDataWithoutHeaders(OutputStream out)
            throws ServletException, IOException, InterruptedException {
            InputStream in = null;
            try {
		if(runFilter && filter != null) {
		    in = filter.run(data, mimeTypeParsed, charset);
		} else {
		    in = data.getInputStream();
		}
                // stream bucket to output
                byte[] buf = new byte[16384];
                int bytes = 0;
                while ((bytes = in.read(buf)) > 0) {
                    // Hmmmm.... yeild a little?
                    if (canceling) {
                        throw new InterruptedException("Request was canceled while copying data");
                    }
                    out.write(buf, 0, bytes);
                }
                
                out.flush();
            }
            finally {
                if (in != null) {
                    try { in.close(); } catch (Exception e) {}
                }
            }
        }
        
	void writeData() throws ServletException, IOException, InterruptedException {
	    if(writeDir == null) throw new IllegalStateException("writeDir null in writeData!");
	    String key = null;
	    String filename = null;
	    int p;
	    if ((p = uri.lastIndexOf("/")) != -1) {
		key = uri.substring(0, p);
		filename = uri.substring(p + 1);
	    } else {
		key = uri;
	    }
	    if(filename == null) filename = key;
	    FileOutputStream os = new FileOutputStream(new File(writeDir, key));
	    sendDataWithoutHeaders(os);
	}
	
        void sendData(HttpServletResponse resp) 
            throws ServletException, IOException, InterruptedException {
            
            synchronized (SFRContext.this) {
                if ((state != STATE_SENDING_DATA) &&
                    (state != STATE_FILTER_FAILED)) {
                    throw new IllegalStateException();
                }
            }
            
            sendDataWithoutHeaders(sendDataHeaders(resp));
        }

        synchronized boolean isUpdating() {
            return (state != STATE_DONE) &&
                (state != STATE_FAILED) && 
                (state != STATE_CANCELED);
        }
        
        // URI to re-run this request from
        // scratch. 
        synchronized String retryURL() {
            return path + "/" + uri;
        }

        synchronized String cancelURL() {
            return path + "/" + makeContextURL(DUMMY_TAG + "/cancel");
        }

        synchronized String overrideFilterURL() {
            return path + "/" + makeContextURL(DUMMY_TAG + "/override_filter");
        }

        synchronized String progressURL() {
            return path + "/" + makeContextURL(DUMMY_TAG + "/status_progress");
        }

        synchronized String parameterURL() {
            return path + "/" + makeContextURL(DUMMY_TAG + "/parameter_form");
        }

        synchronized String splitBottomURL() {
            return path + "/" + makeContextURL(DUMMY_TAG + "/split_bottom");
        }

        synchronized String downloadURL() {
	    int x = uri.lastIndexOf('/');
	    String s = uri;
	    if(x > 0 && x < (uri.length()-1))
		s = s.substring(x, uri.length());
            return path + "/" + makeContextURL(DUMMY_TAG + "/download/"+s);
        }
    }*/
    
    // Sets contexts parameters to the servlet's
    // defaults.
    protected void setDefaultContextValues(SFRContext context) {
        context.logger = logger;
        context.cf = cf;
        context.bf = bf;
       
        context.htl = defaultHtl;
        context.blockHtl = defaultBlockHtl;
        context.retries = defaultRetries;
        context.retryHtlIncrement = defaultRetryHtlIncrement;
        context.healHtl = defaultHealHtl;
        context.healPercentage = defaultHealPercentage;
        context.threads = defaultThreads;
        context.doParanoidChecks = defaultDoParanoidChecks;
        context.refreshIntervalSecs = defaultRefreshIntervalSecs;
        context.forceSave = defaultForceSave;
	context.writeToDisk = defaultWriteToDisk;
	context.disableWriteToDisk = disableWriteToDisk;
        context.skipDS = defaultSkipDS;
        context.useUI = defaultUseUI;
	//context.useUIMinSize = defaultUseUIMinSize;
        context.runFilter = defaultRunFilter;
	context.randomSegs = defaultRandomSegs;
	context.filterParanoidStringCheck = defaultFilterParanoidStringCheck;
    }
    
}








