package freenet;

import freenet.support.Fields;
import java.util.StringTokenizer;

/**
 * Central spot for stuff related to the versioning of the codebase.
 */
public abstract class Version {

    /** FReenet Reference Daemon */
    public static final String nodeName = "Fred";
    
    /** The current version tree */ 
    public static final String nodeVersion = "0.5";
    
    /** The protocolversion supported */
    public static String protocolVersion = "1.46";
    
    /** The build number of the current revision */
    public static final int buildNumber = 5015;

    /** Oldest build of Fred we will talk to */
    public static final int lastGoodBuild = 565;

    /** The highest reported build of fred */
    public static int highestSeenBuild = buildNumber;

    /** Revision number of Version.java as read from CVS */
    public static final String cvsRevision;
    static {    
        StringTokenizer cvsId = new StringTokenizer(
            "$Id: Version.java,v 1.90.2.50.2.41 2003/07/18 21:18:50 amphibian Exp $"
        );
        cvsId.nextToken();
        cvsId.nextToken();
        cvsRevision = cvsId.nextToken();
    }
    
    /**
     * @return  the node's version designators as an array
     */
    public static final String[] getVersion() {
			String[] ret={nodeName, nodeVersion, protocolVersion, ""+buildNumber};
			return ret;
    }

    /**
     * @return  the version string that should be presented
     *          in the NodeReference
     */
    public static final String getVersionString() {
        return Fields.commaList(getVersion());
    }

    /**
     * @return  true if requests should be accepted from nodes
     *          brandishing this version string
     */
    public static final boolean checkGoodVersion(String version) {
        String[] v = Fields.commaList(version);
        if (v.length < 3 || !v[2].equals(protocolVersion)) {
            return false;
        }
        if (sameVersion(v)) {
            try {
                int build = Integer.parseInt(v[3]);
                if (build < lastGoodBuild)
                    return false;
            } catch (NumberFormatException e) {
                return false;
            }
        }
        return true;
    }

    /**
     * @return  why a version string is rejected
     */
    public static final String explainBadVersion(String version) {
        String[] v = Fields.commaList(version);
        if (v.length < 3 || !v[2].equals(protocolVersion)) {
            return "Required protocol version is "+protocolVersion;
        }
        if (sameVersion(v)) {
            try {
                int build = Integer.parseInt(v[3]);
                if (build < lastGoodBuild)
                    return "Build older than last good build "+lastGoodBuild;
            } catch (NumberFormatException e) {
                return "Build number not numeric.";
            }
        }
        return null;
    }


    public static final void seenVersion(String version)  {
        String[] v = Fields.commaList(version);

        if (v.length < 3)
            return; // bad, but that will be discovered elsewhere
        if (sameVersion(v)) {
            
            int buildNo;
            try {
                buildNo = Integer.parseInt(v[3]);
            } catch (NumberFormatException e) {
                return; 
            }
            if (buildNo > highestSeenBuild &&
                (buildNo < 600 || buildNumber >= 600))  {
                // FIX ME. This is a workaround to keep 500 series builds
                // from showing 600+ builds. The first 600 builds had 
                // version set to 0.5 so this was necessary. For the future
                // if we branch a stable version we should always change
                // the version string.
                // It'll probably be OK to remove this in a few months / weeks.
                
                if (Core.logger.shouldLog(Core.logger.MINOR)) {
                    Core.logger.log(Version.class, 
                                    "Seen new highest build: " + buildNo, 
                                    Core.logger.MINOR);
                }
                highestSeenBuild=buildNo;
            }
        }
    }

    /**
     * Returns if the string describes the same version of the same
     * software as us.
     */
    private static boolean sameVersion(String[] v) {
        return v[0].equals(nodeName) && 
            v[1].equals(nodeVersion) && v.length >= 4;
    }
     
    public static void main(String[] args) throws Throwable {
	System.out.println("Freenet: "+nodeName+" "+nodeVersion+" (protocol "+
			   protocolVersion+") build "+buildNumber+" (last good build: "+
			   lastGoodBuild+")");
    }
}



