/* -*- Mode: java; c-basic-indent: 4; tab-width: 4 -*- */
package freenet;
import freenet.support.*;
import freenet.support.Comparable;
import freenet.support.sort.*;
import freenet.thread.ThreadFactory;
import java.util.Vector;
import java.util.Enumeration;
import java.util.Date;
import java.io.PrintWriter;
/*
  This code is part of the Java Adaptive Network Client by Ian Clarke. 
  It is distributed under the GNU General Public Licence (GPL) 
  version 2.  See http://www.gnu.org/ for further details of the GPL.
 */

/**
 * This class sends MessageObjects to a MessageHandler after a specified
 * amout on time. 
 *
 * It's now based on a heap (so it should be pretty fast), but you have to
 * handle keeping track of objects you may wish to cancel yourself 
 * (I didn't want another fucking hashtable). Since this makes it 
 * Freenet specific, I'm moving it to the root (and Freenet's main instance.
 * of it is an instance, rather than a static, variable in freenet.Core).
 *
 * BTW, it doesn't tick anymore (sigh...)
 *
 * @author oskar
 */

// Ians javadoc for the old ticker.
/* 
 * This class allows a callback to be called after a specified
 * amount of time.
 *
 * @author <A HREF="mailto:I.Clarke@strs.co.uk">Ian Clarke</A>
 */

public class Ticker implements Runnable {
    
    private final Heap events = new Heap();
    private final MessageHandler mh;
    private final ThreadFactory tf;

    //private long maxTaskWaitMs = 30000;

    /**
     * Construct a Ticker with the given tick time in milliseconds.
     * @param mh     The MessageHandler to send events to.
     * @param tm  A ThreadManager to get threads to handle events in from.
     *            If this is null, new threads will be created manually for
     *            each event.
     */
    public Ticker(MessageHandler mh, ThreadFactory tf) {
        this.mh = mh;
        this.tf = tf;
    }

    /**
     * Returns the MessageHandler currently used by this Ticker.
     */
    public final MessageHandler getMessageHandler() {
        return mh;
    }

    /**
     * Schedules a MessageObject.
     * @param time  The time to wait (in milliseconds) before handling the 
     *              MessageObject.  If the MessageObject implements the 
     *              Schedulable interface, then it will be given a TickerToken
     *              it can use to cancel itself.
     * @param mo    The MessageObject to schedule.
     * @see Schedulable
     */
    public final void add(long time, MessageObject mo) {
		if(time<0) {
			Core.logger.log(this, "Scheduling "+mo+" for "+time, 
							new IllegalStateException("scheduling in the past!"), Core.logger.NORMAL); // not ERROR since we have an adequate workaround
			time = 0;
		}
        addAbs(time + System.currentTimeMillis(), mo);
    }

    /**
     * Like add(), but the time is given as an absolute in milliseconds
     * of the epoch (System.currentTimeMillis() format).
     */
    public final synchronized void addAbs(long time, MessageObject mo) {

		long x = System.currentTimeMillis();
		if(time < x) time = x;
	
		Event evt = new Event(mo, time);
	
		if(Core.logger.shouldLog(Core.logger.DEBUG))
			Core.logger.log(this, "scheduling " + evt + 
							" to run at " + time + " at " + x, 
							new Exception("debug"), Logger.DEBUG);
	
		evt.heapElement = events.put(evt);
	
		if (mo instanceof Schedulable) {
			((Schedulable) mo).getToken(evt);
		}
	
		this.notify(); // wake up ticker thread
    }
	
    public void run() {
        Vector jobs = new Vector();
        while (true) {
			boolean logDEBUG = Core.logger.shouldLog(Logger.DEBUG);
			if(logDEBUG) timeLog("Running Ticker loop");
            synchronized (this) {
				if(logDEBUG) timeLog("Synchronized in Ticker loop");
                // prepare ready events for immediate execution
                while (events.size() > 0 && 
                       ((Event) events.top()).time() <= System.currentTimeMillis()) {
					if(logDEBUG) timeLog("Popping events from Ticker loop");
					Event e = (Event) events.pop();
					jobs.addElement(e);
                }
				if(logDEBUG) timeLog("No more events");
                // if none were ready, sleep until one is
                if(jobs.isEmpty()) {
					if(logDEBUG) timeLog("Jobs empty");
                    try {
                        if (events.size() > 0) {
							if(logDEBUG) timeLog("Waiting for next event");
                            long wait = ((Event) events.top()).time() 
                                - System.currentTimeMillis();
							if(logDEBUG) timeLog("Waiting for next event in "+wait+" ms");
                            if (wait > 0) wait(wait);
                        } else {
							if(logDEBUG) timeLog("Waiting for more jobs");
							wait(1000);
						}
                    }
                    catch (InterruptedException e) {
						if(logDEBUG) timeLog("Interrupted");
					}
					if(logDEBUG) timeLog("Continuing Ticker loop");
                    continue; // and jump back to preparing ready events
                }
            }
			if(logDEBUG) timeLog("Released monitor");
            // then execute the events *after* releasing the monitor
            Enumeration e = jobs.elements();
            do { 
				Event v = ((Event) e.nextElement());
				if(logDEBUG) timeLog("Executing "+v);
                v.execute();
				if(logDEBUG) timeLog("Executed "+v);
			}
            while (e.hasMoreElements());
			if(logDEBUG) timeLog("Executed jobs");
            jobs.setSize(0);
			if(logDEBUG) timeLog("Looping outer Ticker loop");
        }
    }
    
    protected void timeLog(String s) {
		if(Core.logger.shouldLog(Logger.DEBUG))
			Core.logger.log(this, s+" at "+System.currentTimeMillis(),
							Logger.DEBUG);
    }
    
    public synchronized String toString() {
        StringBuffer sb = new StringBuffer();
        Enumeration e = events.elements();
        while (e.hasMoreElements()) {
            Event evt = (Event) e.nextElement();
            sb.append(evt.time() + " - "+ Long.toHexString(evt.getOwner().id())
                      + " : " + evt.getOwner().getClass().getName() + "\n");
        }
        return sb.toString();
    }

    public synchronized void writeEventsHtml(PrintWriter pw) {
        pw.println("<h2>Fred Ticker Contents</h2> <b>At date:");
        pw.println(new Date());
        pw.println("</b><table border=1>");
        pw.println("<tr><th>Time</th><th>Event</th></tr>");
        Heap.Element[] el = events.elementArray();
        QuickSorter.quickSort(new ArraySorter(el));
        for (int i = 0 ; i < el.length ; i++) {
            pw.print("<tr><td>"); 
            Event ev =(Event) el[i].content();
            pw.print(new Date(ev.time()));
            pw.print("</td><td>");
            pw.print(ev.mo);
            pw.println("</td></tr>");
        }
        pw.println("</table>");
    }

    private static long eventCounter = 0;
    private static Object eventCounterSync = new Object();
    
    private class Event implements Runnable, TickerToken, Comparable {

        private MessageObject mo;
        private final long time;
        private Heap.Element heapElement;
		// 	private Exception initException;
		private long id;

        private Event(MessageObject mo, long time) {
            this.mo = mo;
            this.time = time;
			synchronized(eventCounterSync) {
				id = eventCounter++;
			}
			// 	    if(Core.logger.shouldLog(Core.logger.DEBUG))
			// 		initException = new Exception("debug");
        }

        private final void execute() {
            tf.getThread(this);
        }

        public final boolean cancel() {
            boolean ret;
            synchronized (Ticker.this) {
                ret = heapElement.remove();
            }
			if(Core.logger.shouldLog(Core.logger.DEBUG))
				Core.logger.log(Ticker.this,
								(ret ? "cancelled " : "failed to cancel ") + 
								this, new Exception("debug"),
								Logger.DEBUG);
            return ret;
        }

        public final long time() {
            return time;
        }

        public final void run() {
			long now = System.currentTimeMillis();
			long delay = now - time;
            Core.diagnostics.occurrenceContinuous("tickerDelay", delay);
			if(delay > 500 && Core.logger.shouldLog(Core.logger.DEBUG)) {
				Core.logger.log(this, "Long tickerDelay ("+delay+
								"), for event for "+time+" at "+now,
								new Exception("debug"), Core.logger.DEBUG);
				// 		Core.logger.log(this, "Delay started:", initException,
				// 				Core.logger.DEBUG);
			}
			if(Core.logger.shouldLog(Core.logger.DEBUG))
				Core.logger.log(Ticker.this, "running " + this 
								+ " at "+ now + " for "+time, 
								Logger.DEBUG);
            try {
                mh.handle(mo);
            }
            catch (RuntimeException e) {
                Core.logger.log(mh, "Unhandled throw in message handling",
                                e, Logger.ERROR);
                throw e;
            } catch (Error e) {
                throw e;
            }
        }

        public final int compareTo(Object o) { 
            long m = ((Event) o).time;
			if(time < m) return 1;
			if(time > m) return -1;
			long oid = ((Event) o).id;
			if(id > oid) return 1;
			if(id < oid) return -1;
			Core.logger.log(this, "oid's equal... what's the chance of that?",
							Core.logger.NORMAL);
			return 0;
        }

        public final String toString() {
            return mo + " @ " + time;
        }

        public final MessageObject getOwner() {
            return mo;
        }
    }
}





