package freenet;

import freenet.node.Node;
import freenet.node.NodeMessageObject;
import freenet.node.State;
import freenet.node.BadStateException;
import freenet.client.ClientMessageObject;
import freenet.support.Fields;

/**
 * Messages represent message objects that can be turned into rawmessages
 * and sent between nodes.
 *
 * @author oskar
 */

public abstract class Message implements NodeMessageObject, 
                                         ClientMessageObject {
    


    /** ConnectionHandler the message was received on. */
    public ConnectionHandler source;

    /** A randomly generated unique ID used to identify related messages */
    public /* final */ long id;
    
    /** Any unknown / unused fields */
    public final FieldSet otherFields;

    /** Whether to close the connection after sending this message */
    protected boolean close = false;
    /** Whether to sustain the connection after sending this message */
    protected boolean sustain = false;

    /** The time this message was received from the network, or -1 */
    protected long receivedTime = -1;

    /**
     * Creates a new message.
     * @param  idnum        The message's Unique ID, should be a random long.
     * @param  otherFields  The remaining set of fields which are not directly
     *                      related to routing the message.
     */
    protected Message(long id, FieldSet otherFields) {
        this.id = id;        
        this.otherFields = otherFields;
    }

    /**
     * Creates a new message
     * @param idnum  The messages Unique Id.
     */
    protected Message(long id) {
        this(id, new FieldSet());
    }
    
    /** Creates a message from a RawMessage with the given id number
     * @param  source  The connection on which the message was received.
     * @param  id      The unique id value to give the message.
     * @param  raw     A rawmessage describing the message body. 
     */
    protected Message(ConnectionHandler source, long id, RawMessage raw) {
        this(id, raw.fs);
        this.source = source;
    }
    
    /** Converts this message to something that can be sent over the wire,
      * using the given Presentation.
      */
    public RawMessage toRawMessage(Presentation t) {
        RawMessage r = t.newMessage(getMessageName(), close , sustain,
                                    otherFields, 0, "EndMessage", null);
        
        return r;
    }

    /** Part of the MessageObject implementation.
      * @return the unique ID of this message
      */
    public long id() {
        return id;
    }

    /**
     * Return the name of the message.
     */
    public abstract String getMessageName();

    /**
     * Part of the NodeMessageObject interface that allows messages to be
     * handled by the node.
     *
     * Override for messages that can start a chain - by default this will
     * throw a BadStateException with the comment "This message is a 
     * response".
     */
    public State getInitialState() throws BadStateException {
        throw new BadStateException("This message is a response");
    }
    
    /**
     * Part of the NodeMessageObject interface. Messages are external.
     */
    public boolean isExternal() {
        return true;
    }

    /**
     * Called on a message object when it is dropped because it was undesired, 
     * should clean up  (killing trailing fields and alike). The default 
     * implementation does nothing.
     */ 
    public void drop(Node n) {}

    /**
     * Set receive time.
     */
    public void setReceivedTime(long time) {
        receivedTime = time;
    }

    /**
     * Get receive time.
     */
    public long getReceivedTime() {
        return receivedTime;
    }

    /** 
     * @return  the Identity of the peer this message was received from,
     *          or null if inapplicable
     */
    public Identity peerIdentity() {
        return source == null ? null : source.peerIdentity();
    }
    
    public String toString() {
        return "freenet.Message: "+getMessageName()
                +" @"+source+" @ "+Long.toHexString(id);
    }
}


