/* -*- Mode: java; c-basic-indent: 4; tab-width: 4 -*- */
package freenet;
//import freenet.transport.VoidAddress;
import freenet.crypt.EntropySource;
import freenet.message.*;
import freenet.node.*;
import freenet.session.*;
import freenet.support.*;
import freenet.support.io.*;
import freenet.thread.*;
import freenet.transport.*;
import java.nio.*;
import java.nio.channels.SelectableChannel;
import java.nio.channels.SocketChannel;
import java.io.*;

/*
 *  This code is part of the Java Adaptive Network Client by Ian Clarke.
 *  It is distributed under the GNU General Public Licence (GPL)
 *  version 2.  See http://www.gnu.org/ for further details of the GPL.
 */
/**
 *  Handles both sending and receiving messages on a connection.
 *
 *@author     oskar (90%)
 *@author     <A HREF="mailto:I.Clarke@strs.co.uk">Ian Clarke</A>
 *@author     <a href="mailto:blanu@uts.cc.utexas.edu">Brandon Wiley</a>
 *@created    June 9, 2002
 */

public final class ConnectionHandler implements Runnable, NIOReader, NIOWriter {

    //  private static PrintStream temp;

//      static {
//          try {
//              temp = new PrintStream(new FileOutputStream("chterms"));
//          } catch (IOException e) {
//              e.printStackTrace();
//          }
//      }

    // Protected/Private Fields

    private final     OpenConnectionManager  ocm;
    private           Link                   link;
    private final     Presentation           p;
    private final     Ticker                 t;

    private           int                    meanMessageSize;
    private           long                   startTime;
    private           int                    maxInvalid;
    private 	      int 		invalid =0;
    private           int                    maxPadding;

    private           int                    messages         = 0;

    /**  Description of the Field */
    public final boolean outbound;

    // state variables

    /**  Last time of noted activity */
    private volatile long lastActiveTime;
    // clock time of last activity

    /**  Whether the ConnectionHandler has been started */
//    private final Irreversible started = new Irreversible(false);

    /**  If no more messages will be read */
    private final Irreversible receiveClosed = new Irreversible(false);

    /**  If no more message can be sent */
    private final Irreversible sendClosed = new Irreversible(false);

    /**  If we should never timeout the connection */
    private final Irreversible persist = new Irreversible(false);

    /**  terminate() reached */
    private Irreversible finalized = new Irreversible(false);

    /**  Object to lock on when receiving */
    private final Object receiveLock = new Object();
    /**  Count of number of number of reads going on (should be 0 or 1) */
    //private final Count receiving=new Count(0, receiveLock);
    // AFM (another fucking monitor)
    private volatile int receivingCount = 0;

    /**  Object to lock on when sending */
    private final Object sendLock = new Object();
    /**  Count objects for number of sends in progress of pending */
    //private final Count sending=new Count(0, sendLock);
    // AFM (another fucking monitor)
    private volatile int sendingCount = 0;

    private volatile  long                   sendQueueSize    = 0;
    //private Thread exec_instance; // execution thread
    private static    EntropySource          sendTimer        = new EntropySource(),
        recvTimer        = new EntropySource();


    /** The thread that is running the retrieve loop */
    private Thread receiveThread;

    private volatile boolean cached = false;

    public final void setCached(boolean value) { cached = value; }
    public final boolean isCached() { return cached; }
	
	private volatile boolean alreadyClosedLink = false;

    /** methods related to nio */
    
    //a buffer size; the maximum message size is 64K
	// Store two messages worth. The buffer for tcpConn is half this currently;
	// we want CH's buffer to be substantially bigger than tcpConn's buffer.
    private final static int BUF_SIZE=67*1024;
    
    //the buffer itself
    private ByteBuffer accumulator;
    //ensure there's a backing byte[] 
    //contrary to common sense, we need to do so.
    private byte[] _accumulator;
	private ByteBuffer rawAccumulator;
	private tcpConnection conn;
    private int decryptLen;
    
    /***try to fight the OOMS***/
    //this should be the same as the Connection buffer which should be exported
    private static final int DECRYPT_SIZE=16*1024; 
    private static final byte[] ciphertext = new byte[DECRYPT_SIZE];
    
    private boolean movingTrailingFields = false;
    private boolean doneMovingTrailingFields = false;
    private boolean disabledInSelector = false;
	private boolean initRegisteredInOCM;
	private boolean reregistering = false;
    private ReadSelectorLoop rsl = null;
    private WriteSelectorLoop wsl = null;
	private Peer peer;
    
    //profiling
    //WARNING:remove before release
    public static volatile int instances=0;
	public static volatile long terminatedInstances=0;
    public static volatile int CHOSinstances=0;
    public static volatile int CHISinstances=0;
    public static volatile int SOSinstances=0;
    public static volatile int RISinstances=0;
    private static final Object profLock = new Object();
    private static final Object profLockCHOS = new Object();
    private static final Object profLockCHIS = new Object();
    private static final Object profLockSOS = new Object();
    private static final Object profLockRIS = new Object();
//    private static final Object profLockTerminated = new Object();

	//CHIOS notification flags
	protected volatile boolean CHISwaitingForNotification=false;
	protected volatile boolean CHISalreadyNotified=false;
	protected volatile boolean CHOSwaitingForNotification=false;
	protected volatile boolean CHOSalreadyNotified=false;
	protected volatile Thread currentCHOSThread=null;
	
	private boolean logDEBUG;

    // Constructors
    /**
     *  The ConnectionHandler provides the interface between the session,
     *  presentation, and application layers. Messages received on l, are parsed
     *  using p, and then turned into messageobjects using the MessageFactory which
     *  are scheduled for immediate execution on the ticker. Message objects given
     *  to the sendmessage are serialized using the p and sent on l. <p>
     *
     *  The outbound argument is a hint used by diagnostics to differentiate
     *  inbound from outbound connections. <p>
     *
     *
     *
     *@param  ocm         The cache of open connections to register with
     *@param  p           A presentation for parsing messages
     *@param  l           A live, initialized link, to read messages off
     *@param  t           A ticker to schedule the messages with.
     *@param  maxInvalid  The maximum number of invalid messages to swallow.
     *@param  maxPad      The maximum number of padding bytes to accept between two
     *      messages.
     *@param  outbound    Set true for outbound connections, false otherwise.
     */
    public ConnectionHandler(OpenConnectionManager ocm,
                             Presentation p, Link l, Ticker t,
                             int maxInvalid, int maxPad, boolean outbound) 
		throws IOException {
		try {
			this.ocm = ocm;
			this.p = p;
			this.meanMessageSize = p.exptMessageSize();
			this.link = l;
			this.t = t;
			lastActiveTime = System.currentTimeMillis();
			this.maxInvalid = maxInvalid;
			this.maxPadding = maxPad;
			this.outbound = outbound;
			
			peer  = new Peer(link.getPeerIdentity(), link.getPeerAddress(),
							 link.getManager(), p);
			this.logDEBUG = Core.logger.shouldLog(Logger.DEBUG);
			if(logDEBUG) logDEBUG("New connectionhandler with "+peer, true);
			
			_accumulator = new byte[BUF_SIZE];
			accumulator = ByteBuffer.wrap(_accumulator);
			conn = (tcpConnection)(link.getConnection());
			if(logDEBUG) logDEBUG("Connection: "+conn);
			if(conn == null) throw new IOException("Already closed!");
			if(conn.isClosed()) throw new IOException("Already closed!");
			if(logDEBUG) logDEBUG("Starting");
			rawAccumulator = conn.getInputBuffer();
			accumulator.limit(0);
			accumulator.position(0);
			startTime = System.currentTimeMillis();
			link.setTimeout(600000); // 10 minutes
			initRegisteredInOCM = false;
			//profiling
			//WARNING:remove before release
			synchronized(profLock) {
				instances++;
			}
		} catch (IOException e) {
			Core.logger.log(this, "IOException constructing "+this+": "+e+
							"("+link+") for "+this, e, Logger.MINOR);
			// Probably closed
			terminate();
			throw e;
		} catch (Error e) {
			terminate();
			Core.logger.log(this, "Got "+e+" in CH.<init> for "+this,
							Logger.ERROR);
			throw e;
		} catch (RuntimeException e) {
			terminate();
			Core.logger.log(this, "Got "+e+" in CH.<init> for "+this,
							Logger.ERROR);
			throw e;
		}
    }
    
	private void logDEBUG(String s) {
		logDEBUG(s, false);
	}
	
	private void logDEBUG(String s, boolean trace) {
		String out = logString(s);
		if(!Core.logger.shouldLog(Logger.DEBUG)) return;
		if(trace) Core.logger.log(this, out, new Exception("debug"),
								  Logger.DEBUG);
		else Core.logger.log(this, out, Logger.DEBUG);
	}
	
	private String logString(String s) {
		return s+" ("+this+","+accumStatus()+") "+
			(movingTrailingFields?"(trailing)":"(not trailing)")+
			(disabledInSelector?"(disabled in selector)":"(reading)");
	}
	
    public void configRSL(ReadSelectorLoop rsl) {
		if(rsl == null) throw new IllegalArgumentException("NULL RSL!");
		if(logDEBUG) logDEBUG("Configured RSL");
		this.rsl = rsl;
    }
    
    public void configWSL(WriteSelectorLoop wsl) {
		if(logDEBUG) logDEBUG("Configuring WSL to "+wsl);
		if(wsl == null) throw new IllegalArgumentException("NULL WSL!");
		this.wsl = wsl;
		if(logDEBUG) logDEBUG("Configured WSL to "+wsl);
    }
	
	public void registerOCM() {
		if(logDEBUG) logDEBUG("registerOCM");
		if(!initRegisteredInOCM) {
			if(logDEBUG) logDEBUG("registering OCM");
			if (peerIdentity() != null)
				ocm.put(this);
			initRegisteredInOCM = true;
		}
		if(logDEBUG) logDEBUG("registered in OCM");
	}
	
	public boolean shouldThrottle() {
		tcpConnection c = conn;
		if(c == null) return false;
		return c.shouldThrottle();
	}
	
	public boolean countAsThrottled() {
		tcpConnection c = conn;
		if(c == null) return false;
		return c.countAsThrottled();
	}
	
	String bufStatus(String name, ByteBuffer b) {
		if(b == null) return name+":(null)";
		else return name+":"+b.position()+"/"+b.limit()+
				 "/"+b.capacity()+"/"+toString();
	}
	
	String accumStatus() {
		return bufStatus("accumulator", accumulator);
	}
	
	private void tryReregister() {
		if(logDEBUG) logDEBUG("tryReregister");
		ByteBuffer a = accumulator;
		if(a == null) {
			if(logDEBUG) logDEBUG("Not reregistering because accumulator null");
			return;
		}
		if(logDEBUG) logDEBUG("Still here");
		if(disabledInSelector && 
		   ((a.capacity() - a.limit()) > 
			(a.capacity()/4))) {
				if(logDEBUG) logDEBUG("Reregistering");
			reregister(); // FIXME: hardcoded
		}
		if(logDEBUG) logDEBUG("Left tryReregister");
	}
	
	/**
	 * Must be called synchronized(accumulator)
	 */
	private void reregister() {
		if(logDEBUG) logDEBUG("Reregistering");
		disabledInSelector = false;
		reregistering = true;
		try {
			rsl.scheduleMaintenance(this);
		} catch (Throwable e) {
			Core.logger.log(this, "Cannot reregister "+this+", due to "+e, e, Logger.ERROR);
			terminate();
		}
		Core.logger.log(this, "Reregistered "+this+"("+accumStatus()+
						") with RSL, apparently successful", Logger.MINOR);
	}

	boolean sentHeader = false;
	
	public void registered(){
	}
	public void unregistered(){
		if(logDEBUG) logDEBUG("Unregistered", true);
	}
	public ByteBuffer getBuf() {return rawAccumulator;}
    /**
     * the accumulating method
     * ONLY called by ReadSelectorLoop
     */
    public int process(ByteBuffer b) {
		logDEBUG = Core.logger.shouldLog(Logger.DEBUG);
		synchronized(receiveClosed) {
			if(receiveClosed.state()) {
				logDEBUG("receiveClosed, not processing any more bytes");
				return -1; // close
			}
			// receiveClosed gets toggled if something drastic happens e.g.
			// corrupt trailing fields
		}
		if(b == null) b = rawAccumulator;
		if(b.limit() > 0) {
			int initialLimit = b.limit();
			if (initialLimit > DECRYPT_SIZE)
				throw new Error("you probably changed the size of the tcpConnection's buffer "+
						"and forgot to change the size of the static ciphertext buffer");
			if(b != rawAccumulator)
				throw new IllegalStateException("b NOT EQUAL TO rawAccumulator! - b = "+b+", rawAccumulator="+rawAccumulator);
			if(logDEBUG) logDEBUG("process("+bufStatus("rawAccumulator", rawAccumulator)+")");
			InputStream decrypted=null;
			
			// do not need synchronized(rawAccumulator) since we are run in the RSL thread and only process() uses rawAccumulator - FIXME if we ever change this
			//copy the information from rawAccumulator into ciphertext
			
			//assume the ciphertext is cleared
			if(logDEBUG) logDEBUG("Created ciphertext[], size "+initialLimit);
			rawAccumulator.get(ciphertext,0,initialLimit);
			
			//put rawAccumulator in ready to append state
			rawAccumulator.flip();
			if(logDEBUG) logDEBUG("Flipped rawAccumulator: "+
								  bufStatus("rawAccumulator", rawAccumulator));
			ByteArrayInputStream is = 
				new ByteArrayInputStream(ciphertext,0,initialLimit);
			
			if(is == null)
				throw new IllegalStateException("null ByteArrayInputStream!");
			
			Link link = this.link;
			if(link == null) return -1;
			
			if(logDEBUG) logDEBUG("Getting stream from "+link);
			
			decrypted = link.makeInputStream(is);
			
			if(decrypted == null)
				throw new IllegalStateException("null decrypt stream!");
			
			if((!sentHeader) && is.available() < link.headerBytes()) 
				return 1; // Need more bytes for IV
			
			// we have enough data. 
			// RSL wants it in position = 0, limit = end of bytes to read by RSL
			rawAccumulator.limit(0); // we have eaten all data

			// FIXME: optimize!
			
			// number of bytes in may != number of bytes out - IV
			if(logDEBUG) logDEBUG("Decrypting "+initialLimit+" bytes ");
			
			
			//lets see if we'll get away with this...
			decryptLen = 0;
			try {
				try {
					ByteBuffer a = accumulator;
					if(a == null) return -1; // closed already
					synchronized(a) {
						if(a.limit() + initialLimit >
						   a.capacity()) {
							if(reregistering) {
								Core.logger.log(this, "Buffer still full "+
												"while reregistering!: "+
												rawAccumulator+" in process ("+
												this, Logger.ERROR);
								terminate();
								return -1;
							}
							Core.logger.log(this, "Disabling in selector: "+
											this+" because: "+accumStatus()+
											" ("+initialLimit+
											" bytes ciphertext)", 
											Logger.MINOR);
							disabledInSelector = true;
							// See above comments - don't need to lock rawAccumulator because we are on the RSL thread
							rawAccumulator.limit(initialLimit);
							return 0;
						}
						
						decryptLen = decrypted.read(_accumulator, 
													accumulator.limit(),
													initialLimit);
						if(decryptLen > 0) {
							a.limit(a.limit() + decryptLen);
							/*	if (movingTrailingFields) System.out.println("****TRAILING FIELD****");
								else System.out.println(new String(_accumulator));*/
						}
					}
				} catch (IOException e) { decryptLen = 0; }
				if(logDEBUG) logDEBUG("Decrypted "+decryptLen+" bytes");
				/********/
				
				if(decryptLen <= 0) return 1; 
				// If we just have the IV, it will return -1
				
				if(logDEBUG) logDEBUG(accumStatus()+", reading from: "+
									  bufStatus("raw", rawAccumulator));
				
			} catch (BufferOverflowException e) {
				//TODO: log that we buffered more than 64K
				//and perhaps create another buffer?
				Core.logger.log(this, "Buffer overflowed receiving message in "+
								"ConnectionHandler! tried to cache " + 
								(BUF_SIZE - accumulator.remaining()+b.remaining())+
								" for "+this, e, Logger.ERROR);
				synchronized(receiveLock) {
					receiveClosed.change(true);
					receiveLock.notify();
				}
				return -1;
			}
		}
		if(logDEBUG) logDEBUG("outer process("+
							  bufStatus("rawAccumulator", rawAccumulator)+")");
		if(movingTrailingFields && doneMovingTrailingFields) {
			movingTrailingFields = false;
			doneMovingTrailingFields = false;
		}
		if(reregistering) {
			logDEBUG("Completing reregistration");
			try {
				rsl.register(conn.getSocket().getChannel(), 
							 this);
			} catch (Throwable t) {
				Core.logger.log(this, "Thrown "+t+
								" reregistering "+this, 
								Logger.ERROR);
				terminate();
				return -1;
			} finally {
				reregistering = false;
			}
		}
		return innerProcess();
	}
    
	boolean closeNow = false;
	
	DiscontinueInputStream currentInputStream = null;

    private int innerProcess() {
	while (true) {
	    RawMessage m=null;
	    Message msg = null;
		int processLen = -1;
	    if(!movingTrailingFields) {
			try {
				ByteBuffer x = accumulator;
				if(x == null) return -1;
				synchronized(x) {
					if(accumulator == null) return -1;
					if(logDEBUG) logDEBUG("innerProcess()", true);
					decryptLen = accumulator.limit();
					if(decryptLen > 0) {
						if(logDEBUG) logDEBUG("Trying to readMessage from "+
											  decryptLen+" bytes");
						// Read any padding
						int i=0;
						for(i=0;i<decryptLen;i++) {
							if(_accumulator[i] != 0)
								break;
							// No max padding because we wouldn't do anything different
						}
						if(logDEBUG) logDEBUG(i+" bytes of padding");
						if(i < (decryptLen - 1)) { // at least one byte...
							try {
								m = p.readMessage(_accumulator, i, decryptLen-i);
							} finally {
								processLen = p.readBytes() + i;
								// ASSUMPTION: 1 byte -> 1 byte
								if(logDEBUG) logDEBUG("Tried to readMessage from "+
													  decryptLen+" bytes, used "+processLen);
							}
						}
						//use directly the backing byte[]
						if (m!=null) {
							if(processLen == -1)
								throw new IllegalStateException("Read -1 bytes but still got a message!");
							if(logDEBUG) logDEBUG("Got Message: "+m.toString()+
												  ", clearing buffer after read");
							accumulator.position(processLen);
							accumulator.compact().flip();
							// compact() is designed for write mode
							if(logDEBUG) logDEBUG("Cleared buffer after read: "+accumStatus());
						} else {
							if(i > 0) {
								accumulator.position(i);
								accumulator.compact().flip();
								if(accumulator.limit() == 0) {
									if(logDEBUG) logDEBUG("Packet was all padding");
									// 								return 1;
								}
							}
							if(processLen == -1) processLen = 0;
							if(logDEBUG) {
								try {
									logDEBUG("Didn't get message for "+this+" from "+
										 decryptLen+" bytes: \n"+
											 new String(_accumulator, 0, decryptLen, 
														"ISO-8859-1"));
								} catch (UnsupportedEncodingException e) {
									Core.logger.log(this, "Unsupported Encoding ISO-8859-1!", 
													e, Logger.ERROR);
								}
							}
						}
					} else {
						if(logDEBUG) logDEBUG("Returning to RSL because decrypt buffer empty");
						// 					return 1; // buffer empty
					}
				}
			} catch (InvalidMessageException e){ //almost copypasted from below
				Core.logger.log(this, "Invalid message: " + e.toString()+" for "+this, e, 
								Logger.MINOR);
				invalid++;
				if (invalid >= maxInvalid) {
					Core.logger.log(this, invalid +
									" consecutive bad messages - closing "+this+
									".", Logger.MINOR);
					synchronized(receiveLock) {
						receiveClosed.change(true);
						receiveLock.notify();
					}
					return -1;
				} else {
					if(processLen > 0) {
						synchronized(accumulator) {
							accumulator.position(processLen);
							accumulator.compact().flip();
							if(logDEBUG) logDEBUG("Invalid message for "+this);
						}
					}
					continue; // Don't drop the next message
				}
			}
			//at this point we have returned succesfully from tryPrase
			//if m is null, we need more data
			if (m==null) {
				if(logDEBUG) logDEBUG("Did not get complete message");
			} else {
				
				//if not, get on with processing the message
				
				if(logDEBUG) logDEBUG("Receiving RawMessage: " + m);
				
				//received a close message
				if (m.close) {
					closeNow = true;
					if(logDEBUG) logDEBUG("Will close connection because message ("+
										  m+") said so");
					// FIXME: transfer trailing field first
				}
				
				//this is copy/pasted from below
				if (m.sustain == true && !closeNow)
					persist.change(true);
				
				//if we have trailing field, tell transport to unregister us
				//TODO:move the creation of ReadInputStream for after the ticker
				if (m.trailingFieldLength > 0) {
					Core.diagnostics.occurrenceCounting("readLockedConnections",1);
					movingTrailingFields = true;
					++receivingCount;
					if(logDEBUG) logDEBUG("Starting to transfer trailing fields");
					CHInputStream is = new CHInputStream();
					m.trailingFieldStream = currentInputStream =
						new ReceiveInputStream(is, m.trailingFieldLength, 
											   m.messageType);
				}
				
				try {
					msg = t.getMessageHandler().getMessageFor(this, m);
				} catch (InvalidMessageException e) {
					Core.logger.log(this, "Invalid message: " + e.toString(),
									Logger.MINOR);
					invalid++;
					if (invalid >= maxInvalid) {
						if(logDEBUG) 
							logDEBUG(invalid +
									 " consecutive bad messages - closing");
						synchronized(receiveLock) {
							receiveClosed.change(true);
							receiveLock.notify();
						}
						return -1;
					}
				}
				
				if(msg == null && m.trailingFieldLength > 0) {
					try {
						m.trailingFieldStream.close();
					} catch (IOException e) {
						if(logDEBUG) 
							Core.logger.log(this, "IOException closing trailing "+
											"field stream for "+this+": "+e, e, 
											Logger.DEBUG);
					}
					if(logDEBUG) logDEBUG("Invalid message but started to transfer "+
										  "trailing fields - closing");
					synchronized(receiveLock) {
						receiveClosed.change(true);
						receiveLock.notify();
					}
					return -1;
				}
				
				//on with copypasting
				if(msg != null) msg.setReceivedTime(System.currentTimeMillis());
				
				//do not copy/paste the watchme and debug stuff
				//it can be done later on
				
				messages++;
				//schedule on ticker
				if(msg != null) {
					t.add(0, msg);
					if(logDEBUG) logDEBUG("Scheduled "+msg+" on ticker");
					invalid = 0;
					Core.randSource.acceptTimerEntropy(recvTimer);
				}
			}
			if(closeNow && !movingTrailingFields) {
				if(logDEBUG) logDEBUG("Closing, finished moving trailing fields");
				synchronized(receiveLock) {
					receiveClosed.change(true);
					receiveLock.notify();
				}
				if (!sendClosed.state())
					Core.diagnostics.occurrenceCounting("peerClosed", 1);
				//simply return -1; the RSL will put us on the close queue
				ByteBuffer b = accumulator;
				if (b!=null)
					synchronized(b) {
						b.clear();
					}
				synchronized(sendLock) {
					if(sendClosed.state()) {
						if(logDEBUG) logDEBUG("Forcing close RIGHT NOW");
						if(sendingCount == 0) terminate();
						return -1;
					} else {
						if(logDEBUG) logDEBUG("Taking off RSL, waiting for writes to finish");
						return 0;
					}
				}
			}
			
			// Prevent problems with simultaneous closure
			ByteBuffer a = accumulator;
			if(a == null) return -1; // already closed
			if(a.remaining() == 0 || movingTrailingFields) {
				if(logDEBUG) logDEBUG("Returning to RSL because no remaining "+
									  "or moving trailing fields");
				if(a.capacity() - a.limit() < 
				   2048 /* FIXME: hardcoded */) {
					if(logDEBUG) 
						logDEBUG("Disabling in selector at end of message processing");
					disabledInSelector = true;
					return 0;
				} else {
					return 1;
				}
			} else {
				if(logDEBUG) logDEBUG("Trying to process next message");
				//if (explained) return 1; //perhaps this is what you meant?
				if(m == null) {
					if(logDEBUG) logDEBUG("Did not get a message, awaiting more data");
					return 1;
				} else {
					if(logDEBUG) logDEBUG("Got a message, looping");
					continue;
				}
			}
		} else {
			// Transferring trailing fields - data is on buffer
			invalid = 0;
			ByteBuffer acc = accumulator;
			if(acc == null) 
				throw new IllegalStateException("process() after closure!");
			synchronized(acc) {
				if(accumulator == null) 
					throw new IllegalStateException("process() after closure!");
				if(logDEBUG) logDEBUG("Transferring trailing fields in innerProcess()");
				/*if (!CHISwaitingForNotification){
				  Core.logger.log(this,"scheduler notified CH.innerProcess before CHIS.wait()!!!",Logger.MINOR);
				  CHISalreadyNotified=true;
				  }*/
				//notify it anyways...
				accumulator.notifyAll();
				if(accumulator.capacity() - accumulator.limit() < 
				   2048 /* FIXME: hardcoded */) {
					if(logDEBUG) logDEBUG("Disabling in selector");
					disabledInSelector = true;
					return 0;
				} else {
					return 1;
				}
			}
		}
	}
    }
    
	int CHOSsent;
	//this will be the callback from the WSL
	public void jobDone(int size, boolean status) {
		if(logDEBUG)logDEBUG("CH.jobDone("+size+","+status+") for "+this);
		MessageSend s = currentSender;
		if(s != null) {
			if(logDEBUG)logDEBUG("CH.jobDone("+size+","+status+") doing sender for "+this);
			s.jobDone(size, status);
			if(logDEBUG)logDEBUG("CH.jobDone("+size+","+status+") done sender for "+this);
		} else {
			synchronized(CHOutputStreamLock) {
				if(logDEBUG)logDEBUG("CH.jobDone("+size+","+status+
						 ") doing CHOSL for "+this);
				CHOSsent = size;
				/*if (!CHOSwaitingForNotification){
				  Core.logger.log(this,"scheduler put CH.jobDone before CHOS.wait()!!!",Logger.MINOR);
				  CHOSalreadyNotified=true;
				  }*/
				CHOutputStreamLock.notifyAll();
				if(logDEBUG)logDEBUG("CH.jobDone("+size+","+status+") done CHOSL for "+this);
			}
		}
	}
    
    /**
     * uh, what do we do here?
     */
    public void closed() {
		logDEBUG = Core.logger.shouldLog(Logger.DEBUG);
		if(logDEBUG) logDEBUG("ConnectionHandler closed() called", true);
		//this shouldn't be here but I want to test it
		synchronized(CHOutputStreamLock) {
			if (currentCHOSThread!=null)
				currentCHOSThread.interrupt();
			CHOutputStreamLock.notifyAll();
		}
		terminate();
		if(logDEBUG) logDEBUG("ConnectionHandler closed() completed");
    }
    
	public void queuedClose() {
		logDEBUG = Core.logger.shouldLog(Logger.DEBUG);
		if(logDEBUG) logDEBUG("Queued close", true);
		terminate();
		if(logDEBUG) logDEBUG("Terminated in queuedClose()");
	}
	
    private Object CHOutputStreamLock = new Object();
    
   final class CHOutputStream extends OutputStream {
    	protected boolean dead = false;
	
		private long waitTime=0;
		protected byte[] currentJob;
		protected SocketChannel chan;
		
		protected boolean finalized=false;
		
		public CHOutputStream() throws IOException {
			
			logDEBUG = Core.logger.shouldLog(Logger.DEBUG);
			
			tcpConnection c = conn;
			if(c == null) {
				throw new IOException("already terminated");
			}
			
			java.net.Socket sock = conn.getSocket();
			if(sock == null) {
				terminate();
				throw new IOException("null socket");
			}
			chan = sock.getChannel();
			if(chan == null) {
				terminate();
				throw new IOException("null channel");
			}
			
			//profiling
			//WARNING:remove before release
			synchronized(profLockCHOS){
				CHOSinstances++;
			}
		}
		
		public void write(byte[] b) throws IOException{
			//this is happening very often. TODO: find out if normal
			//if (CHOSsent!=0)
			//	Core.logger.log(this,"CHOS.write(byte[]) called while previous write in progress",Logger.ERROR);
			currentCHOSThread = Thread.currentThread();
			if(dead || alreadyClosedLink) throw new IOException("already closed");
			if(logDEBUG) logDEBUG("CHIS Trying to write(byte[]) "+b.length+
								  " bytes to "+chan);
			synchronized(CHOutputStreamLock) {
				CHOSsent = 0;
				if(!wsl.send(b,chan,ConnectionHandler.this)) {
					if(logDEBUG)
						logDEBUG("Couldn't send data: WSL.send returned false");
					throw new IOException("Couldn't send data: WSL.send returned false");
				}
				if(logDEBUG)
					logDEBUG("Sent write of "+b.length+", waiting", (b.length == 1));
				try {
				//	if (!CHOSalreadyNotified) {
						waitTime=System.currentTimeMillis();
				//		CHOSwaitingForNotification=true;
						CHOutputStreamLock.wait(5*60*1000);
						currentCHOSThread=null;
						if (dead) 
							throw new IOException("stream already closed");
				//		CHOSwaitingForNotification=false;
						if (System.currentTimeMillis() - waitTime >= 5*60*1000) {
							Core.logger.log(this,"waited more than 5 mins in CHOS.write(byte []) "+conn+":"+ConnectionHandler.this+" closing.",Logger.MINOR);
							close();
							throw new IOException("waited more than 5 mins in CHOS.write(byte [])");
						}
				//	}
				} catch (InterruptedException e) {
					if(logDEBUG) logDEBUG("Wait interrupted");
					throw new IOException("interrupted in CHOS.write(byte [])");
				} finally {
					CHOSwaitingForNotification=false;
					CHOSalreadyNotified=false;
				}
				if(logDEBUG)
					logDEBUG("CHOSsent now "+CHOSsent+", b.length="+
							 b.length);
				if (CHOSsent < b.length) throw new IOException("Short write "+
															   CHOSsent+" of "+
															   b.length);
				currentJob=null;
				CHOSsent=0;
			}
		}
		
		public void write(byte[] b, int off, int len) throws IOException{
			currentCHOSThread = Thread.currentThread();
			if(dead || alreadyClosedLink) throw new IOException("already closed");
			if(logDEBUG) logDEBUG("Trying to write(byte[],"+off+","+len+")");
			synchronized (CHOutputStreamLock) {
				CHOSsent = 0;
				if(!wsl.send(b,off,len,chan,ConnectionHandler.this)) {
					if(logDEBUG) logDEBUG("Couldn't send data: WSL.send returned false");
					throw new IOException("Couldn't send data: WSL.send returned false");
				}
				if(logDEBUG) logDEBUG("Sent "+len+" to WSL, waiting");
				try {
				//	if (!CHOSalreadyNotified) {
						waitTime=System.currentTimeMillis();
				//		CHOSwaitingForNotification=true;
						CHOutputStreamLock.wait(5*60*1000);
						currentCHOSThread=null;
						if (dead) 
							throw new IOException("stream already closed");
				//		CHOSwaitingForNotification=false;
						if (System.currentTimeMillis() - waitTime >= 5*60*1000) {
							Core.logger.log(this,"waited more than 5 mins in CHOS.write(byte [],int,int) "+conn+":"+ConnectionHandler.this+" closing.",Logger.MINOR);
							close();
							throw new IOException("waited more than 5 mins in CHOS.write(byte []),int,int");
						}
				//	}
				} catch (InterruptedException e) {
					if(logDEBUG) logDEBUG("Wait interrupted");
					throw new IOException("interrupted in CHOS.write(byte [],int,int)");
				} finally {
					CHOSwaitingForNotification=false;
					CHOSalreadyNotified=false;
				}				
				if(logDEBUG) logDEBUG("Actual write: "+CHOSsent+" of "+len);
				if (CHOSsent < len) throw new IOException("Short write: "+
														  CHOSsent+" of "+
														  len);
				currentJob=null;
				CHOSsent=0;
			}
	}
		
		byte[] oneBuf = new byte[1];
		
		public void write(int b) throws IOException{
			if(logDEBUG) logDEBUG("write(int)");
			synchronized(oneBuf) {
				currentJob = oneBuf;
				currentJob[0]=(byte)b;
				write(currentJob);
			}
		}
		
		/**
		 * this doesn't really flush.  it just throws if we didn't finish writing
		 */
		public void flush() throws IOException{
			if(logDEBUG) logDEBUG("Starting flushing", true);
			//try{
			if (currentJob!=null)
				throw new IOException("job still active");
				/*synchronized(CHOutputStreamLock) {
					CHOutputStreamLock.wait(1000); //1 second seems ok
				}*/
			//}catch(InterruptedException e){}
			synchronized(sendLock){
				synchronized(CHOutputStreamLock) {
					CHOSsent=0;
					CHOutputStreamLock.notifyAll();
				}
			if(logDEBUG) logDEBUG("flushing");
			//if(dead || alreadyClosedLink) throw new IOException("already closed");
			}
		}
		
		public void close() throws IOException{
			synchronized(CHOutputStreamLock) {
				dead = true;
				CHOutputStreamLock.notifyAll();
			}
			if(logDEBUG) logDEBUG("closing");
			//if(dead || alreadyClosedLink) throw new IOException("already closed");
			if (dead || alreadyClosedLink) return;
			wsl.queueClose(conn,ConnectionHandler.this,chan);
		}
		
	//profiling
	//WARNING:remove before release
	protected void finalize() {
		synchronized(profLockCHOS) {
			if(finalized) return;
			finalized = true;
			CHOSinstances--;
		}
	}
    }
    
    final class CHInputStream extends DiscontinueInputStream {
		// We are AKO DiscontinueInputStream so that RIS can call us
	
	
	protected boolean dead = false;
	protected boolean finalized = false;

	private long waitTime=0;

	public CHInputStream(InputStream s) {
	    super(s);
	    //profiling
	//WARNING:remove before release
	synchronized(profLockCHIS){
	    	CHISinstances++;
	    }
	}
	
	private void logDEBUG(String s, boolean trace) {
		String out = logString(s)+" ("+this+")";
		if(trace)
			Core.logger.log(this, out, new Exception("debug"), Logger.DEBUG);
		else
			Core.logger.log(this, out, Logger.DEBUG);
	}
	
	private void logDEBUG(String s) {
		logDEBUG(s, false);
	}
	
	public CHInputStream() {
	    this(new NullInputStream());
	}
	
	public int available() {
		// Might be called after closure?
		ByteBuffer a = accumulator;
		if(a == null) return -1;
	    synchronized(a) {
			if(accumulator == null) return -1;
			return accumulator.remaining();
	    }
	}
	
	public long skip(long n) throws IOException {
	    if(dead || doneMovingTrailingFields || !movingTrailingFields)
			throw new IOException("Trying to read finished CHInputStream");
		ByteBuffer a = accumulator;
		if(a == null) return -1;
	    synchronized(a) {
			if(logDEBUG) logDEBUG("Trying to skip "+n+" bytes");
		while(true) {
			if(accumulator == null) return -1;
		    if(accumulator.remaining() >= 1) {
				int got = accumulator.remaining();
				if(n < got) got = (int)n;
				accumulator.position(got);
				accumulator.compact();
				accumulator.limit(accumulator.position());
				accumulator.position(0);
				if(logDEBUG) logDEBUG("Skipped "+got+"/"+n+" bytes");
				tryReregister();
				return got;
		    } else {
				if(alreadyClosedLink) throw new IOException("Closed");
				// Uh oh...
				try {
					if(logDEBUG) logDEBUG("Waiting to skip "+n+" bytes");
					/*if (!CHISalreadyNotified) {
						CHISwaitingForNotification = true;*/
						if (dead) return -1;
						waitTime=System.currentTimeMillis();
						accumulator.wait(5*60*1000);
						if (dead) return -1;
						//CHISwaitingForNotification = false;
						if (System.currentTimeMillis() - waitTime >= 5*60*1000) {
							Core.logger.log(ConnectionHandler.this,"waited more than 5 minutes in CHIS.skip() for "+conn+":"+ConnectionHandler.this,Logger.MINOR);
							close();
							throw new IOException("waited more than 5 minutes in CHIS.skip()");
						}
					//}
					/*if (accumulator.remaining() < 1)
								Core.logger.log(this,"CHIS.skip() sync screwed up! " 
								+ ConnectionHandler.this 
								+ "alreadyNotified : " + CHISalreadyNotified 
								+ "waiting for notification " +CHISwaitingForNotification,Logger.ERROR);
					CHISalreadyNotified=false;*/
				} catch (InterruptedException e) {
					if(logDEBUG) logDEBUG("Interrupted skip wait for "+n+" bytes");
					throw new IOException("interrupted in CHIS.skip()");
				} finally {
					CHISwaitingForNotification=false;
					CHISalreadyNotified=false;
				}
				
			}
	    }
		}
	}
	
	public int read(byte[] b) throws IOException {
	    if(dead || doneMovingTrailingFields || !movingTrailingFields)
		throw new IOException("Trying to read finished CHInputStream");
		ByteBuffer a = accumulator;
		if(a == null) return -1;
//                long startedRead=0;
	    synchronized(a) {
			if(logDEBUG) logDEBUG("Trying to skip "+b.length+" bytes");
		
		while(true) {
			if (accumulator == null) return -1;
			if (alreadyClosedLink) throw new IOException("Closed");
			if (accumulator.remaining() < 1)
				// read returns what is available up to the length
				// it DOES NOT necessarily read the whole buffer!
				try {
					if(logDEBUG) logDEBUG("Waiting to skip "+b.length+" bytes");
					//if (!CHISalreadyNotified) {
					//	CHISwaitingForNotification = true;
						if (dead) throw new IOException("Trying to skip finished CHInputStream");
						waitTime=System.currentTimeMillis();
						accumulator.wait(5*60*1000);
						if (dead) throw new IOException("Trying to skip finished CHInputStream");
					//	CHISwaitingForNotification=false;
						if (System.currentTimeMillis() - waitTime >= 5*60*1000) {
							Core.logger.log(ConnectionHandler.this,"waited more than 5 minutes in CHIS.read(byte[]): "+conn+":"+ConnectionHandler.this,Logger.MINOR);
							close();
							throw new IOException("Trying to skip finished CHInputStream");
						}
					//}
					/*if (accumulator.remaining() < 1)
								Core.logger.log(this,"CHIS.read(byte []) sync screwed up! " 
								+ ConnectionHandler.this 
								+ "alreadyNotified : " + CHISalreadyNotified 
								+ "waiting for notification " +CHISwaitingForNotification,Logger.ERROR);
				CHISalreadyNotified=false;*/
				} catch (InterruptedException e) {
					if(logDEBUG) logDEBUG("Interrupted read wait for "+b.length+" bytes");
					throw new IOException("Trying to skip finished CHInputStream");
				} finally {
					CHISwaitingForNotification=false;
					CHISalreadyNotified=false;
				}
			else {
				int get = accumulator.limit();
				int got = accumulator.position();
				get -= got;
				if(b.length < get) get = b.length;
				accumulator.get(b, 0, get);
				got = accumulator.position() - got;
				accumulator.compact().flip();
				tryReregister();
				if(logDEBUG) logDEBUG("Read "+got+"/"+b.length+" bytes");
				return got;
			}
		}
		
	    }
	}
	
	public int read (byte[] b, int off, int len) throws IOException {
	    if(dead || doneMovingTrailingFields || !movingTrailingFields)
		throw new IOException("Trying to read finished CHInputStream");
		ByteBuffer a = accumulator;
		if(a == null) return -1;
	    synchronized(a) {
			if(logDEBUG) logDEBUG("Trying to CHIS.read(byte[],"+off+","+len+")");
		while(true) {
//                        long currentWait=0;
			if(accumulator == null) return -1;
			if (alreadyClosedLink) throw new IOException("Closed");
			if (accumulator.remaining() < 1)
				try {
					if(logDEBUG) logDEBUG("Waiting to read "+len+" bytes");
					//if (!CHISalreadyNotified) {
					//	CHISwaitingForNotification = true;
						waitTime=System.currentTimeMillis();
						if (dead) return -1;
						accumulator.wait(5*60*1000);
						//if (dead) return -1;
					//	CHISwaitingForNotification=false;
						if (System.currentTimeMillis() - waitTime >= 5*60*1000) {
							Core.logger.log(ConnectionHandler.this,"waited more than 5 minutes in CHIS.read(byte[],int,int): "+conn+":"+ConnectionHandler.this,Logger.MINOR);
							close();
							return -1;
						}
					//}
					/*if (accumulator.remaining() < 1)
								Core.logger.log(this,"CHIS.read(byte[],int,int) sync screwed up! " 
								+ ConnectionHandler.this 
								+ "alreadyNotified : " + CHISalreadyNotified 
								+ "waiting for notification " +CHISwaitingForNotification,Logger.ERROR);
					CHISalreadyNotified=false;*/
					//continue; //WARNING: does this belong here?
				} catch (InterruptedException e) {
					if(logDEBUG) logDEBUG("Interrupted read wait for "+len+" bytes");
					return -1;
				} finally {
					CHISwaitingForNotification=false;
					CHISalreadyNotified=false;
				}
			else {
				int get = accumulator.limit();
				int got = accumulator.position();
				get -= got;
				if(len < get) get = len;
				accumulator.get(b, off, get);
				got = accumulator.position() - got;
				accumulator.compact().flip();
				tryReregister();
				if(logDEBUG) logDEBUG("Read "+got+"/"+len+" bytes");
				return got;
			}
		}
	    }
	}
	
	public int read() throws IOException {
	    if(dead || doneMovingTrailingFields || !movingTrailingFields)
		throw new IOException("Trying to read finished CHInputStream");
		ByteBuffer a = accumulator;
		if(a == null) return -1;
		long startedRead=0;
	    synchronized(a) {
			if(logDEBUG) logDEBUG("Trying to read 1 byte");
		while(true) {
			if(accumulator == null) return -1;
		    if(accumulator.remaining() >= 1) {
				int x = (int)accumulator.get();
				accumulator.compact().flip();
				tryReregister();
				if(logDEBUG) logDEBUG("Read 1 byte");
				return (x & 0xff);
		    } else {
				// Uh oh...
				if(alreadyClosedLink) throw new IOException("Closed");
				try {
					if(logDEBUG) logDEBUG("Waiting to read() 1 byte");
		//			if (!CHISalreadyNotified) {
						//CHISwaitingForNotification=true;
						if (dead) return -1;
						startedRead=System.currentTimeMillis();
						accumulator.wait(5*60*1000);
						if (dead) return -1;
						//CHISwaitingForNotification=false;
						if (System.currentTimeMillis() - startedRead >= 5*60*1000) {
							Core.logger.log(this,"waited more than 5 minutes on CHIS.read() "+conn+":"+ConnectionHandler.this,Logger.MINOR);
							close();
							return -1;
						}
		//			}
				/*if (accumulator.remaining() < 1)
						Core.logger.log(this,"CHIS.read() sync screwed up! " 
								+ ConnectionHandler.this 
								+ "alreadyNotified : " + CHISalreadyNotified 
								+ "waiting for notification " +CHISwaitingForNotification,Logger.ERROR);
				CHISalreadyNotified=false;*/
				} catch (InterruptedException e) {
					if(logDEBUG) logDEBUG("Interrupted wait: "+e);
					return -1;
				} finally {
					CHISwaitingForNotification=false;
					CHISalreadyNotified=false;
				}
		    }
			if(logDEBUG) logDEBUG("Waited to read() 1 byte", true);
		}
	    }
	}
	
	public void discontinue() {
		if(logDEBUG) logDEBUG("Discontinuing read stream");
	    innerClose();
	    //we should not close this stream when called with discontinue, because it will in turn
	    //close the entire connection; discontinue() is called only when we finish reading a trailing
	    //field, but we should leave the connection open.
	}
	
	/**
	 * sets up outside visible flags and closes the connection
	 * as it should.  In the old code this stream was the same stream
	 * comming out of the connection's socket, so calling close() was expected
	 * to close the socket itself.
	 */
	public void close() throws IOException{
		if (dead) return;
		innerClose();
		rsl.queueClose(((tcpConnection)conn).getSocket().getChannel(),ConnectionHandler.this);
	}
	
	/**
	 * only sets up outside visible flags
	 * for some reason this still ends up closing the connection
	 * TODO: find out why
	 */
	private void innerClose() {
		logDEBUG = Core.logger.shouldLog(Logger.DEBUG);
		if(logDEBUG) logDEBUG("Closing CHIS", true);
	    	dead = true;
	    	doneMovingTrailingFields = true;
		// Why was this commented out?
		ByteBuffer a = accumulator;
		if(a != null) {
			synchronized(a) {
				//if(accumulator != null)
				a.notifyAll();
			}
		}
		tryReregister();
	    if(rsl != null && a != null && 
		   (closeNow || (a.remaining() > 0))) {
			logDEBUG("Scheduling maintenance");
			rsl.scheduleMaintenance(ConnectionHandler.this);
			logDEBUG("Scheduled maintenance");
		}
		if(rsl == null) {
			throw new IllegalStateException("Do not know my ReadSelectorLoop ! ("+
											this+","+ConnectionHandler.this+")");
	    }
		logDEBUG("Closed CHIS");
	}
	
	//profiling
	//WARNING:remove before release
	protected void finalize() {
		synchronized(profLockCHIS) {
			if(finalized) return;
			finalized = true;
			CHISinstances--;
		}
	}
    }
    
    /**  Description of the Method */
    public void start() {
		throw new UnsupportedOperationException();
//         (new Thread(this, "" + this)).start();
    }
	

    // Public Methods
    /**  Main processing method for the ConnectionHandler object */
	public void run() {

//         receiveThread = Thread.currentThread();

//         try {
//             started.change();
//         }
//         catch (IrreversibleException e) {
//             Core.logger.log(this, "Can not restart connectionhandlers",
//                             Logger.ERROR);
//             throw new RuntimeException("C.H. already run once");
//             // ??? -tavin <- what's weird, this implies something is badly fucked
//         }

//         Core.logger.log(this, "Started listening for messages",
//                         Core.logger.DEBUG);

//         lastActiveTime = System.currentTimeMillis();

//         // Get the inputstream for messages (decrypted);
//         PushbackInputStream  in        =
//             new PushbackInputStream(link.getInputStream());

//         boolean              wait;
//         int                  padCount;
//         //int                  invalid   = 0;

//         //boolean close = false;

//         // loop over each message
//         message : do {

//             padCount = 0;
//             lastActiveTime = System.currentTimeMillis();
//             try {
//                 link.setTimeout(3000);
//             }
//             catch (IOException e) {
//                 Core.logger.log(this, "Error resetting socket timeout",
//                                 e, Logger.MINOR);
//                 break message;
//             }
            
//             waiting : while (!receiveClosed.state()) {
//                 int  i;
//                 try {
//                     i = in.read();
//                     if (i == -1) {
//                         Core.diagnostics.occurrenceCounting("peerClosed", 1);
//                         break message;
//                     } else if (i == 0 && ++padCount >= maxPadding) {
//                         Core.logger.log(this, "Too much padding.", Logger.DEBUG);
//                         break message;
//                     }
//                         else {
//                             in.unread(i);
//                             break waiting;
//                         }
//                 } catch (InterruptedIOException e) {
//                     //if (sending.count() == 0 && !persist.state() &&
//                     if (sendingCount == 0 && !persist.state() &&
//                         System.currentTimeMillis() >=
//                         (lastActiveTime + Core.connectionTimeout)) {
                        
//                         if (sendClosed.state()) {
//                             // no reply within one timeout of sending
//                             // the closing messages...
//                             //break waiting;  -- we are certain to break message;
//                             //                   a few lines down anyway
//                             break message;
//                         } else {
//                             // timeout
//                             Core.diagnostics.occurrenceBinomial("connectionTimedout",
//                                                                 1, 1);
//                             close();
//                             continue message;
//                         }
//                     }
//                 }
//                 catch (IOException e) {
//                     Core.diagnostics.occurrenceCounting("peerClosed", 1);
//                     Core.logger.log(this, "I/O error on socket", e, Logger.MINOR);
//                     break message;
//                 }
//             }
            
//             if (receiveClosed.state())
//                 break message;
            
//             boolean  needToRemoveFromOCM  = false;
//             try {
//                 synchronized (receiveLock) {
//                     // now that we are synced
//                     if (receiveClosed.state())
//                         break message;
                    
//                     try {
//                         // note the deliberate placement of these variables.
//                         // so that the Message variables are inside, which keeps
//                         // it possible for the finalizer on the ReceiveInputStream
//                         // to be called while we are waiting on it (preventing
//                         // deadlock if the stream is just ignored).
//                         RawMessage  m    = null;
//                         Message     msg  = null;
                        
//                         //receiving.incCount();
//                         ++receivingCount;
//                         link.setTimeout(600000);
//                         /*
//                          *  10 minutes
//                          */
//                         // read message from protocol
//                         m = p.readMessage(in);
//                         long receivedTime = System.currentTimeMillis();
//                         if (m.close) {
//                             needToRemoveFromOCM = true;
//                             receiveClosed.change(true);
//                             if (!sendClosed.state())
//                                 Core.diagnostics.occurrenceCounting("peerClosed", 1);
//                         }
//                         // avoid actually setting the receiveClosed flag
//                         // until we've read the trailing field
//                         //close = m.close;
//                         //if (close) removeFromOCM();
//                         if (m.trailingFieldLength > 0)
//                             m.trailingFieldStream =
//                                 new ReceiveInputStream(in, 
//                                                        m.trailingFieldLength, 
//                                                        m.messageType);
                        
//                         if (m.sustain == true)
//                             persist.change(true);
                        
//                         Core.randSource.acceptTimerEntropy(recvTimer);
                        
//                         if (Core.logger.shouldLog(Logger.DEBUG))
//                             Core.logger.log(this, "Receiving RawMessage: " + m,
//                                             Logger.DEBUG);
                        
//                         msg = t.getMessageHandler().getMessageFor(this, m);
//                         invalid = 0;
                        
//                         // Some temporary debug stuff
//                         if (m.trailingFieldStream != null) {
//                             ((ReceiveInputStream) m.trailingFieldStream).messageId = msg.id();
//                         }

//                         // succeeded this time.
//                         msg.setReceivedTime(receivedTime);
//                         if (Main.watchme != null) {
//                             if (!((msg instanceof QueryRejected) &&
//                                   (((QueryRejected) msg).
//                                    reason.indexOf("protocol") != -1))) {
//                                 int htl;
//                                 if (msg instanceof HTLMessage)
//                                     htl = ((HTLMessage) msg).hopsToLive();
//                                 else
//                                     htl = -1;
                                
//                                 Main.watchme.logReceiveMessage(m.messageType,
//                                                                Long.toHexString(msg.id()),
//                                                                link.getPeerAddress().toString(),
//                                                                htl);
//                             }
//                         }
                        
//                         // i think users like this one too..
//                         if (Core.logger.shouldLog(Logger.MINOR))
//                             Core.logger.log(this,
//                                             m.messageType + " @ "
//                                             + Long.toHexString(msg.id())
//                                             + " <- " + link.getPeerAddress(),
//                                             Logger.MINOR);

//                         // Schedule
                        
//                         t.add(0, msg);
//                         messages++;
//                         if (m.trailingFieldLength > 0)
//                                 // &&
//                                 //!((ReceiveInputStream) m.trailingFieldStream).done)
//                                 // ^^^ there is a risk of decrementing receivingCount twice
//                             wait = true;
//                         else {
//                                 //receiving.decCount();
//                             --receivingCount;
//                             wait = false;
//                         }
//                     }
//                     catch (InvalidMessageException e) {
//                         Core.logger.log(this, "Invalid message: " + e.toString(),
//                                         Logger.MINOR);
//                         invalid++;
//                         if (invalid >= maxInvalid) {
//                             Core.logger.log(this, invalid +
//                                             " consecutive bad messages - closing.",
//                                             Logger.DEBUGGING);
//                             break message;
//                         }
//                         else
//                             continue message;
//                     }
//                     catch (EOFException e) {
//                         Core.diagnostics.occurrenceCounting("peerClosed", 1);
//                         break message;
//                         // this stream is over
//                     }
//                     catch (IOException e) {
//                         Core.diagnostics.occurrenceCounting("peerClosed", 1);
//                         break message;
//                     }
                    
//                     if (wait) {
//                         Core.diagnostics.occurrenceCounting("readLockedConnections",
//                                                             1);
//                         try {
//                             receiveLock.wait();
//                         }
//                         catch (InterruptedException e) {
//                             Core.logger.log(this, "I got interrupted!",
//                                             Logger.DEBUGGING);
//                         }
//                         Core.diagnostics.occurrenceCounting("readLockedConnections",
//                                                             -1);
//                         //Core.logger.log(this,"I am awake!",Logger.DEBUGGING);
//                     }
//                     //Core.logger.log(this,"Finished with message",
//                     //                Logger.DEBUGGING);
//                         lastActiveTime = System.currentTimeMillis();
//                 }
//             }
//             finally {
//                 if (needToRemoveFromOCM)
//                     removeFromOCM();
                
//             }
//         } while (!receiveClosed.state());

//         // Unlocked.  Groovy.
//         receivingCount = 0; // definitely not reading now.
//         removeFromOCM();
//         receiveClosed.change(true);
//         receiveThread = null;
//         // in case we break;'d out

//         // should this be in terminate() instead?
//         Core.diagnostics.occurrenceContinuous("connectionLifeTime",
//                                               System.currentTimeMillis()
//                                               - startTime);
//         Core.diagnostics.occurrenceContinuous("connectionMessages",
//                                               messages);

//         Core.logger.log(this, "Receiving terminated: "+this+".", Logger.DEBUG);

//         if (sendClosed.state()) {
//             // everybody done.
//             // The race condition we need to worry about here is that
//             // neither would call terminate(), but that can't happen
//             // because both reading and writing sets themselves closed
//             // before checking that the other is. So if the sending
//             // thread got that we were still open, it must have set
//             // itself close before we readed sendClosed.state()
// 			Core.logger.log(this, "SendClosed "+this, Logger.DEBUG);
//             if (sendingCount == 0) {
// 				Core.logger.log(this, "Terminating "+this, Logger.DEBUG);
//                 terminate();
// 				Core.logger.log(this, "Terminated "+this, Logger.DEBUG);
// 			}
//         }
//         else {
//             // note that close does nothing if p.getMessage is null
// 			Core.logger.log(this, "Closing "+this, Logger.DEBUG);
//             close();
// 			Core.logger.log(this, "Closed "+this, Logger.DEBUG);
// 		}
	}


    /**
     * Flushes the sending stream (if no message is being sent).
     */
    public void flushOut() {
    }
	
	public final class MessageSend {
		Message m;
		RawMessage raw;
		SendFailedException sfe;
		boolean hasTrailing = false;
		byte[] toSend;
		boolean done = false;
		boolean success = false;
		
		// Call synchronized(sendLock)
		MessageSend(Message m, RawMessage raw, byte[] b) 
			throws SendFailedException {
			this.m = m;
			this.raw = raw;
			sfe = null;
			toSend = b;
			hasTrailing = (raw.trailingFieldLength > 0);
		}
		
		void start() throws SendFailedException {
			try {
				if(conn == null) 
					throw new IllegalStateException("conn null! for "+
													ConnectionHandler.this);
				java.net.Socket sock = conn.getSocket();
				if(sock == null)
					throw new IOException("Can't get socket");
				SelectableChannel chan = sock.getChannel();
				if(chan == null)
					throw new IOException("Can't get channel");
				if(wsl == null) 
					throw new IllegalStateException("wsl null in "+
													ConnectionHandler.this);
				if(!wsl.send(toSend, chan, ConnectionHandler.this)) {
					throw new IOException("Can't write");
				}
			} catch (IOException e) {
				Core.logger.log(this, "Got IOException trying to sendMessage: "+this+" ("+ConnectionHandler.this+")", e, Logger.DEBUG);
				needToRemoveFromOCM = true;
				sendClosed.change(true);
				// terminal failure
				String s = e.getMessage();
				if(s != null) s = s.getClass().getName()+": "+s;
				if(s != null)
					throw new SendFailedException(peer.getAddress(), null,
												  s, true);
				else
					throw new SendFailedException(peer.getAddress(), true);
			}
		}
		
		void closed() {
			logDEBUG = Core.logger.shouldLog(Logger.DEBUG);
			if(logDEBUG) logDEBUG("MessageSend closed(): "+this);
			if(!done) jobDone(0, false);
		}
		
		// Override this to get called when done
		public void notifyDone() {
			synchronized(this) {
				this.notifyAll();
			}
			//System.out.println("notified MessageSend");
		}

		void jobDone(int size, boolean status) {
			logDEBUG = Core.logger.shouldLog(Logger.DEBUG);
			try {
				if(size == toSend.length) {
					currentSender = null;
					success = true;
					if(logDEBUG) logDEBUG("Successfully sent message "+this);
					messages++;
				} else {
					sfe = new SendFailedException(peer.getAddress(), true);
					Core.logger.log(this, "Send failed for "+this+" ("+
									ConnectionHandler.this+") in jobDone("+
									size+","+status+") for "+toSend.length, sfe,
									Logger.MINOR);
				}
				if(logDEBUG) logDEBUG("MessageSend finished sending message "+this);
				done = true;
				currentSender = null;
				synchronized(sendLock) {
					sendQueueSize = Math.max(sendQueueSize - meanMessageSize, 0);
				}
			} finally {
				// only if there is a trailing field to write do we not
				// decrement sendingCount and notify() immediately
				if(logDEBUG) logDEBUG("in jobDone finally{} "+this);
				try {
					if (!hasTrailing) {
						//sending.decCount(); // done
						synchronized(sendLock) {
							--sendingCount;
							if(logDEBUG) logDEBUG("send == null");
							if(logDEBUG) logDEBUG("Receiving "+receivingCount+
									 " messages, sending "+sendingCount+
									 " messages, receiveClosed="+
									 receiveClosed.state()+", sendClosed="+
									 sendClosed.state()+" ("+this+")");
							if (receiveClosed.state() && receivingCount == 0
								&& sendClosed.state() && sendingCount == 0) {
								if(logDEBUG) logDEBUG("terminating");
								terminate();
							}
							sendLock.notifyAll();
							if(logDEBUG) logDEBUG("notified");
						}
					}
					if (needToRemoveFromOCM) {
						if(logDEBUG) logDEBUG("Need to remove from OCM");
						// Not synced, this is called directly on the WSL thread
						needToRemoveFromOCM = false;
						if(logDEBUG) logDEBUG("Removed from OCM");
					}
				} catch (Throwable t) {
					Core.logger.log(this, "Caught throwable "+t+" in "+
									ConnectionHandler.this+"."+this+"."+
									"jobDone",t, Logger.ERROR);
				}
				// notify MUST get called no matter what
				if(logDEBUG) logDEBUG("Notifying "+this);
				notifyDone();
				if(logDEBUG) logDEBUG("Notified "+this);
			}
		}
	}
	
	MessageSend currentSender = null;
	
	// Call with sendLock held
	private void waitForOneSender() throws SendFailedException {
		if (sendingCount > 1) {
			long  time  = System.currentTimeMillis();
			try {
				sendLock.wait(60000);
			}
			catch (InterruptedException e) {
			}
			if(conn == null || conn.isClosed())
				throw new SendFailedException(peer.getAddress(),
											  "Already closed!");
			if (sendingCount > 1 &&
				// this can give false pos.
				System.currentTimeMillis() - time >= 60000) {
				Core.logger.log(this, this + 
								" waited one minute on " +
								"connection to "
								+ peer + " (" + hashCode() +
								") free up, but it didn't:"+
								this+", sendingCount now "+
								sendingCount, Core.logger.NORMAL);
				throw new SendFailedException(peer.getAddress(),
											  "Gave up wait");
			}
			else if (sendClosed.state())
				// check again
				throw new SendFailedException(peer.getAddress(),
											  false);
		}
	}
	
	boolean needToRemoveFromOCM = false;
	
	void logWatchme(Message m, RawMessage raw) {
		int  htl;
		if (m instanceof HTLMessage)
			htl = ((HTLMessage) m).hopsToLive();
		else
			htl = -1;
		
		Main.watchme.logSendMessage(raw.messageType,
									Long.toHexString(m.id()),
									peer.getAddress().toString(),
									htl);
	}
	
	public MessageSend sendMessageAsync(Message m, RawMessage raw) 
		throws SendFailedException {
		logDEBUG = Core.logger.shouldLog(Logger.DEBUG);
		try {
			if (sendClosed.state())
				throw new SendFailedException(peer.getAddress(), false);
			needToRemoveFromOCM = false;
			synchronized(sendLock) {
				// Do not remove sendLock without seeing comments below
				if (sendClosed.state())
				// check again when synced
					throw new SendFailedException(peer.getAddress(), false);
				MessageSend ms = null;
				try {
					sendingCount++;
					waitForOneSender();
					// if we were just waiting for the connection to become free..
					if(m == null) return null;
					if (raw.close) {
						needToRemoveFromOCM = true;
						sendClosed.change(true);
						//} else if (raw.sustain) {
						//    persist.change(true);
						// hm, we only want to persist if we *receive*
						// a sustain message, right?
					}
					
					sendQueueSize += meanMessageSize + raw.trailingFieldLength;
					if(logDEBUG) logDEBUG(sendingCount + " messages in sendqueue");
					
					if (Core.logger.shouldLog(Logger.DEBUG))
						if(logDEBUG) logDEBUG("Sending RawMessage: " + raw, true);
					
					// WATCHME
					if (Main.watchme != null) {
						logWatchme(m, raw);
					}
					
					// users are fond of this log entry
					if (Core.logger.shouldLog(Logger.MINOR))
						Core.logger.log(this,
										raw.messageType + " @ " +
										Long.toHexString(m.id())
										+ " -> " + peer.getAddress(),
										Logger.MINOR);
					
					// Now send the message
					ByteArrayOutputStream bais = new ByteArrayOutputStream();
					Link l = link;
					if(l == null) return null;
					OutputStream os = l.makeOutputStream(bais);
					// We are synchronized on sendLock, so we do not need
					// to lock os
					try {
						raw.writeMessage(os);
						os.flush();
					} catch (IOException e) {
						e.printStackTrace();
						Core.logger.log(this, "Impossible exception for "+
										ConnectionHandler.this+": "+e, e,
										Logger.ERROR);
						throw new IllegalStateException("Impossible exception!: "+
														e);
					}
					byte[] b = bais.toByteArray();
					ms = new MessageSend(m, raw, b);
					if(logDEBUG) logDEBUG("Got MessageSend "+ms+": "+b.length+" bytes");
					currentSender = ms;
					try {
						ms.start();
					} catch (SendFailedException e) {
						Core.logger.log(this, "Got IOException trying to "+
										"sendMessage: "+this, e, Logger.DEBUG);
						ms = null;
						throw e;
					}
					if(logDEBUG) logDEBUG("Started MessageSend: "+ms);
					return ms;
				} finally {
					// only if there is a trailing field to write do we not
					// decrement sendingCount and notify() immediately
					// Can't possibly have started moving trailing if we have an error here
					if(logDEBUG) logDEBUG("in sendMessageAsync finally{}");
					//sending.decCount(); // done
					if(ms == null) {
						--sendingCount;
						if(logDEBUG) logDEBUG("send == null");
						if(logDEBUG) logDEBUG("Receiving "+receivingCount+
								 " messages, sending "+sendingCount+
								 " messages, receiveClosed="+
								 receiveClosed.state()+", sendClosed="+
								 sendClosed.state()+" ("+this+")");
						if (receiveClosed.state() && receivingCount == 0
							&& sendClosed.state() && sendingCount == 0) {
							if(logDEBUG) logDEBUG("terminating");
							terminate();
						}
						sendLock.notifyAll();
						if(logDEBUG) logDEBUG("notified");
					}
				}
			}
		} finally {
			tcpConnection c = conn;
			if(c != null) {
				if(c.isClosed()) {
					terminate();
				} else if (needToRemoveFromOCM) {
					if(logDEBUG) logDEBUG("Need to remove from OCM");
					removeFromOCM();
					needToRemoveFromOCM = false;
					if(logDEBUG) logDEBUG("Removed from OCM");
				}
			}
		}
	}
	
	public OutputStream sendMessage(Message m) 
		throws SendFailedException {
		if (conn == null)
			throw new SendFailedException(peer.getAddress(), false);
		if(logDEBUG) logDEBUG("SendMessage("+m+")", true);
		RawMessage    raw   = m.toRawMessage(p);
		if(logDEBUG) logDEBUG("raw is "+raw);
		MessageSend ms = sendMessageAsync(m, raw);
		if(logDEBUG) logDEBUG("sendMessageAsync called");
		long startedsend=0;
		synchronized(ms) {
			while(!ms.done) {
				try {
					if(logDEBUG) logDEBUG("Waiting for MessageSend "+ms);
					startedsend=System.currentTimeMillis();
					ms.wait(5*60*1000); //FIXME:hardcoded 5 minutes
				} catch (InterruptedException e) {
					throw new SendFailedException(peer.getAddress(), true);
				};
				if (System.currentTimeMillis() - startedsend >= 5*60*1000)
					throw new SendFailedException(peer.getAddress(), true);
			}
			if(logDEBUG) logDEBUG("Waited for MessageSend");
		}
		OutputStream send = null;
		if(ms.sfe != null) {
			Core.logger.log(this, "MessageSend produced SendFailedException "+
							" for "+this+": "+ms.sfe, ms.sfe, Logger.MINOR);
			throw ms.sfe;
		}
		boolean knowAboutTrailing = false;
		if(ms.success) {
			if(logDEBUG) logDEBUG("Success!");
			try {
				if (ms.hasTrailing) {
					// has trailing
					knowAboutTrailing = true;
					if(logDEBUG) logDEBUG("Has trailing");
					OutputStream os = new CHOutputStream();
					Link l = link;
					if(l == null) throw new IOException("Closed in middle of send");
					os = l.makeOutputStream(os);
					send = new SendOutputStream(os, raw.trailingFieldLength);
					if(logDEBUG) logDEBUG("Started SendOutputStream");
				} else {
					if(logDEBUG) logDEBUG("Does not have trailing");
					// does not
					//Core.logger.log(this, "Message sent.", Logger.DEBUG);
					Core.randSource.acceptTimerEntropy(sendTimer);
					lastActiveTime = System.currentTimeMillis();
				} 
			} catch (IOException e) {
				Core.logger.log(this, "Got IOException trying to sendMessage: "+this, e, Logger.DEBUG);
				tcpConnection c = conn;
				if(c != null) {
					if(c.isClosed()) {
						terminate();
					} else {
						removeFromOCM();
					}
				} // else already closed
				synchronized(sendClosed) {
					sendClosed.change(true);
				}
				// terminal failure
				throw new SendFailedException(peer.getAddress(), true);
			}
		}
		if(logDEBUG) logDEBUG("Returning "+send);
		return send;
	}
	
    /**
     *  Sends a message using this connection. This will return after the message
     *  headers have been sent.
     *
     *@param  m                        the message to send. If m is null, the
     *      method will lock until m _could_ have been sent, and then return.
     *@return                          the OutputStream to write the trailing field
     *      to if there is one, otherwise null.
     *@exception  SendFailedException  Description of the Exception
     */
//     public OutputStream sendMessage(Message m)
//         throws SendFailedException {

//         // non terminal SFEs if we just stopped sending on this pipe

//         if (sendClosed.state())
//             throw new SendFailedException(link.getPeerAddress(), false);

//         boolean  needToRemoveFromOCM  = false;
//         try {
//             synchronized (sendLock) {
//                 if (sendClosed.state())
//                     // check again when synced
//                     throw new SendFailedException(link.getPeerAddress(), false);

// 				//int n = sending.incCount();
//                 OutputStream  send  = null;
//                 try {
//                     ++sendingCount;
//                     //if (sending.count() > 1) {
					
// 					waitForOneSender();
					
//                     // if we were just waiting for the connection to become free..
//                     if (m == null)
//                         return null;

//                     RawMessage    raw   = m.toRawMessage(p);
//                     //if (raw == null) {
//                     //    return null;
//                     //}
//                     if (raw.close) {
//                         needToRemoveFromOCM = true;
//                         sendClosed.change(true);
//                         //} else if (raw.sustain) {
//                         //    persist.change(true);
//                         // hm, we only want to persist if we *receive*
//                         // a sustain message, right?
//                     }

//                     sendQueueSize += meanMessageSize + raw.trailingFieldLength;
// 					logDEBUG(sendingCount + " messages in sendqueue");

//                     // dont need now
//                     //raw.close = (sendClosed.state() && sendingCount <= 1);
//                     // defensive coding .. b/c i'm tired
//                     // of puzzling over this

//                     if (Core.logger.shouldLog(Logger.DEBUG))
// 						logDEBUG("Sending RawMessage: " + raw, true);
					
//                     // WATCHME
//                     if (Main.watchme != null) {
// 						logWatchme(m);
//                     }

//                     // users are fond of this log entry
//                     if (Core.logger.shouldLog(Logger.MINOR))
//                         Core.logger.log(this,
//                                         raw.messageType + " @ " +
//                                         Long.toHexString(m.id())
//                                         + " -> " + link.getPeerAddress(),
//                                         Logger.MINOR);

//                     OutputStream  mout;
//                     try {
// 						logDEBUG("Still sending message");
// 						mout = link.makeOutputStream(new CHOutputStream());
// 						logDEBUG("Got CHOutputStream: "+mout);
//                         raw.writeMessage(mout);
// 						logDEBUG("Written rawMessage");
//                         mout.flush();
// 						logDEBUG("Flushed message");
//                         messages++;
//                     }
//                     catch (IOException e) {
// 						Core.logger.log(this, "Got IOException trying to sendMessage: "+this, e, Logger.DEBUG);
//                         needToRemoveFromOCM = true;
//                         sendClosed.change(true);
//                         // terminal failure
//                         throw new SendFailedException(link.getPeerAddress(), true);
//                     }
// 					logDEBUG("Got stream, still here");
//                     //sendQueueSize -= Math.min(meanMessageSize, sendQueueSize);
//                     // i am ever so slightly worried this syntax is bad with volatiles
//                     sendQueueSize = Math.max(sendQueueSize - meanMessageSize, 0);

//                     if (raw.trailingFieldLength > 0) {
//                         // has trailing
// 						logDEBUG("Has trailing");
//                         send = new SendOutputStream(mout, 
//                                                     raw.trailingFieldLength);
// 						logDEBUG("Started SendOutputStream");
//                     } else {
// 						logDEBUG("Does not have trailing");
//                         // does not
//                         //Core.logger.log(this, "Message sent.", Logger.DEBUG);
//                         Core.randSource.acceptTimerEntropy(sendTimer);
//                         lastActiveTime = System.currentTimeMillis();
//                     }
// 					logDEBUG("About to return send: "+send);
//                     return send;
//                 }
//                 finally {
//                     // only if there is a trailing field to write do we not
//                     // decrement sendingCount and notify() immediately
// 					logDEBUG("in finally{}");
//                     if (send == null) {
//                         //sending.decCount(); // done
//                         --sendingCount;
// 						logDEBUG("send == null");
// 						logDEBUG("Receiving "+receivingCount+
// 								 " messages, sending "+sendingCount+
// 								 " messages, receiveClosed="+
// 								 receiveClosed.state()+", sendClosed="+
// 								 sendClosed.state()+" ("+this+")");
//                         if (receiveClosed.state() && receivingCount == 0
//                             && sendClosed.state() && sendingCount == 0) {
// 							logDEBUG("terminating");
//                             terminate();
//                         }
//                         sendLock.notify();
// 						logDEBUG("notified");
//                     }
//                 }
//             }
//         }
//         finally {
//             if (needToRemoveFromOCM) {
// 				logDEBUG("Need to remove from OCM");
//                 removeFromOCM();
// 				logDEBUG("Removed from OCM");
// 			}
//         }
//     }


    // IMPORTANT: Don't call this while holding sendLock
    //            or receiveLock or you will cause a
    //            deadlock.
    //
    /**  Removes this CH from the OCM. */
    private final void removeFromOCM() {
        if (peerIdentity() != null)
            ocm.remove(this);
    }


    /**
     *  Initiates a close dialog on this connection by sending a closeMessage() as
     *  specified by the Presentation object if one can be created.
     */
    public void close() {
	Message  m  = p.getCloseMessage();
	if (m != null)
	    try {
		OutputStream  out  = sendMessage(m);
		if (out != null)
		    // wtf...
		    out.close();
	    }
	catch (IOException e) {
		Core.logger.log(this, "Impossible exception in "+this+".close(): "+e,
						e, Logger.NORMAL);
	    // wtf wtf...
	}
	catch (SendFailedException e) {
	    Core.logger.log(this,
			    "Failed to send connection close message for "
			    + this + ": " + m, e, Core.logger.MINOR);
	   	terminate(); 
	}
	/*finally{
		terminate();
	}*/
    }


    /**  Closes the connection utterly and finally. */
    public void terminate() {
		logDEBUG = Core.logger.shouldLog(Logger.DEBUG);
		if(logDEBUG) logDEBUG("Terminating", true);
//        synchronized(temp) {
//               temp.println();
//               temp.println("--------------------------------------------------");
//              (new Exception("Entered ConnectionHandler.teminate()")).printStackTrace(temp);
        //        }
		boolean wasClosed = alreadyClosedLink;
		ByteBuffer a = accumulator;
		if(a != null)
			synchronized(a) {
				alreadyClosedLink = true;
				a.notifyAll();
				accumulator = null;
				_accumulator = null;
			}
        try {
            finalized.change();
        }
        catch (IrreversibleException e) {
            return;
            // terminate called twice
        }
		// Log stats
		if(initRegisteredInOCM) {
			// If it's actually started
			Core.diagnostics.occurrenceContinuous("connectionLifeTime",
												  System.currentTimeMillis() -
												  startTime);
			Core.diagnostics.occurrenceContinuous("connectionMessages",
												  messages);
		}
		if(logDEBUG) logDEBUG("flipped finalized in terminate()");
		// Need to release sendLock first...
		synchronized(CHOutputStreamLock) {
			CHOutputStreamLock.notifyAll();
		}
		if(logDEBUG) logDEBUG("notified CHOS in terminate()");
		if(currentSender != null)
			currentSender.jobDone(0, false);
		if(logDEBUG) logDEBUG("notified current sender in terminate()");
		//terminatedInstances++;
        if (!sendClosed.state())
            // ???  Is there a codepath that leaves sendClose.state()
            // ???  true without removing the connection?
            //removeFromOCM();
            synchronized (sendLock) {
                sendClosed.change(true);
                sendLock.notifyAll();
            }
		if(logDEBUG) logDEBUG("changed and notified sendClosed/sendLock in terminate()");
        if (!receiveClosed.state())
            synchronized (receiveLock) {
                receiveClosed.change(true);
                receiveLock.notify();
            }
		if(logDEBUG) logDEBUG("changed and notified receiveClosed/receiveLock in terminate()");
		if(!wasClosed) {
			try {
				Link l = link;
				if(l != null) l.close();
			}
			catch (IOException e) {
				Core.logger.log(this, "Exception closing link for "+this+": "+
								e, e, Logger.NORMAL);
			}
		}
		if(logDEBUG) logDEBUG("ConnectionHandler closed link", true);
		
		if(currentInputStream != null) {
			try {
				currentInputStream.close();
			} catch (IOException e) {
				if(logDEBUG) Core.logger.log(this, "IOException caught closing stream: "+e,
								e, Logger.DEBUG);
			}
		}
		if(logDEBUG) logDEBUG("Closed CHIS in terminate()");
		// Unconditionally remove the connection handler
		// from the OCM
		removeFromOCM();
		if(logDEBUG) logDEBUG("Removed from OCM in terminate()");
		link = null;
		conn = null;
		if(logDEBUG) logDEBUG("terminate()d");
    }
	
	
    /**
     * Returns the thread that is being used in the receive loop, if it is 
     * running.
     */
    public Thread getReceiveThread() {
        return receiveThread;
    }

    /**
     *  Checks whether the connection is alive and can send messages.
     *
     *@return    The open value
     */
    public final boolean isOpen() {
        return !sendClosed.state();
    }


    /**
     *  The number of messages in sendqueue.
     */
    int sendingCount() {
        return sendingCount;
    }


    /**
     *  Returns the number milliseconds since this connection was active.
     *
     *@return    Description of the Return Value
     */
    public final long idleTime() {
        //return (sending.count() > 0 || receiving.count() > 0 ?
        return (sendingCount > 0 || receivingCount > 0 ?
                0 : System.currentTimeMillis() - lastActiveTime);
    }


    /**
     *@return    the number of bytes waiting to be sent on this connection
     */
    public final long sendQueueSize() {
        //return sendQueueSize(); -- any chance this could have led to an infinite
        //                           loop in OCM that made it never release its
        //                           monitor?
        return sendQueueSize;
    }


    /**
     *@return    whether the connection is currently sending something
     */
    public final boolean sending() {
        //return sending.count() > 0;
        return sendingCount > 0;
    }


    /**
     *@return    whether the connection is current receiving something
     */
    public final boolean receiving() {
        return receivingCount > 0;
    }


    /**
     *@return    identity of the Peer on the other end of the connection
     */
    public final Identity peerIdentity() {
        return peer.getIdentity();
    }


    /**
     *@return    the Transport used for this connection
     */
    public final Transport transport() {
        return peer.getAddress().transport();
    }


    public final LinkManager sessionType() {
        return peer.getLinkManager();
    }


    public final Address peerAddress() {
        return peer.getAddress();
    }

    public final Presentation presentationType() {
        return p;
    }


    /**
     *  The time since this was started.
     */
    public final long runTime() {
        return System.currentTimeMillis() - startTime;
    }


    /**
     *  The number of messages sent and received over this connection.
     */
    public final long messages() {
        return messages;
    }

    protected void finalize()
        throws Throwable {
	    
        if (!finalized.state()) {
            Core.logger.log(this, "I ("+this+
							") wasn't terminated properly! Doing it now..",
                            Logger.ERROR);
            terminate();
		}
	//profiling
		//WARNING:remove before release
	synchronized(profLock) {
		instances--;
	}
    }



    //=========================================================================
    // the rest is inner classes for streams that handle the trailing
    // fields when sending/receiving messages
    //=========================================================================

    /**
     *  An InputStream that is allows reading of a limited number of bytes, and
     *  unlocks the receiving when that many bytes are read or it is closed.
     *
     */
    private final class ReceiveInputStream extends DiscontinueInputStream {

        private final  long     toRead;
        private        long     read    = 0;
        private        boolean  done    = false;
        private        String  messageType;

        private long messageId = -1;
		
        /**
         *  Constructor for the ReceiveInputStream object
         *
         *@param  in      Description of the Parameter
         *@param  toRead  Description of the Parameter
         */
        public ReceiveInputStream(InputStream in, long toRead,
                                  String messageType) {
            super(in);
            this.toRead = toRead;
            this.messageType = messageType;
	    //profiling
	//WARNING:remove before release
		synchronized(profLockRIS) {
			RISinstances++;
		}
        }


        public int read()
            throws IOException {
			if(logDEBUG) logDEBUG("trying RIS.read()");
            try {
                if (read >= toRead)
                    return -1;
                int  i  = in.read();
				if(logDEBUG) logDEBUG("read from RIS.read(): "+i);
                if (i > -1)
                    read++;
                if (read == toRead)
                    done();
                return i;
            }
            catch (IOException e) {
                close();
                throw (IOException) e.fillInStackTrace();
            }
        }


        public int read(byte[] b, int off, int length)
            throws IOException {
			if(logDEBUG) logDEBUG("trying RIS.read(byte[],"+off+","+length+")", true);
            try {
                if (read >= toRead) {
					Core.logger.log(this,"returning -1 from RIS.read(byte[],int,int) because read >=toRead (for "+ConnectionHandler.this+" read is "+read+" toRead is "+toRead+")" ,Logger.NORMAL);
                    return -1;
				}
                else if (read + length > toRead)
                    length = (int) (toRead - read);
				
				// System.err.println("CONN STREAM READING SUPER IN: " + in.toString());
                int  i  = in.read(b, off, length);
				if(logDEBUG) logDEBUG("read in RIS.read(byte[],int,int) "+i+" from "+in, true);
                if (i > -1)
                    read += i;
                if (read == toRead)
                    done();
				if(logDEBUG) logDEBUG("read(byte[],"+off+","+length+") - i = "+i+", read="+read+
						 ",toRead="+toRead, true);
                return i;
            } catch (IOException e) {
				Core.logger.log(this, "Got IOException: "+e+" in RIS.read(,"+off+
								","+length+") for "+this+" ("+
								ConnectionHandler.this+")", e, Logger.NORMAL);
                close();
				throw new IOException("Got IOException: "+e);
            }
        }
		
        public void close()
            throws IOException {
            //if (read < toRead) {
            //    // not good, bad data present on Connection
            //    Core.logger.log(ConnectionHandler.this, "Close after " + read
            //                    + " of " + toRead + " bytes", Logger.MINOR);
            //}
			if(logDEBUG) logDEBUG("Closing", true);
            done();
            //discontinue();
            //super.close();
	    
        }


        public final void discontinue()
            throws IOException {
			if(logDEBUG) logDEBUG("Discontinuing");
	//Core.logger.log(this,"discontinuing RIS",Logger.ERROR);
            read = toRead;
            done();
        }

        private void done() throws IOException {
			if(logDEBUG) logDEBUG("in RIS.done()", true);
			currentInputStream = null;
            if (!done) {
                synchronized (receiveLock) {
                    done = true;
                    lastActiveTime = System.currentTimeMillis();
                    //receiving.decCount();
                    if (read < toRead) {
						Core.logger.log(this, "Closing because of premature "+
										"done(): "+this, Logger.MINOR);
                        // can't read any more
                        receiveClosed.change(true);
						closeNow = true;
                    }
                    // messages on this conn.
                    --receivingCount;
                    receiveLock.notify();
                    // wake read thread
                }
                if (read < toRead)
                    Core.logger.log(ConnectionHandler.this, "Close after " +
									read+ " of " + toRead + " bytes: "+this+
									" for "+ConnectionHandler.this,
									Logger.MINOR);
				else {
					if(logDEBUG) logDEBUG("Trailing field fully received: "+read+" of "+toRead);
					//note to toad: this is assuming we only use it on CHIS/NIOIS
					//where close == discontinue
					//in.close();
					if(in instanceof DiscontinueInputStream)
						((DiscontinueInputStream)in).discontinue();
				}
            }
        }
		
        public String toString() {
            return "ConnectionHandler$ReceiveInputStream allocated for " 
                + messageType + ", " + read + " of " + toRead + " bytes done ("+
				ConnectionHandler.this+")";
        }
		
        protected void finalize()
            throws Throwable {
			if(logDEBUG) logDEBUG("finalizing");
            if (!done) {
                Core.logger.log(ConnectionHandler.this,
                                "I was finalized without being properly "+
								"deallocated: " + this + " ("+
								ConnectionHandler.this+")", Logger.ERROR);
				
                if (messageId != -1) { // yes I know it can be -1, but this is debug
                    System.err.println("This query failed to deallocate " + 
									   this);
                    t.getMessageHandler().printChainInfo(messageId,
                                                         System.err);
                }
                
                done();
            }
	    //profiling
	//WARNING:remove before release
	synchronized(profLockRIS){
		RISinstances--;
	}
            super.finalize();
        }
    }
	
	
    /**
     *  An OutputStream that only allows writing of so many bytes, and that
     *  releases the sendLock that many bytes are written or it closed.
     */
    private final class SendOutputStream extends FilterOutputStream {

        private        long     written  = 0;
        private final  long     toWrite;
        private        boolean  done     = false;
	


        public SendOutputStream(OutputStream out, long toWrite) {
            super(out);
            this.toWrite = toWrite;
	    //profiling
	//WARNING:remove before release
	synchronized(profLockSOS) {
		SOSinstances++;
	}
        }


        public void write(int i)
            throws IOException {
            try {
                if (written >= toWrite)
                    throw new IOException("Overwriting: " + written +
                                          " + 1 > " + toWrite);
                out.write(i);
                written++;
                sendQueueSize = Math.max(sendQueueSize - 1, 0);
                if (toWrite == written)
                    done();
            }
            catch (IOException e) {
                close();
                throw (IOException) e.fillInStackTrace();
            }
        }


        public void write(byte[] b, int off, int length)
            throws IOException {
            try {
                if (written + length > toWrite)
                    throw new IOException("Overwriting: " + written + " + " + length
                                          + " > " + toWrite);
                out.write(b, off, length);
                written += length;
				//sendQueueSize -= Math.min(length,sendQueueSize);
				// i am ever so slightly worried this syntax is bad with volatiles
                sendQueueSize = Math.max(sendQueueSize - length, 0);
                if (written == toWrite)
                    done();
            }
            catch (IOException e) {
                close();
                throw (IOException) e.fillInStackTrace();
            }
        }


        public void close()
            throws IOException {
            if (!done && written < toWrite)
                Core.logger.log(ConnectionHandler.this,
                                "Close after " + written + " of " + toWrite + " bytes on: " + this,
                                Logger.MINOR);

            //sendQueueSize -= Math.min((toWrite - written), sendQueueSize);
            // i am ever so slightly worried this syntax is bad with volatiles
            sendQueueSize = Math.max(sendQueueSize - (toWrite - written), 0);
            written = toWrite;
            done();
        }


        public void done() {
            if (!done) {
                done = true;
                try {
                    out.flush();
                }
                catch (IOException e) {
                    Core.logger.log(ConnectionHandler.this, "I/O error "+
									"when flushing SendOutputStream: "+e,
                                    e, Logger.MINOR);
                }
                synchronized (sendLock) {
                    //sending.decCount(); // done!
                    --sendingCount;
                    //if (close) {
                    //    removeFromOCM();
                    //    sendClosed.change(true);
                    //}
                    // ^^^ this has already happened
                    if (receiveClosed.state() && receivingCount == 0
                        && sendClosed.state() && sendingCount == 0) {
                        terminate();
                    }

                    Core.logger.log(ConnectionHandler.this,
                                    "Message and trailing field sent.",
                                    Logger.DEBUGGING);

                    Core.randSource.acceptTimerEntropy(sendTimer);

                    lastActiveTime = System.currentTimeMillis();
                    sendLock.notify();
                    // wake a possible waiting thread.
                }
            }
        }

        /*
         *  Removed because it prevents early GCing of these objects. - someone
         *  Restored because it prevents an eternal lock on sending, and I'm
         *  pretty sure we were seeing that. - oskar
         */
        protected void finalize() throws Throwable {
            if (!done) {
                Core.logger.log( ConnectionHandler.this,
                                 "I was finalized without being " + 
                                 "properly deallocated: "+this,
                                 Logger.ERROR);
                done();
            }
	    synchronized(profLockSOS) {
	    	SOSinstances--;
	    }
            super.finalize();
        }
    }
	
	public String toString() {
		return super.toString()+" for "+conn+","+link;
	}
	
	
}




