-- Copyright (C) 2009 Papavasileiou Dimitris                             
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

require "nature"
require "moremath"
require "aviation"
require "shading"
require "toon"
require "transforms"
require "imaging"

local modulated = resources.pipeline {resources.modulate(0.4, 0.4, 1),
				      textures.clamped}

local elevation = resources.dofile ("aviation/" .. aviation.dataset ..
				    "/" .. aviation.dataset .. ".lua")

aviation.nodes.land = (options.toon and nature.earth or nature.land) {
   albedo = 0.45,
   
   elevation = nature.elevation (elevation) {
      isground = true,
      target = options.groundlod or 10000,
      
      link = function(self)
		local t_0 = os.clock()
		local m, n, l = unpack(elevation.size)
		local s, t = unpack(elevation.resolution)
		local i_0, i_1 = aviation.latitudes[1], aviation.latitudes[2]
		local j_0, j_1 = aviation.longitudes[1], aviation.longitudes[2]

		print ("\nLoading imagery and elevation data.  " ..
		       "This may take a while.\n")

		for i = 0, n - 1 do
		   for j = 0, m - 1 do
		      local suffix, samples, bounds, imagery

		      suffix = string.format("%d.%d.lc", i, j)
		      samples, bounds =
			 resources.dofile ("aviation/" .. aviation.dataset ..
					   "/elevation_" .. suffix)

		      if i >= i_0 and i <= i_1 and
			 j >= j_0 and j <= j_1 and
			 not options.noimagery then
			 imagery = 
			    resources.dofile ("aviation/" .. aviation.dataset ..
					      "/imagery_" .. suffix)
		      end
		      
		      self[j * n + i] = {samples, bounds, imagery}
		   end
		end

		print (string.format("\nImagery and elevation data loaded " ..
				     "in %.1f seconds.\n", os.clock() - t_0))

		-- Center ourselves over (0, 0)

		self.position = units.meters {
		   -0.5 * m * math.pow(2, l) * s,
		   -0.5 * n * math.pow(2, l) * t,
		   0}

		-- No need to load more than once.

		self.link = nil
	     end
   }
}

aviation.nodes.sky = nature.atmosphere {
   size = {1024, 512},

   turbidity = aviation.turbidity,

   rayleigh = math.scale({6.95e-06, 1.18e-05, 2.44e-05}, 1),
   mie = 7e-5,

--       frames.timer {
--          period = 0.2,

--          tick = function (self, tick, delta, elapsed)
-- 		local n = 2 + tick * 0.1

--     	        print (n)
-- 		self.parent.rayleigh = math.scale({6.95e-06, 1.18e-05, 2.44e-05}, n)
-- 		self.parent.mie = math.scale({4e-7, 6e-7, 2.4e-6}, n)
--    		self.parent.sun = {math.pi, math.pi / n}
--  		self.parent.turbidity = n
--  		self.parent.eccentricity = self.parent.eccentricity + 0.05
-- 		print (self.parent.eccentricity)
--    	     end
--       },

   clock = frames.timer {
      period = 1 / 0,

      link = function (self)
		self.tick(self, 0, 0, 0)
	     end,

      tick = function (self, tick, delta, elapsed)
	 local phi, delta, a, A, H, f, dst
		
	 -- Advance the time of day.

	 aviation.time = {aviation.time[1], aviation.time[2] + 0.01}	

	 -- Calculate the position of the sun.

	 dst = aviation.time[1] > 90 and aviation.time[1] < 300
	 phi = math.rad(aviation.location[2])
	 _, f = math.modf (aviation.location[1] / 15)
	 H = (12 - aviation.time[2] - f + (dst and 1 or 0)) * math.rad(15)
	 delta = -math.rad(23.45) * math.cos(math.rad (360 / 365) *
					     (aviation.time[1] + 10))

         a = math.asin (math.sin (phi) * math.sin(delta) + 
		        math.cos(phi) * math.cos(delta) * math.cos(H))

	 cosA = math.cos(phi) * math.sin(delta) -
                math.sin(phi) * math.cos(delta) * math.cos(H)
	 sinA = math.cos(delta) * math.sin(H) / math.cos(a)

	 if sinA > 0 then
	    A = math.acos(cosA)
	 else
	    A = 2 * math.pi - math.acos(cosA)
	 end

	 -- print (math.deg(A), math.deg(a))

	 self.parent.sun = {A, a}
	 self.parent.sunlight.position =
	    math.transform (transforms.euler (0, math.deg(a), math.deg(A)),
			    {1e5, 0, 0})

	 if self.parent.sunlight.islight then
	    self.parent.sunlight.intensity = self.parent.intensity
	 else
	    local intensity = self.parent.intensity

	    self.parent.sunlight.intensity = {
	       [0] = math.scale (intensity, 0.4),
	       [0.04] = math.scale (intensity, 0.6),
	       [0.15] = math.scale (intensity, 0.8),
	       [0.45] = math.scale (intensity, 1)
	    }
	 end
	
	 self.parent.haze.color = self.parent.intensity
	 self.parent.haze.linear = 5 * (self.parent.turbidity - 1) * self.parent.mie
      end
   },

   sunlight = options.toon and toon.lamp {
      intensity = {[0] = 0.4, [0.04] = 0.6, [0.15] = 0.8, [0.45] = 1},
      orientation = transforms.euler (180, 0, 0),
      ambience = 0.4,

      islamp = true
   } or shading.light {
      offset = {1, 2},
      volume = {-1000, 1000, -1000, 1000, 0.1, 1000},
      intensity = math.scale({1, 1, 0.8}, 1),
      size = {0, 0},

      islight = true
   },

   skylight = not options.toon and shading.ambient {
      orientation = transforms.euler (0, 180, 0),
      intensity = modulated "aviation/imagery/horizon.lc",
   },

   haze = options.toon and toon.haze{} or shading.fog {},
   
   system = bodies.system {
   }
}

aviation.nodes.sky.parent = graph
aviation.nodes.land.parent = aviation.nodes.sky.system